#include <klib.h>
#include <asm/kl_arch.h>
#include <asm/kl_mem.h>
#include <linux/dump.h>
#include <asm/dump.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <cksum.h>
#include <crashtool.h>

#define MAX_LONG_LONG   0xffffffffffffffff
#define MAX_STR 128
static char *Version = "1";

/*
 * check_type_dev() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a block device type.
 */
int
check_type_dev(char *path)
{
        struct stat sbuf;
	int ret=0;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Checking dump device",lentitle);
		fflush(stdout);
	}
        if (stat(path, &sbuf) < 0) {
                return (1);
        }
	if ( S_ISBLK(sbuf.st_mode) ) {
		if ( verbose_set == 1 ) {
			printf("OK\n");
		}
		ret=0;
	} else {
		printf("ERROR: can't open dump device.\n");
		ret=1;
	}

        return (ret);
} 

/*
 * Name: kl_dump_retrieve()
 * Func: Gather a dump from a given device and save it into the dump
 *       directory.  The dump directory points to the possibility of
 *       a "bounds" file, which would be read, and be used to save the
 *       index number of the dump.  Returns 0 on failure, 1 if the
 *       core dump is short or if there was a problem finishing the
 *       read/write of the dump, or 2 on success.
 */

int kl_dump_retrieve(char *dumpdev, char *dumpdir, int local_debug)
{
        dump_page_t dp;
        dump_header_t dh;
        dump_header_asm_t dha;
	char buffer[DUMP_PAGE_SZ];
	char	*toffset;
        int bounds, devf, outf, indf, addr_set=0, chunk_nb=0, chunk_new=0;
        char *tfile, *dfile, *tmpstr, *ptrstr, *ofile, *uncompr_page, *ptr_page;
	time_t curtime;
	uint64_t start_addr,chunk_size_max,chunk_size, page_size;
	unsigned long sum_dump=0,nb_physpages=0;
	int	i,ret=0;


        /* check the types */
        if (check_type_dev(dumpdev) != 0) {
                fprintf(KL_ERRORFP,
                        "Error: kl_check_type() on dumpdev failed!\n");
                return (0);
        }

        /* check the types */
        if (check_type_dir(dumpdir) != 0) {
                fprintf(KL_ERRORFP,
                        "Error: kl_check_type() on dumpdir failed!\n");
                return (0);
        }

        /* allocate the buffer for file data */
        tfile = (char *)malloc(KL_PAGE_SIZE);
        ofile = (char *)malloc(KL_PAGE_SIZE);
        dfile = (char *)malloc(KL_PAGE_SIZE);
	uncompr_page = (char *)malloc(KL_PAGE_SIZE);

        /* try to read the bounds file */
        bounds = kl_bounds_retrieve(dumpdir);

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Open dump device",lentitle);
		fflush(stdout);
	}

        /* try to open the output file */
        if ((devf = open(dumpdev, O_RDWR, 0)) < 0) {
                fprintf(KL_ERRORFP, "Error: open() on dumpdev failed!\n");
                return (0);
        }

        /* move beyond the swap header */
        if (lseek(devf, KL_PAGE_SIZE, SEEK_SET) < 0) {
                fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
                close(devf);
                return (0);
        }
	if ( verbose_set == 1 ) {
		printf("Ok\n");
		savecrash_gentitle("Read dump header",lentitle);
		fflush(stdout);
		fflush(stdout);
	}

        /* try to read the dump header */
	/* first check the size of both header base and arch specifique */
        if (read(devf, (char *)&dh, sizeof(dump_header_t)) < sizeof(dump_header_t)) {
                        fprintf(KL_ERRORFP, "Error: read() of dump header failed!\n");
                        close(devf);
                        return (0);
        }
	
        /* try to read the dump header */
        if (read(devf, (char *)&dha, sizeof(dump_header_asm_t)) < sizeof(dump_header_asm_t)) {
                        fprintf(KL_ERRORFP, "Error: read() of dump header failed!\n");
                        close(devf);
                        return (0);
        }

        /* validate the dump header */
        if (dh.dh_magic_number != DUMP_MAGIC_NUMBER) {
                        fprintf(KL_ERRORFP,
                                "Error: DUMP_MAGIC_NUMBER in "
                                "dump header invalid!\n");
                close(devf);
                return (0);
        }

	/* check if dump already saved */
	if (dh.dh_dump_flags & DUMP_FLAGS_SAVED) {
		if ( resave_set == 0 ) {
			fprintf(KL_ERRORFP,
				"Error: DUMP_FLAGS_SAVED set!\n");
			close(devf);
			return(0);
		}
	} 
	
        /* validate the architecture-specific dump header */
        if (dha.dha_magic_number != DUMP_ASM_MAGIC_NUMBER) {
                        fprintf(KL_ERRORFP,
                                "Error: DUMP_ASM_MAGIC_NUMBER in "
                                "dump arch header invalid: 0x%llx!\n",dha.dha_magic_number);
                close(devf);
                return (0);
        }

	if ( verbose_set == 1 ) {
		printf("Ok\n");
		savecrash_gentitle("Start retrieving modules",lentitle);
		fflush(stdout);
	}

	/* create the directory */
	sprintf(tfile, "%s/crash.%d", dumpdir, bounds);
	if ((outf = mkdir(tfile,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) < 0 ) {
		fprintf(KL_ERRORFP,
				"Error: mkdir() of dump directory  \"%s\" failed!\n",
				tfile);
		close(devf);
		return (0);
	}

	sprintf(tfile, "%s/crash.%d/INDEX", dumpdir, bounds);
	if ((indf = open(tfile, O_CREAT|O_RDWR|O_TRUNC,(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
		fprintf(KL_ERRORFP, "Error: open() of INDEX file \"%s\" failed!\n",tfile);
		close(devf);
		return (0);
	}

	/* Now create the INDEX */
	sprintf(tfile,"%-16s%s\n","comment","savecrash crash dump INDEX file");
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s\n","dump version",Version);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s\n","OS",dh.dh_utsname_sysname);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s\n","hostname",dh.dh_utsname_nodename);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s\n","modelname",dh.dh_utsname_machine);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s\n","panic",dh.dh_panic_string);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s","dumptime",ctime(&(dh.dh_time.tv_sec)));
	write(indf, tfile, strlen (tfile));
	curtime = time((time_t *)0);
	sprintf(tfile,"%-16s%s","savetime",ctime(&curtime));
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%s%s\n","release",dh.dh_utsname_release,dh.dh_utsname_version);
	write(indf, tfile, strlen (tfile));
	sprintf(tfile,"%-16s%lld\n\n","memsize",(long long)dh.dh_memory_size*dh.dh_page_size);
	write(indf, tfile, strlen (tfile));

        /* now make sure we have more than just a header -- if not, leave */
        
	if (dh.dh_dump_level == DUMP_LEVEL_HEADER) {
                close(devf);
                kl_bounds_update(dumpdir, bounds + 1);
                return (1);
        }

	/* now retrieve module list */
	/* first module is the kernel */
	sprintf(tfile,"%s",dh.dh_boot_kernel);
	sum_dump=cksum(tfile);
	if ( sum_dump != 0 ) {
		tmpstr=strrchr(dh.dh_boot_kernel,'/');
		tmpstr++;
		sprintf(ofile,"%s/crash.%d/%s",dumpdir, bounds,tmpstr);
		if (kl_cp_file(tfile,ofile)) {
			sprintf(tfile,"%-16s%s %s %lu\n","module",dh.dh_boot_kernel,tmpstr,sum_dump);
			write(indf, tfile, strlen (tfile));
		} else {
			sprintf(tfile,"%-16s%s\n","module","failed to copy kernel file");
			write(indf, tfile, strlen (tfile));
		}
	} else {
		sprintf(tfile,"%-16s%s\n","module","failed to open kernel file !");
		write(indf, tfile, strlen (tfile));
	}
	strcpy(uncompr_page,dh.dh_dump_modules);
	tmpstr=uncompr_page;
	for(i=1;i<dh.dh_dump_nbr_modules;i++) {
		ptrstr=strchr(tmpstr,';');
		if ( ptrstr != NULL ) {
			ptrstr[0]='\0';
			sprintf(tfile,"%s",tmpstr);
		}
		if ( kl_find_module(tfile) != 0 ) {
			sprintf(dfile,"%-16s%s%s\n","module","can't find module ",tmpstr);
		} else {
			sum_dump=cksum(tfile);
			if ( sum_dump != 0 ) {
				tmpstr=strrchr(tfile,'/');
				tmpstr++;
				sprintf(ofile,"%s/crash.%d/%s",dumpdir, bounds,tmpstr);
				if (kl_cp_file(tfile,ofile)) {
					sprintf(dfile,"%-16s%s %s %lu\n","module",tfile,tmpstr, sum_dump);
				} else {
					sprintf(dfile,"%-16s%s %s\n","module","failed to copy",tfile);
				}
			} else {
				sprintf(dfile,"%-16s%s %s %s\n","module",tfile,tmpstr,"cksum failed");
			}
		}
		write(indf, dfile, strlen (dfile));
		tmpstr=ptrstr+1;
	}

	if ( verbose_set == 1 ) {
		printf("Ok\n");
		savecrash_gentitle("Start retrieving dump",lentitle);
		fflush(stdout);
	}

        /* move to the right offset in the device file */
        (void)lseek(devf, DUMP_PAGE_SZ + KL_PAGE_SIZE, SEEK_SET);

	/* now calculate how many chunk to do */
	chunk_nb=0;
	/* so we need to calculate the size of a chunk */
	chunk_size_max=(dh.dh_memory_size*dh.dh_page_size)/chunk_max;
	chunk_size=0;
	chunk_new=0;	
        
	/* now start rolling through the blocks */
        while (1) {
		/* check if we are not above chunk_size_max */
		if ( chunk_size >= chunk_size_max ) {
			chunk_new=0;
			chunk_size=0;
			sum_dump=cksum(dfile);
			sprintf(tfile,"chunk\t\t0x%.16llx 0x%.16llx core.%d.%d %lu\n"
				,start_addr,(long long)nb_physpages,chunk_nb,bounds,sum_dump);
			write(indf, tfile, strlen (tfile));
			if ( compression_flag == 1 ) {
				ret=kl_compress_module(dfile);
				if ( ret != 0 ) {
					return(1);
				}
			}
			chunk_nb=chunk_nb+1;
			nb_physpages=0;
			addr_set=0;
			close(outf);
		}
			
                /* try to read in the block */
		if (read(devf, (char *)&buffer, DUMP_PAGE_SZ) < DUMP_PAGE_SZ ) {
			/* if this fails, we may have run out of space */
                        fprintf(KL_ERRORFP, "Warning: short dump in dumpdev!\n");
                        close(devf);
                        close(outf);
                        close(indf);
                        kl_bounds_update(dumpdir, bounds + 1);
                        return (1);
		}
	 	/* set toffset to the start of the buffer */
		toffset=&buffer;	
		/* now we have our buffer loop to retrieve all pages */
		while ( bool == 0 ) {
			/* first retrieve the dump_page header */
			memcpy((char *)&dp,toffset, sizeof(dp));
			
                	if (local_debug) {
                       		fprintf(KL_ERRORFP,
                                	"dp.dp_address = 0x%"FMT64"x, "
                                	"dp.dp_flags = 0x%x, dp.dp_size = %d\n",
                                	dp.dp_address, dp.dp_flags, dp.dp_size);
                	}

                	/* sanity check */
                	if ((!dp.dp_flags) || ((!dp.dp_size) && (!(dp.dp_flags & DUMP_DH_END))) || (dp.dp_size > dh.dh_page_size)) {
                                	fprintf(KL_ERRORFP,
                                        	"Error: dump flags in dumpdev "
                                        	"page index invalid!\n");
                                	close(devf);
                                	close(outf);
                        		close(indf);
                                	kl_bounds_update(dumpdir, bounds + 1);
                                	return (1);
                	}

                	/* make sure we aren't at the end of the dump */
                	if (dp.dp_flags & DUMP_DH_END) {
				/* so now I can mark the dump as saved */
				/* so go back to the start of the sump */
				if (lseek(devf, KL_PAGE_SIZE, SEEK_SET) < 0) {
                			fprintf(KL_ERRORFP, "Error: lseek() on dumpdev failed!\n");
                			close(devf);
					close(outf);
					close(indf);
					kl_bounds_update(dumpdir, bounds + 1);
                		return (1);
        		}
			/* then change the flag and write on the header */
			dh.dh_dump_flags=dh.dh_dump_flags+DUMP_FLAGS_SAVED;
			if ((write(devf, (char *)&dh, sizeof(dump_header_t))) != sizeof(dump_header_t)) {
				fprintf(KL_ERRORFP, "Error: write() on dumpdev failed!\n");
				close(devf);
				close(outf);
                                close(indf);
                                kl_bounds_update(dumpdir, bounds + 1);
				return (1);
			}
                        close(devf);
                        close(outf);
			/* if a new chunk has just been open, nothing exist in it */
			/* so I don't have to write in INDEX file                 */
			if ( chunk_new != 0 ) {
				sum_dump=cksum(dfile);
				sprintf(tfile,"chunk\t\t0x%.16llx 0x%.16llx core.%d.%d %lu\n"
					,start_addr,(long long)nb_physpages,chunk_nb,bounds,sum_dump);
				write(indf, tfile, strlen (tfile));			
				if ( compression_flag == 1 ) {
					ret=kl_compress_module(dfile);
					if ( ret != 0 ) {
						return(1);
					}
				}
			}
                        kl_bounds_update(dumpdir, bounds + 1);
			if ( verbose_set == 1 ) {
				printf("Ok\n");
			}
                        close(indf);
                        return(0);
                }

                /* try to read in the dump page itself */
                if (read(devf, tfile, dp.dp_size) < dp.dp_size) {
                        /* if this fails, we may have run out of space */
                        fprintf(KL_ERRORFP, "Warning: short dump in dumpdev!\n");
                        close(devf);
                        close(outf);
                        close(indf);
                        kl_bounds_update(dumpdir, bounds + 1);
                        return (1);
                }

		/* need to setup start_addr if not set */
		if ( addr_set == 0 ) {
			start_addr=dp.dp_address;
			addr_set=1;
		}

		/* if it is a compressed page, need to uncompress */
		/* actually don't support page_size != KL_PAGE_SIZE */
                if ( dp.dp_flags == DUMP_DH_COMPRESSED ) {
			page_size=kl_uncompress_page((unsigned char*)tfile, (unsigned char*)uncompr_page, dp.dp_size);
                        if ( page_size != KL_PAGE_SIZE) {
				fprintf(KL_ERRORFP, "Error: kl_uncompress_page() failed!\n");
                        	close(devf);
                        	close(outf);
                        	close(indf);
                        	kl_bounds_update(dumpdir, bounds + 1);
                        	return (1);
			}
			ptr_page=uncompr_page;
		} else {
			ptr_page=tfile;
			page_size=dp.dp_size;
                } 
	
		/* calculate now the chunk size */
		chunk_size=chunk_size+page_size;
		
		/* need to count the number of pages */
		/* so -> page_size/dh_page_size      */
		 nb_physpages=nb_physpages+(page_size/dh.dh_page_size);

                /* try to write in the dump page itself */
		/* check if it is a new chunk, if yes need create the file */
		/* make the new filename */
		if ( chunk_new == 0 ) {
			sprintf(dfile, "%s/crash.%d/core.%d.%d", dumpdir, bounds, chunk_nb, bounds);
			if ((outf = open(dfile, O_CREAT|O_RDWR|O_TRUNC, (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
				fprintf(KL_ERRORFP, "Error: open() of dump file \"%s\" failed!\n", tfile);
				close(devf);
                		return (0);
        		}
			chunk_new = 1;
		}
                
		if (write(outf, (char *)ptr_page, page_size) != page_size) {
                        /* if this fails, we may have run out of space */
                        fprintf(KL_ERRORFP, "Warning: write() to dump file failed!\n");
                        close(devf);
                        close(outf);
                        close(indf);
                        kl_bounds_update(dumpdir, bounds + 1);
                        return (1);
                }
        }
	}
}

/*
 * check_type_dir() -- run the stat() operation without having all the
 *                     inline definitions conflicted by the kernel.  Make
 *                     sure <path> is a directory.
 */
int
check_type_dir(char *path)
{
        struct stat sbuf;
	int ret=0;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Checking directory",lentitle);
		fflush(stdout);
	}
        if (stat(path, &sbuf) < 0) {
                return (1);
        }
	if ( S_ISDIR(sbuf.st_mode) ) {
                if ( verbose_set == 1 ) {
			printf("OK\n");
		}
                ret=0;
        } else {
                printf("ERROR: can't open dump directory.\n");
                ret=1;
        }     
        return (ret);
}

/*
 * Name: kl_get_bounds()
 * Func: Get the bounds number from the bounds file located in <dumpdir>
 *       Returns the bounds number, or 0 if no bounds file exists.
 */
int
kl_bounds_retrieve(char *dumpdir)
{
        int dirf, bounds;
        char *tfile;

	if ( verbose_set == 1 ) {
		savecrash_gentitle("Read bound",lentitle);
		fflush(stdout);
	}
        tfile = (char *)malloc(strlen(dumpdir) + 10);
        if (!tfile) {
                return (0);
        }

        sprintf(tfile, "%s/bounds", dumpdir);
        if ((dirf = open(tfile, O_RDONLY, 0)) < 0) {
		if ( verbose_set == 1 ) {
			printf("No bound: 0\n");
		}
                return (0);
        }

        (void)read(dirf, tfile, strlen(tfile));
        bounds = atoi(tfile);
        close(dirf);
	if ( verbose_set == 1 ) {
		printf("%d\n",bounds);
	}
	return (bounds);
}

/*
 * Name: kl_bounds_update()
 * Func: Update the bounds number with a new value.  The function takes
 *       the <dumpdir> along with the new bounds number.  Returns 0 for
 *       failure, 1 for success.
 */
void
kl_bounds_update(char *dumpdir, int bounds)
{
        int dirf;
        char *tfile;

        tfile = (char *)malloc(strlen(dumpdir) + 10);
        if (!tfile) {
                return;
        }

        sprintf(tfile, "%s/bounds", dumpdir);
        if ((dirf = open(tfile, O_RDWR|O_CREAT,
                (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
                        fprintf(KL_ERRORFP, "Warning: could not "
                                "create new bounds file in %s!\n", dumpdir);
                        return;
        }
	sprintf(tfile, "%d\n", bounds);
        if (write(dirf, tfile, strlen(tfile)) != strlen(tfile)) {
                fprintf(KL_ERRORFP, "Warning: could not "
                        "write to bounds file in %s!\n", dumpdir);
                close(dirf);
                return;
        }
        close(dirf);
        return;
}

int
kl_cp_file(char *src,char *dest)
{
	int infile, outfile, loop;
	char buf[8192];
	size_t count;

	/* open src file */
	if (( infile = open(src, O_RDONLY, 0)) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on %s failed!\n",src);
		return (0);
	}

	/* open dest file */
	if (( outfile = open(dest, O_CREAT|O_RDWR|O_TRUNC,(S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH))) < 0) {
		fprintf(KL_ERRORFP, "Error: open() on %s failed!\n",src);
		return (0);
	}
	loop=0;
	while (loop == 0) {
		count=read(infile , buf, sizeof(buf));
		if ( count == 0 ) {
			loop=1;
		}
		if ( write(outfile, buf, count) != count) {
			loop=1;
			fprintf(KL_ERRORFP, "Error: write() on %s failed!\n",dest);
			return (0);
		}
	}

	if ( compression_flag == 1 ) {
		loop=kl_compress_module(dest);
		if ( loop != 0 ) {
			return(0);
		}
	}
	return(1);
}

int
kl_find_module(char *mod)
{
  char buf[KL_PAGE_SIZE];
  char *ptrav;
  FILE *fic;

	sprintf(buf,"insmod -n -q %s 2>/dev/null",mod);
	if ( (fic=popen(buf, "r")) != NULL ) {
		fgets(buf, sizeof(buf), fic);
      		pclose(fic);
      		if ((ptrav=strchr(buf, ' ')) != NULL) {
        		ptrav++;
			ptrav[strlen(ptrav)-1]='\0';
			strncpy(mod,ptrav,KL_PAGE_SIZE);
      		} else {
			buf[0]='\0';
			fprintf(KL_ERRORFP, "Error: insmod -n -q %s failed!\n",mod);
			return(1);
		}
	} else {
		fprintf(KL_ERRORFP, "Error: insmod -n -q %s failed!\n",mod);
		return(1);
	}
	return(0);
}

int
kl_compress_module(char *mod)
{
  char buf[KL_PAGE_SIZE];
  int ret;

	sprintf(buf,"gzip %s",mod);
	if ( (ret=system(buf)) != 0) {
		fprintf(KL_ERRORFP, "Error: gzip() on %s failed!\n",mod);
	}
	return(ret);
}

/*
 * kl_uncompress_page() -- Uncompress a buffer of data
 *
 * Return the size, otherwise 0.
 */

int 
kl_uncompress_page(unsigned char *cbuf, unsigned char *ucbuf, uint32_t blk_size)
{
	int i;
	unsigned char value, count, cur_byte;
	kaddr_t ri, wi;
	
	/* initialize the read / write indices */ 
	ri = wi = 0;
	
	/* otherwise decompress using run length encoding */
	while(ri < blk_size) {
		cur_byte = cbuf[ri++];
		if (cur_byte == 0) {
			count = cbuf[ri++];
			if (count == 0) {
				ucbuf[wi++] = 0;
			} else {
				value = cbuf[ri++];
				for (i = 0; i <= count; i++) {
					ucbuf[wi++] = value;
				}
			}
		} else {
			ucbuf[wi++] = cur_byte;
		}

		/* if our write index is beyond the page size, exit out */
		if (wi > KL_PAGE_SIZE) {
			fprintf(KL_ERRORFP,
				"ERROR: Attempted to decompress beyond page boundaries: buffer corrupted!\n");
			return (0);
		}
	}
	/* set return size to be equal to uncompressed size (in bytes) */
	return (int)wi;
}
