/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *  COMDAT functions
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <strings.h>

#include "scnhdr.h"
#include "syms.h"

#include "ldlimits.h"
#include "std.h"
#include "util.h"
#include "errors.h"
#include "driver.h"
#include "symbols.h"
#include "subspaces.h"
#include "comdat.h"

extern Boolean comdat_strict_semantics;
extern Boolean bad_obj;
extern struct symbol_record null_symbol_record;
extern char *base_file_name;
extern char *cur_file_name;
extern Boolean relinkable;

char *get_symbol_filename(int sym_ind);
int get_symbol_module_id(int sym_ind);
void comdat_snazzy_sym_resolve (int, int);

/*
 * Function: comdat_global_conflict
 *
 * Description: Handles error reporting for the case where a comdat symbol
 *              conflicts with a global symbol of the same name.
 *
 * Arguments: global_sym:  the global symbol
 *            comdat_sym:  the comdat symbol
 *
 */

void comdat_global_conflict(int global_sym, int comdat_sym) {
    if (comdat_strict_semantics) {
	bad_obj = TRUE;

	warning(STRICT_CTTI_FAULT,
		Sym_Name(comdat_sym),
		get_symbol_filename(comdat_sym),
		get_symbol_filename(global_sym),
		0);
    } else {
	found_change_warn = TRUE;

	if (verbose & V_DETAIL_CHANGE_WARN) {
	    warning(COMDAT_GLOBAL_OVERRIDE,
		    Sym_Name(comdat_sym),
		    get_symbol_filename(comdat_sym),
		    get_symbol_filename(global_sym),
		    0);
	}
    }
}


/*
 * Function: comdat_process
 *
 * Description: examines a symbol and determines whether the symbol should be
 *              added or not.
 *  
 *              performs any necessary processing on the symbol's subspace
 *              such as nullifying and linking together with a COMDAT set
 *
 * Arguments:   name    - the name of the symbol.
 *              mq      - no clue.
 *              qname   - no clue.
 *              hashval - no clue.
 *              sym_ind - the index of the symbol in question
 *              type    - the type of the symbol
 *
 * Returns:     true if the symbol should be added, false if it was discarded
 *
 */

Boolean comdat_process(char *name,
		       Boolean mq,
		       char *qname,
		       unsigned int hashval,
		       int sym_ind,
		       int type) 
{
    int ind;
    int subsp_ind;

    /* have to ignore these stupid FRU relink things... */
    if (relinkable && base_file_name == cur_file_name) {
	return TRUE;
    }

    /* only code and data symbols are interesting to comdat */
    if ((type != ST_DATA) 
	&& (type != ST_ENTRY)
	&& (type != ST_CODE)) {
	return TRUE;
    }

    if (Sym_Dict(sym_ind).is_comdat) {
	/* a comdat key symbol */

	/* We only initialize the subspaces when we see the key or
	 * ST_COMDAT symbols. This way, we know that the subspace
	 * won't be reinitialized once it is already linked 
	 */

	subsp_ind = Sym_Subsp(sym_ind);
	Subsp_Misc(subsp_ind).comdat_next = BAD_SUBSP;
	Subsp_Misc(subsp_ind).comdat_head = subsp_ind;

	if ((ind = univ_find(name, mq, qname, hashval, type)) != BAD_SYM) {

	    if (Sym_Dict(ind).is_comdat) {
		/* there is already a key symbol for this comdat subspace */
		
		assert(ComdatHead(Sym_Subsp(ind)) == Sym_Subsp(ind));

		/*  Now use unique sequence nums instead of 
		 *              file names for comparison. File names
		 *              fail to work with concatenated SOMs 
		 */

		if (get_module_id(sym_ind) == get_module_id(ind)) {

		   /* We should never have two COMDAT groups in the 
		      same file */

		   external_error(CTTI_DUPLICATE_KEY_SYM, 
				  get_symbol_filename(sym_ind),
				  name,
				  0);
		}
	    } else {
		comdat_global_conflict(ind, sym_ind);
	    } 

	    comdat_nullify_subsp(Sym_Subsp(sym_ind));
	    comdat_snazzy_sym_resolve(sym_ind, ind);
	    
	    return FALSE;
	}

	if ((ind = univ_find(name, mq, qname, hashval, ST_COMDAT)) 
	    != BAD_SYM) {

	    /* 
	     * Even in relocatable links, this assertion should hold. If
	     * there was a previously added set, then the conflict should
	     * have been with the key symbol, not with one of the ST_COMDATs
	     * that is hanging around because of the -r
	     */

	    if (get_module_id(ind) != get_module_id(sym_ind)) {

	       external_error(CTTI_MISSING_KEY_SYM,
			      name,
			      get_symbol_filename(ind),
			      0);
	    }

	    /* link the subspaces */
	    comdat_link_subsp(Sym_Subsp(sym_ind), Sym_Subsp(ind));

	    /* 
	     * we have to treat relocatable links differently -- we
	     * allow the ST_COMDAT symbols to stick around in the final
	     * output.
	     */

	    if (!relocatable) {
		univ_replace(ind, name, mq, qname, hashval, sym_ind, type);
		comdat_nullify_symbol(ind);
	    }

#if 0
	    if (!relocatable) {
		comdat_nullify_symbol(ind);
	    }
#endif

	    /* now return and let the caller add the symbol */
	    return FALSE;
	}
	
	/* no conflicts, so we're going to take this key symbol */
	
	return TRUE;

    } else if ((ind = univ_find(name, mq, qname, hashval, type)) != BAD_SYM) {
	/* not a key symbol, but there is some sort of conflict */

	if (Sym_Subsp(sym_ind) == BAD_SUBSP || Sym_Subsp(ind) == BAD_SUBSP) {
	    /* something weird is going on. Just let the normal routines
	     * try and take care of it 
	     */

	    return TRUE;
	}

	if ((Subsp_Dict(Sym_Subsp(sym_ind)).is_comdat) &&
	    (Subsp_Dict(Sym_Subsp(ind)).is_comdat)) {

	    /* both symbols are in a comdat subspace */

	    if (comdat_is_nullified(Sym_Subsp(ind))) {
		/* old symbol is in a deleted comdat subspace */
		/* discard the old symbol, keep the new */

		assert(Subsp_Misc(Sym_Subsp(ind)).symbol_count == 0);

		univ_switch(ind, sym_ind, hashval);
		comdat_snazzy_sym_resolve(ind, sym_ind);
		
		return TRUE;
	    } else {
		/* the old symbol's subspace is still around, so we
		 * presume that that is the one we are keeping. This
		 * assumption falls apart if we allow multiple conflicting
		 * comdat sets in the same file, so that is not allowed
		 */
				
		assert(get_module_id(sym_ind) != get_module_id(ind));

		comdat_snazzy_sym_resolve(sym_ind, ind);

		return FALSE;
	    }
	} else if (Subsp_Dict(Sym_Subsp(sym_ind)).is_comdat) {
	    
	    /* the new symbol is in a comdat, the old one is not */

	    /* Have the global override the comdat */

	    comdat_snazzy_sym_resolve(sym_ind, ind);

	    return FALSE;
	
	} else if (Subsp_Dict(Sym_Subsp(ind)).is_comdat) {

	    /* The old symbol is in a comdat, the new one is not */

	    if (Sym_Dict(ind).is_comdat) {
		/* if the conflicting symbol is a key symbol, also 
		   erase the comdat section 
		   */

		comdat_global_conflict(sym_ind, ind);
		comdat_nullify_subsp(Sym_Subsp(ind));
	    }
		
	    univ_switch(ind, sym_ind, hashval);
	    comdat_snazzy_sym_resolve(ind, sym_ind);
		
	    return TRUE;
	} else {
	    /* Neither one is in comdat. Just let the normal routines
	     * handle the conflict. */

	    return TRUE;
	}
    
    } else if ((ind = univ_find(name, mq, qname, hashval, ST_COMDAT)) 
	       != BAD_SYM) {

	assert(0);
#if 0
	/*
	 * a "normal" symbol conflicting with an ST_COMDAT symbol. Should
	 * probably never happen, but if it does, we should throw out the 
	 * the ST_COMDAT symbols and its subspace(s).
	 */

	comdat_nullify_subsp(Sym_Subsp(ind));
	comdat_nullify_symbol(ind);

	return TRUE;
#endif

    } else if (comdat_is_nullified(Sym_Subsp(sym_ind))) {
	/* symbol points into a dead comdat subspace */

	comdat_nullify_symbol(sym_ind);
	assert(Subsp_Misc(Sym_Subsp(sym_ind)).symbol_count == 0);

	return FALSE;
    }

    /* it's just a plain ol' symbol */

    return TRUE;
}

/*
 * Function: comdat_st_comdat
 *
 * Description: examines an ST_COMDAT symbol and determines whether 
 *              the symbol should be added or not.
 *  
 *              performs any necessary processing on the symbol's subspace
 *              such as nullifying and linking together with a COMDAT set
 *
 * Arguments:   name    - the name of the symbol.
 *              mq      - no clue.
 *              qname   - no clue.
 *              hashval - no clue.
 *              sym_ind - the index of the symbol in question
 *
 * Returns:     true if the symbol should be added, false if it was discarded
 *
 */

Boolean comdat_st_comdat(char *name, Boolean mq, char *qname,
			 unsigned int hashval, int sym_ind) {
    int ind;
    int subsp_ind;

    /* first, do a little initialization on the subspace */

    subsp_ind = Sym_Subsp(sym_ind);
    assert(Subsp_Dict(subsp_ind).is_comdat);

    Subsp_Misc(subsp_ind).comdat_next = BAD_SUBSP;
    Subsp_Misc(subsp_ind).comdat_head = subsp_ind;

    if ((ind = univ_find(name, mq, qname, hashval, ST_NULL))
	!= BAD_SYM) {
	
	switch (Sym_Dict(ind).symbol_type) {
	  case ST_DATA:
	  case ST_CODE:
	  case ST_ENTRY:

	    if (get_module_id(sym_ind) == get_module_id(ind)) {
		
		/* We should only reach this case if the symbol is
		   a key symbol. We should not have a non-key symbol
		   conflicting with an ST_COMDAT in the same file. */

		assert(Sym_Dict(ind).is_comdat);
		
		/* this symbol is part of an already added set */
		
		comdat_link_subsp(Sym_Subsp(ind), Sym_Subsp(sym_ind));
		
		/* 
		 * For normal links, we throw out duplicate ST_COMDAT
		 * symbols. For -r links we keep them all.
		 */
		
		if (!relocatable) {
		    comdat_nullify_symbol(sym_ind);
		    return FALSE;
		} else {
		    return TRUE;
		}
		
	    } else {
		    
		/* either this COMDAT set has already been added or
		 * it conflicts with a global symbol in another file.
		 * We assume that any messages pertaining to comdat/global
		 * conflicts will be printed when the key symbol is seen.
		 * If the key symbol is missing AND there is a comdat/global
		 * conflict, all bets are off. 
		 */
		
		comdat_nullify_subsp(Sym_Subsp(sym_ind));
		comdat_nullify_symbol(sym_ind);
		    
		return FALSE;
	    }
	    break;

	  default:
	    
	    /* 
	       there shouldn't be a name conflict between
	       a comdat symbol and a non-code or -data symbol */

	    return TRUE;
	    break;
	}
    } else if ((ind = univ_find(name, mq, qname, hashval, ST_COMDAT))
	       != BAD_SYM) {

	if (get_module_id(sym_ind) == get_module_id(ind)) {
	    
	    /* this symbol is part of an already added set w/o 
	       a key symbol */
	    
	    
	    comdat_link_subsp(Sym_Subsp(ind), Sym_Subsp(sym_ind));
		
	    /* 
	     * for normal links, we throw out the duplicate ST_COMDAT,
	     * but we keep it for -r links.
	     */

	    if (!relocatable) {
		comdat_nullify_symbol(sym_ind);
		return FALSE;
	    } else {
		return TRUE;
	    }

	} else {

	    external_error(CTTI_MISSING_KEY_SYM,
			   name,
			   get_symbol_filename(ind),
			   0);
	}
    } else {

	/* there is no symbol of that name, so start a new set */

	return TRUE;
    }

    assert(0);        /* we should have handled all cases by now */
    return TRUE;
}


/*
 * Function: comdat_nullify_subsp
 *
 * Description: removes an unnecessary COMDAT subspace and all others linked
 *              to in the set.
 *
 * Arguments:   subsp_ind - index of the subspace
 *
 */

void comdat_nullify_subsp(int subsp_ind) 
{
    int tmp;
    int index;

    tmp = ComdatHead(subsp_ind);

    while (tmp != BAD_SUBSP) {

	assert(Subsp_Dict(tmp).is_comdat);

	/* move to the start */
	for (index = tmp; index > 0 && Subsp_Dict(index).continuation; index--)
	  ;

	Subsp_Dict(index).subspace_length = 0;
	Subsp_Dict(index).initialization_length = 0;
	Subsp_Dict(index).fixup_request_index = -1;
	Subsp_Dict(index).fixup_request_quantity = 0;
	Subsp_Misc(index).symbol_count = 0;
	Subsp_Misc(index).comdat_nulled = 1;

	/* now remove the rest of the subspaces in this continuation group */

	for (index++; 
	     index < subsp_dict_size && Subsp_Dict(index).continuation; 
	     index++) {
	    
	    Subsp_Dict(index).subspace_length = 0;
	    Subsp_Dict(index).initialization_length = 0;
	    Subsp_Dict(index).fixup_request_index = -1;
	    Subsp_Dict(index).fixup_request_quantity = 0;
	    Subsp_Misc(index).symbol_count = 0;
	    Subsp_Misc(index).comdat_nulled = 1;
	}

	tmp = ComdatNext(tmp);
    }   
}

/*
 * Function: comdat_nullify_symbol
 *
 * Description: removes an unnecessary COMDAT symbol
 *
 * Arguments:   sym_ind - index of the symbol
 *
 */

void comdat_nullify_symbol(int sym_ind) {
    Sym_Dict(sym_ind).symbol_type = ST_NULL;
}

/*
 * Function: comdat_link_subsp
 *
 * Description: links two sets of comdat subspaces together
 *
 * Arguments:   subsp_head - the set whose head will serve as the head of the 
 *                           output set
 *              subsp_ind  - the new set of subspaces to link onto the head
 *
 * Notes:       subsp_head must already be the head of its list of subspaces
 *
 */

void comdat_link_subsp(int subsp_head, int subsp_ind) {
    int tmp1, tmp2;
    int real_head;

    real_head = ComdatHead(subsp_head);

    assert(ComdatHead(real_head) == real_head);
    assert(Subsp_Dict(subsp_head).is_comdat);
    assert(Subsp_Dict(real_head).is_comdat);

    /* get the head of the set to add */
    tmp1 = ComdatHead(subsp_ind);

    /* now point all of the entries to the new head */
    for (tmp2 = tmp1; ComdatNext(tmp2) != BAD_SUBSP; tmp2 = ComdatNext(tmp2)) {
	assert(Subsp_Dict(tmp2).is_comdat);
	ComdatHead(tmp2) = real_head;
    }
    assert(Subsp_Dict(tmp2).is_comdat);
    ComdatHead(tmp2) = real_head;

    /* tmp1 now points to the front of the list to add, tmp2 to the back */
    /* hook the new list in right after the head */
    ComdatNext(tmp2) = ComdatNext(real_head);
    ComdatNext(real_head) = tmp1;
}

/*
 * Function:    comdat_is_nullified
 *
 * Description: determines whether or not a subspace has been nullified
 *
 * Arguments:   subsp_int - the subspace 
 *
 * Returns:     TRUE if it is nullified, FALSE o/w
 *
 */

Boolean comdat_is_nullified(int subsp_ind) {
    assert(subsp_ind != BAD_SUBSP);

    if (Subsp_Dict(subsp_ind).is_comdat && 
	(Subsp_Misc(subsp_ind).comdat_nulled)) {
       
        assert(Subsp_Dict(subsp_ind).subspace_length == 0);
        assert(Subsp_Dict(subsp_ind).initialization_length == 0);
	return TRUE;
    } else {
	return FALSE;
    }
}

/* it not only remaps symbol a to symbol b, it remaps all symbols mapped to
/* a and a to b */
/*
**    now that all Remap_Sym_Records macro is just calling this function,
**    the scope of the function has broadened, causing some changes
**    to its processing.  
*/
void remap_sym_records(int delete_idx, int to_idx) 
{
    struct symbol_record *from, *to;
    int i;

    from = _sym_record[delete_idx];
    to = _sym_record[to_idx];

    /* sometimes this happens, even though it shouldn't.  the
    ** macro used to handle this condition, so the function must
    ** do it as well
    */
    if (to == from) return;

#ifdef DEBUG
    if (from->misc.mapped_to_a && to->misc.mapped_to_a) {
      /**      note: if the condition below happens, then there is
      **      a very high chance that you have changed the code to
      **      have the following statement: Sym_Misc(ind1) = Sym_Misc(ind2)
      **      this is a no-no.  remember that you should at least 
      **      perform a deep copy (not needed in most cases).  
      **      usually, you should just reset the misc of one of the
      **      indices (mapped_to_a, mapped_to_sz, and mapped_to_cnt).
      */
      assert(from->misc.mapped_to_a != to->misc.mapped_to_a);
    }
    for (i=0; i<from->misc.mapped_to_cnt; i++) {
      assert(from->misc.mapped_to_a[i] <= sym_dict_size);
    }
#endif

    assert(to->misc.mapped_to_sz >= to->misc.mapped_to_cnt);
    assert(from->misc.mapped_to_sz >= from->misc.mapped_to_cnt);

    if (!comdat_strict_semantics) {
	if (to->misc.mapped_to_sz == 0) {
	    to->misc.mapped_to_sz = from->misc.mapped_to_cnt + 4;
	    to->misc.mapped_to_a = (int*) malloc((to->misc.mapped_to_sz) *
						 sizeof(int));
	    memset(to->misc.mapped_to_a, '\0', 
		   to->misc.mapped_to_sz * sizeof(int)); 
	} else if (to->misc.mapped_to_sz < (to->misc.mapped_to_cnt + 
					    from->misc.mapped_to_cnt + 1) ) {
	    to->misc.mapped_to_sz = to->misc.mapped_to_cnt + 
	      from->misc.mapped_to_cnt + 4;
	    to->misc.mapped_to_a = (int *) realloc(to->misc.mapped_to_a,
						   (to->misc.mapped_to_sz *
						    sizeof(int)) );
	    memset(to->misc.mapped_to_a + to->misc.mapped_to_cnt,
		   '\0', (to->misc.mapped_to_sz - to->misc.mapped_to_cnt) *
		   sizeof(int) ); 
	}
	
	for (i = 0; i < from->misc.mapped_to_cnt; ++i) {
	    
	    _sym_record[from->misc.mapped_to_a[i]] = _sym_record[to_idx];
	    to->misc.mapped_to_a[to->misc.mapped_to_cnt++] = 
	      from->misc.mapped_to_a[i];
	}

	to->misc.mapped_to_a[to->misc.mapped_to_cnt++] = delete_idx;
	/*  Also need to put the to_idx in
	   the mapped_to_a array so that if we ever happen to
	   reference the sym_record at delete_idx, to_idx will be in
	   the mapped_to_a array 
	*/
	to->misc.mapped_to_a[to->misc.mapped_to_cnt++] = to_idx;

    } else {
	/* 
	 * If we're using strict semantics, there is no need to do all of
	 * the fancy remap tracking, since the most we'll do is one level
	 * of remapping.  
	 */

	assert(from->misc.mapped_to_cnt == 0);
	assert(to->misc.mapped_to_cnt == 0);
	to->misc.mapped_to = TRUE;
    }

    from->misc.mapped_to_cnt = 0;
    Free_Sym_Record(from);
    _sym_record[delete_idx] = to;

    assert(to->misc.mapped_to_sz >= to->misc.mapped_to_cnt);
}
	
/*    
 * This functionis like the old symbol_resolve() except that it not only 
 * resolves symbol a to symbol b, but it also resolved all symbols resolved
 * to symbol a to symbol b
 */

void comdat_snazzy_sym_resolve (int old_index, int new_index)
{
    int i;
    int type, t;
    struct symbol_dictionary_record *old, *new;
    extern void sym_check_match();

    if (old_index == new_index)
        return;
    old = &Sym_Dict (old_index);
    new = &Sym_Dict (new_index);
    type = new->symbol_type;

    if (type == ST_ENTRY ||
	    type == ST_PRI_PROG ||
	    type == ST_SEC_PROG ||
	    type == ST_DATA ||
	    type == ST_CODE) {

	if (old->check_level >= 1 && new->check_level >= 1)
	    sym_check_match (old_index, new_index);
    	/* mark all extension records as NULL too! */
	for (i = old_index+1; i < sym_dict_size; i++) {
	    t = Sym_Dict(i).symbol_type;
	    if (t != ST_SYM_EXT && t != ST_ARG_EXT)
		break;
	    Free_Sym_Record(_sym_record[i]);
	    _sym_record[i] = &null_symbol_record;
	}
    }

    remap_sym_records(old_index, new_index);
} /* end symbol_resolve */


