/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
***************************************************************************
**
** File:         dead_proc.c
**
** Description:  Routines to support dead procedure elimination. 
**
**		 Note! currently routines and data to build and manipulate
**	         a call graph are defined in this file.  If the call graph
**		 is used in the future for other purposes it should be 
**		 defined in its own file.
**
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***************************************************************************
*/
/*
**************************************************************************
**      INCLUDE FILES
**************************************************************************
*/
#include <stdio.h>
#include <string.h>

#ifndef DEBUG
#define NDEBUG
#endif /* !DEBUG */
#include <assert.h>

#include "std.h"
#include "ldlimits.h"
#include "aouthdr.h"
#include "driver.h"
#include "fixups.h"
#include "reloc.h"
#include "scnhdr.h"
#include "syms.h"
#include "spacehdr.h"
#include "shl.h"
#include "spaces.h"
#include "subspaces.h"
#include "symbols.h"
#include "libraries.h"
#include "util.h"
#include "dead_proc.h"

/* doom support */
#include "ld_linkmap.h"

/*
**************************************************************************
**      DEFINE VALUES
**************************************************************************
*/
#define CG_RANGE_CHECK(subsp_index)	assert((subsp_index >= 0) && \
					    (subsp_index < subsp_dict_size));

#define IS_VISITABLE		0x1
#define REFERENCED_IN_FIXUP	0x2 

/* #define OLD_MALLOC */
/* #define NO_RBTREE */
/*
**************************************************************************
**      EXTERNAL 
**************************************************************************
*/

/*
** Enable the default behavior of dead procedure elimination. 
*/
Boolean		delete_dead_procs = FALSE;

/*
**************************************************************************
**      LOCAL 
**************************************************************************
*/

struct callee_entry {
#ifndef OLD_MALLOC 
#ifndef NO_RBTREE
    unsigned int	red:1;
    int			subsp_called:31;
    int			l;
    int			r;
#else
    int			next;		/* index into callee_entry list	*/
    int			subsp_called;	/* index into call_graph list	*/
#endif /* !NO_RBTREE */
#else
#ifndef NO_RBTREE
    unsigned int	red:1;
    int			subsp_called:31;
    struct callee_entry *l;
    struct callee_entry *r;
#else
    struct callee_entry	*next; 
    unsigned int	subsp_called; 
#endif /* !NO_RBTREE */
#endif /* !OLD_MALLOC */
};

struct call_graph_entry {
#ifndef OLD_MALLOC
    int			callee_list;	/* index into callee_entry_list */
#else
    struct callee_entry	*callee_list;	
#endif /* !OLD_MALLOC */
    int 		last_caller_subsp;		
    unsigned short 	num_callers;	/* number of unique callers */
    unsigned short	subsp_info;	/* info about this subsp as callee */
};

static struct call_graph_entry	*call_graph	= NULL;


#ifndef OLD_MALLOC
static struct callee_entry	*callee_entry_list	= NULL;
static int			callee_entry_list_size	= 0;
static int			callee_entry_list_max	= 0;
#endif /* !OLD_MALLOC */

/* 
** Accumulator for size of deleted subspaces. 
*/
static int 			accum_size;		

#ifndef NO_RBTREE
static int	bytes_allocated = 0;
static int	bytes_deleted = 0;

#ifndef OLD_MALLOC
static int	z = BAD_SUBSP;
static int 	gg, g, p, x;
void split(int head, int v);
void rbtreeinsert(int head, int v);
int treesearch(int head, int v);
int rbtreeinitialize(void);
int rotate(int v, int y);
void rbtreetraverse(int head, void (*fptr)(int));

static int      bytes_retrieved = 0;
#else

static struct callee_entry	*z = NULL;
static struct callee_entry	*gg, *g, *p, *x;
void split(struct callee_entry *head, int v);
void rbtreeinsert(struct callee_entry *head, int v);
struct callee_entry *treesearch(struct callee_entry *head, int v);
struct callee_entry *rbtreeinitialize(void);
struct callee_entry *rotate(int v, struct callee_entry *y);
void rbtreetraverse(struct callee_entry *head, void (*fptr)(int));

static int	z_allocated = 0;
static int	head_allocated = 0;
static int	node_allocated = 0;
static int	traverse_deleted = 0;
static int	head_deleted = 0;
#endif /* !OLD_MALLOC */
#endif /* !NO_RBTREE */

static void add_call_graph_fixup(int);

/*
**************************************************************************
**
** init_call_graph()
**
** This routine sets up the roots of a call graph to be constructed off
** the subspaces referencing each other via symbols in fixups.
** Build dynamic array of structs parallel to the subspace table.
** Initialize to zero (NULL pointers, FALSE Booleans, zero counters.
** Initialize is_visitable field as appropriate - used to speed searches.
**
**************************************************************************
*/
static void init_call_graph(void)
{
    int i;  /* loop counter */

    /*
    ** Initialize the fixup to BAD_SUBSP so that the initialization is
    ** not affected by self referencing subspaces.  See add_call_graph_fixup(). 
    */
    fixup_subsp = BAD_SUBSP;

    /*
    ** Allocate an array of call graph entries which parallel the subspace
    ** dictionary.
    */
    call_graph = (struct call_graph_entry *)
    	ecalloc(subsp_dict_size, sizeof(struct call_graph_entry));

    /* 
    ** Set the is_visitable field, to speed up searches over the call graph 
    */
    for (i=0; i < subsp_dict_size; i++) {
	CG_RANGE_CHECK(i);
        call_graph[i].num_callers = 0;
        call_graph[i].last_caller_subsp = BAD_SUBSP;
        call_graph[i].subsp_info = 0;
#ifndef OLD_MALLOC
        call_graph[i].callee_list = BAD_SUBSP;
#else
        call_graph[i].callee_list = NULL;
#endif /* !OLD_MALLOC */

        if (Sub_Space_Dict(i).is_loadable && 
	    !Sub_Space_Dict(i).is_private &&
	    (Subsp_Misc(i).entry_sym != BAD_SYM) &&
	    /*(Subsp_Misc(i).FDP_order_specified == 0) &&*/
	    (Subsp_Dict(i).fixup_request_quantity != 0) ) {

	    call_graph[i].subsp_info |= IS_VISITABLE;
#ifndef NO_RBTREE
#ifdef NO_SAVE_HEAD
            call_graph[i].callee_list = rbtreeinitialize();
#endif /* !NO_SAVE_HEAD */
#endif /* !NO_RBTREE */
        }
    }

    /*
    ** Mark the primary program entry point so that its not eliminated.
    ** Assume that entry_symbol_index will be set properly if it is not
    ** defined such as when we're building shared libraries.
    */
    if (entry_symbol_index != BAD_SYM)
        add_call_graph_fixup(Sym_Subsp(entry_symbol_index));

    /*
    ** If we are building a shared library or an incomplete executable
    ** then make a pass over the export list and mark the subspaces so
    ** that they are not eliminated.  Any exported symbol could be 
    ** potentially be accessed at run time when a shared library is
    ** explicitly loaded.
    */
    if (building_shlib || building_incomp_exec) {

	for (i = (export_entry_count - 1); i >= 0; i--) {

	    /*
	    ** Dont bother checking for type.  All data space subspaces
	    ** will be ignored because they are not visitable when the
	    ** call_graph is processed.
	    */
            add_call_graph_fixup(out_explist_subsp[i]);
        }
    }
}


#ifdef OLD_MALLOC
/*
**************************************************************************
**
** destroy_call_graph()
**
** This routine reclaims the memory from a call graph of
** the subspaces referencing each other via symbols in fixups.
** Run down linked lists of callee entries, freeing them one at a time.
** If this becomes a performance problem, we can buffer them up later.
** Free the "call_graph" array itself.
**
**************************************************************************
*/
static void destroy_call_graph(void)
{
    int 		i;	/* loop counter */
    struct callee_entry *p, *q;	/* working pointers */
    for (i=0; i < subsp_dict_size; i++) {
	CG_RANGE_CHECK(i);
#ifndef NO_RBTREE
        if ((call_graph[i].subsp_info & IS_VISITABLE) && 
	    call_graph[i].callee_list) {
	    efree(call_graph[i].callee_list);
	    call_graph[i].callee_list = NULL;
	    bytes_deleted += sizeof(struct callee_entry);
	    head_deleted += sizeof(struct callee_entry);
	}
#else
	for(p = call_graph[i].callee_list; p != NULL; p = q) {
	    q = p->next;
	    efree(p);
        }
#endif /* !NO_RBTREE */
    }
    efree(call_graph);
    call_graph = NULL;

#ifndef NO_RBTREE
    efree(z);
    bytes_deleted += sizeof(struct callee_entry);
#endif /* !NO_RBTREE */
}
#endif /* OLD_MALLOC */

/*
**************************************************************************
**
** add_call_graph_fixup()
**
** This routine records the fact that the "subsp_index" subspace
** is referenced in a fixup - implying that it cannot be safely
** eliminated as dead code.
**
** Compute the output-order index of the input subspace, adjusted to be 
** relative to the first subsp in the $TEXT$ space.  If "subsp_index" is 
** within $TEXT$, record that this subspace is referenced.
**
**************************************************************************
*/
static void add_call_graph_fixup(int subsp_index)
{
    /* 
    ** protective code for stubs, etc. 
    */
    if (subsp_index == BAD_SUBSP)
	return;

    CG_RANGE_CHECK(subsp_index);

    call_graph[subsp_index].subsp_info |= REFERENCED_IN_FIXUP;
}


/*
**************************************************************************
**
** add_call_graph_arc()
**
** This routine records the fact that the "subsp_index" subspace
** is referenced in a R_*_CALL fixup, building a call graph entry for it.
**
** The fixup_subsp is the subspace from which the procedure is being
** called or referenced.  The callee subspace is the subspace of
** the procedure being called.
**
** Record that "callee_subsp" is called by traversing the linked list 
** of callee records and adding "callee_subsp" if it is not found.
**
**************************************************************************
*/
static void add_call_graph_arc(int fixup_subsp, int callee_subsp)
{
#ifndef OLD_MALLOC
    int p, q;
#else
    struct callee_entry	*p, *q;	/* working pointers */
#endif /* !OLD_MALLOC */

    /* 
    ** protective code for stubs, etc. 
    */
    if (callee_subsp == BAD_SUBSP)
	return;

    CG_RANGE_CHECK(fixup_subsp);
    CG_RANGE_CHECK(callee_subsp);

    /* 
    ** Check that the subspaces are positionable relative to each other 
    */
    if (callee_subsp != fixup_subsp) {		/* suppress self-references */

	/* suppress non-sortable refs */
	if ((Subsp_Dict(callee_subsp).sort_key == 
	    Subsp_Dict(fixup_subsp).sort_key)) { 

	    /* 
	    ** Search list in case this is a duplicate 
	    */
#ifndef NO_RBTREE
#ifndef NO_SAVE_HEAD
	    if (call_graph[fixup_subsp].callee_list == BAD_SUBSP) {
		call_graph[fixup_subsp].callee_list = rbtreeinitialize();
		p = BAD_SUBSP;
            } else
#endif /* !NO_SAVE_HEAD */
	    p = treesearch(call_graph[fixup_subsp].callee_list, callee_subsp);
#else
#ifndef OLD_MALLOC
	    q = BAD_SUBSP; /* trailing subspace index */
	    for(p = call_graph[fixup_subsp].callee_list; 
	        p != BAD_SUBSP && 
		callee_entry_list[p].subsp_called != callee_subsp; 
	        p = callee_entry_list[p].next) {
	    	q = p;
            }
#else
	    q = NULL; /* trailing pointer */
	    for(p = call_graph[fixup_subsp].callee_list; 
	        p != NULL && p->subsp_called != callee_subsp; 
	        p = p->next) {
	    	q = p;
            }
#endif /* !OLD_MALLOC */
#endif /* !NO_RBTREE */

    	    /* 
	    ** if found, forget it - already recorded 
	    */
#ifndef NO_RBTREE
#ifndef OLD_MALLOC
            if (p == BAD_SUBSP)
#else
            if (p == NULL)
#endif /* !OLD_MALLOC */
	        rbtreeinsert(call_graph[fixup_subsp].callee_list, callee_subsp);
#else
#ifndef OLD_MALLOC
	    if (p == BAD_SUBSP) {
	        /* if not found, insert at end of list */
  		/*
		** retrieve a callee_entry.  If storage is not available
		** then allocate more.
		*/
		if (callee_entry_list_size == 0) {
		    callee_entry_list_max = subsp_dict_size;
		    callee_entry_list = (struct callee_entry *)
			emalloc(callee_entry_list_max * 
				sizeof(struct callee_entry));
		} else {
		    if (callee_entry_list_size == callee_entry_list_max) {
			callee_entry_list_max += subsp_dict_size;
			callee_entry_list = (struct callee_entry *)
			    erealloc(callee_entry_list, 
				     callee_entry_list_max * 
				     sizeof(struct callee_entry));
                    }
		}
		p = callee_entry_list_size++;

	        if (q == BAD_SUBSP)
	    	    call_graph[fixup_subsp].callee_list = p; /* empty list */
	        else
	            callee_entry_list[q].next = p;
	        callee_entry_list[p].subsp_called = callee_subsp;
	        callee_entry_list[p].next = BAD_SUBSP;
            } 
#else
	    if (p == NULL) {
	        /* if not found, insert at end of list */
	        p = (struct callee_entry *) 
		    emalloc(sizeof(struct callee_entry));

	        if (q == NULL)
	    	    call_graph[fixup_subsp].callee_list = p; /* empty list */
	        else
	            q->next = p;
	        p->subsp_called = callee_subsp;
	        p->next = NULL;
	    }
#endif /* !OLD_MALLOC */
#endif /* !NO_RBTREE */
        }

        /* 
        ** found or not, positionable or not, a call is a call,
	** but only count each subsp as a caller once and dont count
	** self-referencing subspaces.
        */
        if (call_graph[callee_subsp].last_caller_subsp != fixup_subsp) {
            call_graph[callee_subsp].num_callers++;
    	    call_graph[callee_subsp].last_caller_subsp = fixup_subsp;
        }
    }
}


/*
**************************************************************************
**
** delete_dead_subsp()
**
** Remove the root subspace from the call graph since it has become a
** candidate for removal.  Traverse this subspaces callee list to 
** determine if any other subspaces can be removed.
**
** root_subsp specifies the root subspace which can be removed.
**
**************************************************************************
*/
static void delete_dead_subsp(int root_subsp)
{
#ifndef OLD_MALLOC
    int			p;
#else
    struct callee_entry	*p;	/* working pointer */
#endif /* !OLD_MALLOC */
    int			size;	/* Size of deleted subspace */

    CG_RANGE_CHECK(root_subsp);
    call_graph[root_subsp].subsp_info &= ~IS_VISITABLE;
    accum_size += Subsp_Dict(root_subsp).subspace_length;
    size = Subsp_Dict(root_subsp).subspace_length;
    delete_subsp(root_subsp);

    /* 
    ** Print debug output.  Currently printing out subspace name, entry
    ** name, size of entry in bytes, and the object from which it came.
    */
    if ((verbose & V_DEAD_PROC) && 
	(Subsp_Misc(root_subsp).entry_sym != BAD_SYM)) {
    	printf(ld_gets(1, 1257, "Subspace %s with entry %s and size %d"
	       " from %s deleted as dead code\n"),
	       Subsp_Dict(root_subsp).NAME_PT, 
	       Sym_Name(Subsp_Misc(root_subsp).entry_sym), size,
	       Subsp_Misc(root_subsp).file_name);
    }

#ifndef NO_RBTREE
#ifndef NO_SAVE_HEAD
    if (call_graph[root_subsp].callee_list != BAD_SUBSP)
#endif /* !NO_SAVE_HEAD */
    rbtreetraverse(call_graph[root_subsp].callee_list, delete_dead_subsp);     
#else
#ifndef OLD_MALLOC
    for(p = call_graph[root_subsp].callee_list; 
	p != BAD_SUBSP; 
	p = callee_entry_list[p].next) {
        /* 
	** decrement the number of callers for every callee, test for no 
	** live callers.  The current caller is considered dead, therefore
	** we decrement the number of callers before the check. 
	*/
        CG_RANGE_CHECK(callee_entry_list[p].subsp_called);
	if ((call_graph[callee_entry_list[p].subsp_called].subsp_info 
	     & IS_VISITABLE) &&
	    ((--call_graph[callee_entry_list[p].subsp_called].num_callers)
	     == 0) &&
	    !(call_graph[callee_entry_list[p].subsp_called].subsp_info 
	      & REFERENCED_IN_FIXUP)) {
	    delete_dead_subsp(callee_entry_list[p].subsp_called);
        }
    }
#else
    for(p = call_graph[root_subsp].callee_list; p !=NULL; p = p->next) {
        /* 
	** decrement the number of callers for every callee, test for no 
	** live callers.  The current caller is considered dead, therefore
	** we decrement the number of callers before the check. 
	*/
        CG_RANGE_CHECK(p->subsp_called);
	if ((call_graph[p->subsp_called].subsp_info & IS_VISITABLE) &&
	    ((--call_graph[p->subsp_called].num_callers) == 0) &&
	    !(call_graph[p->subsp_called].subsp_info & REFERENCED_IN_FIXUP)) {

	    delete_dead_subsp(p->subsp_called);
        }
    }
#endif /* !OLD_MALLOC */
#endif /* !NO_RBTREE */
}


/*
**************************************************************************
**
** check_and_delete_subspaces()
**
** This routine sets up the sort order for subspaces by a depth-first
** traversal of the call graph, using various heuristics for ordering.
**
** Set the repositioning sort keys for all code subspaces by doing
** a depth-first traversal of the call graph, positioning subspaces that
** call each other along the call chains together.  Opportunities for
** dead code elimination are also identified and prosecuted.
**
**************************************************************************
*/
static void check_and_delete_subspaces(void)
{
    int i;  /* loop counter */

    /* find and delete all nonempty dead subspaces */
    accum_size = 0;
    for (i=0; i < subsp_dict_size; i++) {
        CG_RANGE_CHECK(i);
	if ((call_graph[i].subsp_info & IS_VISITABLE) &&
	    (call_graph[i].num_callers == 0) &&
	    !(call_graph[i].subsp_info & REFERENCED_IN_FIXUP) )
	    delete_dead_subsp(i);
    }

    if (verbose & V_DEAD_PROC) {
        printf(ld_gets(1, 
		       1258, 
		       "Size of deleted dead subspaces: %d bytes\n"), 
	       accum_size);
    }
}


/*
**************************************************************************
**
** dead_proc_pass()
**
** Routine determines if a symbol has referenced by the current subspace
** being fixed up and calls either add_call_graph_fixup() or 
** add_call_graph_arc() to record the reference.
**
**************************************************************************
*/
Fixup *dead_proc_pass(struct fix_desc	*fix_info, 
		     Fixup		*fixup, 
		     int 		arg0, 
		     int		arg1)
{

    switch (fix_info->type) {
        case R_DATA_ONE_SYMBOL:
        case R_DATA_PLABEL:
        case R_MILLI_REL:
        case R_CODE_PLABEL:
        case R_CODE_ONE_SYMBOL:
            arg0 += sym_bias;
            arg0 = Sym_Remap(arg0);
            add_call_graph_fixup(Sym_Subsp(arg0));
            break;

	case R_DP_RELATIVE:
	    /* 
	    ** This catches an  piece of code in the kernel which found
	    ** the address of a routine like this...
	    **
	    **    addil   L%realmain-$global$,dp
	    **    ldo     R%realmain-$global$(r1),r1
	    */
	    arg0 += sym_bias;
            arg0 = Sym_Remap(arg0);
	    /* Only mark subspace as needed if this symbol is a CODE symbol. */
	    if ((Sym_Type(arg0) == ST_ENTRY)	|| 
		(Sym_Type(arg0) == ST_CODE)	||
		(Sym_Type(arg0) == ST_PRI_PROG)	||
		(Sym_Type(arg0) == ST_SEC_PROG)) {

	        add_call_graph_fixup(Sym_Subsp(arg0));
	    }
	    break;

        case R_COMP2: 
        case R_PCREL_CALL:
        case R_ABS_CALL:
            arg1 += sym_bias;
            arg1 = Sym_Remap(arg1);
            add_call_graph_arc(fixup_subsp, (Sym_Subsp(arg1)));
            break;

    }
    return (NULL);
}


/*
**************************************************************************
**
** do_dead_proc_elim()
**
** Routine performs dead procedure elimination.  First initializes the
** call graph mechanism, then traverses each subspace and looks for
** references in the fixup stream.  Once all the subspaces have been
** traversed the unreferenced subspaces are removed and the subspaces are
** re-sorted. Lastly, all the allocated memory is free'ed up.
**
**************************************************************************
*/
void do_dead_proc_elim(void)
{
    int subsp_index;

    /*
    ** Allocate call graph and initialize call graph by marking subspaces
    ** as visitable.  Also, pre-mark any exported symbols and primary
    ** entry points.
    */
    init_call_graph();

    /* for all loadable, shared (e.g., code) subspaces */
    for (subsp_index = (subsp_dict_size-1); (subsp_index>=0) ; subsp_index--) {

    /*
    ** For now go through all the subspaces.  This will protect us against
    ** the debugger situations and any other unloadable subspaces that
    ** other people might be using.
    */
#if 1
	/* 
	** skip non-loadable subspaces, but $DATA$ can have 
	** R_DATA_PLABEL 
	*/

	/* doom support */
        if (!strip_debug && lm_should_skip_subspace(subsp_index)) {
	      continue;  
        } else if (strip_debug &&  
                (!Sub_Space_Dict(subsp_index).is_loadable))
	       continue;  
#endif /* 1 */

        if (Subspace_Length(subsp_index) != 0) /* prevent useless call */
            traverse_fixups(subsp_index, DEAD_PROC_PASS);

    } /* loop over subspaces */

    /* check each subspace and delete any dead subspaces */
    check_and_delete_subspaces();  

    /* resort according to new procedure order */
    subspaces_sort();  

#ifndef OLD_MALLOC
    if (callee_entry_list) {
	efree(callee_entry_list);
	callee_entry_list = NULL;
    }
    if (call_graph) {
        efree(call_graph);
        call_graph = NULL;
    }
#else
    destroy_call_graph();
#endif /* !OLD_MALLOC */

#ifndef NO_RBTREE
#ifndef OLD_MALLOC
    if (verbose & V_MALLOC) {
        printf("memory allocated for dead procedure elimination: %d bytes\ndeleted: %d bytes\n", 
	    bytes_allocated, bytes_deleted);
#ifdef DEBUG
        printf("bytes used for call graph nodes: %d\n", bytes_retrieved);
#endif /* DEBUG */
    }
#endif /* !OLD_MALLOC */
#endif /* !NO_RBTREE */
}

#ifndef NO_RBTREE
#ifndef OLD_MALLOC
static int get_callee_entry(void)
{
    if (callee_entry_list_size == 0) {
        callee_entry_list_max = (subsp_dict_size/2);
        callee_entry_list = (struct callee_entry *)
        emalloc(callee_entry_list_max * sizeof(struct callee_entry));
	bytes_deleted = bytes_allocated = 
	    (callee_entry_list_max * sizeof(struct callee_entry));
    } else {
        if (callee_entry_list_size == callee_entry_list_max) {
            callee_entry_list_max += (subsp_dict_size/2);
            callee_entry_list = (struct callee_entry *)
                erealloc((char *) callee_entry_list,
                         callee_entry_list_max * sizeof(struct callee_entry));
	    bytes_deleted = bytes_allocated = 
	      (callee_entry_list_max * sizeof(struct callee_entry));
        }
    }
    bytes_retrieved += sizeof(struct callee_entry);
    return callee_entry_list_size++;
}
#else
static struct callee_entry *get_callee_entry(void)
{
    bytes_allocated += sizeof(struct callee_entry);
    return (struct callee_entry *) emalloc(sizeof(struct callee_entry));
}
#endif /* !OLD_MALLOC */

#ifndef OLD_MALLOC
void rbtreeinsert(int head, int v)
{
    x = head;
    p = head;
    g = head;

    while (x != z) {
	gg = g;
	g = p;
	p = x;
	x = (v < callee_entry_list[x].subsp_called) ? 
            callee_entry_list[x].l : callee_entry_list[x].r;
	if (callee_entry_list[callee_entry_list[x].l].red && 
            callee_entry_list[callee_entry_list[x].r].red) {
	    split(head, v);
        }
    }

    x = get_callee_entry();
    callee_entry_list[x].subsp_called = v;
    callee_entry_list[x].l = z;
    callee_entry_list[x].r = z;

    if (v < callee_entry_list[p].subsp_called) 
        callee_entry_list[p].l = x; 
    else 
    	callee_entry_list[p].r = x;

    split(head, v);
}

int rotate(int v, int y)
{
    int c, gc;

    c = (v < callee_entry_list[y].subsp_called) ? 
        callee_entry_list[y].l : callee_entry_list[y].r;
    if (v < callee_entry_list[c].subsp_called) {
	gc = callee_entry_list[c].l;
	callee_entry_list[c].l = callee_entry_list[gc].r;
	callee_entry_list[gc].r = c;
    } else {
	gc = callee_entry_list[c].r;
	callee_entry_list[c].r = callee_entry_list[gc].l;
	callee_entry_list[gc].l = c;
    }
    if ( v < callee_entry_list[y].subsp_called )
	callee_entry_list[y].l = gc;
    else
	callee_entry_list[y].r = gc;

    return gc;	
}

void split(int head, int v) 
{
    callee_entry_list[x].red = 1;
    callee_entry_list[callee_entry_list[x].l].red = 0;
    callee_entry_list[callee_entry_list[x].r].red = 0;
    if (callee_entry_list[p].red) {
	callee_entry_list[g].red = 1;
	if (v < callee_entry_list[g].subsp_called != 
            v < callee_entry_list[p].subsp_called)
	    p = rotate(v,g);
	x = rotate(v,gg);
	callee_entry_list[x].red = 0;
    }
    callee_entry_list[callee_entry_list[head].r].red = 0;
}

int rbtreeinitialize(void) 
{
    int head;

    if (z == BAD_SUBSP) {
        z = get_callee_entry();
        callee_entry_list[z].l = z;
        callee_entry_list[z].r = z;
        callee_entry_list[z].red = 0;
        callee_entry_list[z].subsp_called = BAD_SUBSP;
    }

    head = get_callee_entry();
    callee_entry_list[head].l = z;
    callee_entry_list[head].r = z;
    callee_entry_list[head].red = 0;
    callee_entry_list[head].subsp_called = BAD_SUBSP;

    return head;
}

treesearch(int head, int v)
{
    int x = callee_entry_list[head].r; 
    while (x != z && v != callee_entry_list[x].subsp_called) {
	x = ( v < callee_entry_list[x].subsp_called ) ? 
            callee_entry_list[x].l : callee_entry_list[x].r;
    }

    if (callee_entry_list[x].subsp_called == BAD_SUBSP)
	return BAD_SUBSP;
    else
        return x;
}

void rbtreetraverse(int head, void (*fptr)(int))
{
    if (head != z) {
	rbtreetraverse(callee_entry_list[head].l, fptr);

	if (callee_entry_list[head].subsp_called != BAD_SUBSP) {
	    CG_RANGE_CHECK(callee_entry_list[head].subsp_called);
	    if ((call_graph[callee_entry_list[head].subsp_called].subsp_info & IS_VISITABLE) &&
	        ((--call_graph[callee_entry_list[head].subsp_called].num_callers) == 0) &&
	        !(call_graph[callee_entry_list[head].subsp_called].subsp_info & REFERENCED_IN_FIXUP))
	        (*fptr)(callee_entry_list[head].subsp_called);
        }

	rbtreetraverse(callee_entry_list[head].r, fptr);
    }
}

#else /* start OLD MALLOC */

void rbtreeinsert(struct callee_entry *head, int v)
{
    x = head;
    p = head;
    g = head;

    while (x != z) {
	gg = g;
	g = p;
	p = x;
	x = (v < x->subsp_called) ? x->l : x->r;
	if (x->l->red && x->r->red) {
	    split(head, v);
        }
    }

    x = get_callee_entry();
    node_allocated += sizeof(struct callee_entry);
    x->subsp_called = v;
    x->l = z;
    x->r = z;

    if (v < p->subsp_called) 
        p->l = x; 
    else 
    	p->r = x;

    split(head, v);
}

struct callee_entry *rotate(int v, struct callee_entry *y)
{
    struct callee_entry *c, *gc;

    c = (v < y->subsp_called) ? y->l : y->r;
    if (v < c->subsp_called) {
	gc = c->l;
	c->l = gc->r;
	gc->r = c;
    } else {
	gc = c->r;
	c->r = gc->l;
	gc->l = c;
    }
    if ( v < y->subsp_called )
	y->l = gc;
    else
	y->r = gc;

    return gc;	
}

void split(struct callee_entry *head, int v) 
{
    x->red = 1;
    x->l->red = 0;
    x->r->red = 0;
    if (p->red) {
	g->red = 1;
	if (v < g->subsp_called != v < p->subsp_called)
	    p = rotate(v,g);
	x = rotate(v,gg);
	x->red = 0;
    }
    head->r->red = 0;
}

struct callee_entry *rbtreeinitialize(void) 
{
    struct callee_entry *head;

    if (!z) {
	z_allocated += sizeof(struct callee_entry);
        z = get_callee_entry();
        z->l = z;
        z->r = z;
        z->red = 0;
        z->subsp_called = BAD_SUBSP;
    }

    head = get_callee_entry();
    head->l = z;
    head->r = z;
    head->red = 0;
    head->subsp_called = BAD_SUBSP;
    head_allocated += sizeof(struct callee_entry);

    return head;
}

struct callee_entry *treesearch(struct callee_entry *head, int v)
{
    struct callee_entry *x = head->r; 
    while (x != z && v != x->subsp_called) {
	x = ( v < x->subsp_called ) ? x->l : x->r;
    }

    if (x->subsp_called == BAD_SUBSP)
	return NULL;
    else
        return x;
}

void rbtreetraverse(struct callee_entry *head, void (*fptr)(int))
{
    if (head != z) {
	rbtreetraverse(head->l, fptr);

	if (head->subsp_called != BAD_SUBSP) {
	    CG_RANGE_CHECK(head->subsp_called);
	    if ((call_graph[head->subsp_called].subsp_info & IS_VISITABLE) &&
	        ((--call_graph[head->subsp_called].num_callers) == 0) &&
	        !(call_graph[head->subsp_called].subsp_info & REFERENCED_IN_FIXUP))
	        (*fptr)(head->subsp_called);
        }

	rbtreetraverse(head->r, fptr);

	efree(head);
	bytes_deleted += sizeof(struct callee_entry); 
	traverse_deleted += sizeof(struct callee_entry);
    }
}
#endif /* !OLD_MALLOC */

#endif /* !NO_RBTREE */
