/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *  Embedded Systems Support Routines
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef DEBUG
#define NDEBUG
#endif  /* !DEBUG */
#include <assert.h>

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "aouthdr.h"
#include "filehdr.h"
#include "spacehdr.h"
#include "scnhdr.h"
#include "compunit.h"
#include "reloc.h"
#include "syms.h"

#include "std.h"
#include "embed.h"
#include "ldlimits.h"
#include "allocate.h"
#include "driver.h"
#include "errors.h"
#include "fixups.h"
#include "libraries.h"
#include "ldnls.h"
#include "linker.h"
#include "output.h"
#include "spaces.h"
#include "ld_strings.h"
#include "stubs.h"
#include "subspaces.h"
#include "symbols.h"
#include "util.h"

/* This should go in driver.h when I want to recompile */
#define        DEFAULT_ENTRY_NAME      "$START$"
#define        DEFAULT_SUBSP_ALIGN     8

/* This shouldn't */
#define MAX_WORD_SIZE     1024

/* Forward Declarations */
FILE *open_file();
int get_cmd_line_args();
void read_cmd_file();

/* Externals */
extern char **argv;
extern int argc;
extern argv_size;
extern process_cmd_line_args();
extern char *split_name();
extern int qualify_subsp();
extern Boolean fgetword();
extern char *get_next_word();

/* Globals */
static char **nargv;
static int nargc = 0;
Boolean embed_subsp_map = FALSE;
Boolean embed_sym_map = FALSE;
Boolean copyram_created = FALSE;         /* copyram subspace already created? */
int copyram_subsp = BAD_SUBSP;        /* subspace index of copyram subsp */

int nargv_size = 1000;

/* Initialize keyword structure */
#define C_NOMATCH     0

#define C_ALIGN       1
#define C_ENTRY       2
#define C_UNSAT       3
#define C_INCLUDE     4
#define C_MAP         5
#define C_OBJECTS     6
#define C_RESERVE     7
#define C_RENAME      8
#define C_SUBSPACE    9
#define C_SPACE       10

/* qualifiers (see cmd struct initialization for which go with what */
/* These are the indices into the qualifier string array */
#define Q_NONE        0
#define Q_ALIGN       1
#define Q_AT          2
#define Q_COPYRAM     3
#define Q_SUBSPACE    4
#define Q_SYMBOLS     5
#define Q_ALL         6
#define Q_RELOCATABLE 7
#define Q_NOCOMPRESS  8
#define Q_UNLOADABLE  9

char *qual_strings[] = { 
		       NULL,
		       "align",
		       "at",
		       "copyram",
		       "subspaces",
		       "symbols",
		       "all",
		       "relocatable",
		       "nocompress",
		       "unloadable",
		       NULL };

/* If this changes the init of the cmd structure must change */
#define MAX_QUAL_PER_CMD        7


/* type definition of an command */
typedef struct cmd {
    char *string;          /* actual command string */
    int name;              /* token for this command */
    int msgnum;            /* message catalog number of descriptive message */
    int qual_list[MAX_QUAL_PER_CMD];   /* list of qualifiers for this cmd */
} cmd_type,*cmd_type_ptr;

/* options holds the valid options for the cmd line */
/* the order given is the order of string matching. If 'a' is given as */
/* an option than the first word in this list which matches the 'a' is used */
/* This means if an option is a subset of another option the subset goes */
/* before the longer string in the list.                                */
/* Note qualifiers is an array initialized up to a Q_NONE */
/*                                                                      */
/*    string             name          msgnum      possible qualifiers */
static cmd_type commands[] = {   
   {"align",           C_ALIGN,         0,        {Q_NONE}},
   {"entry",           C_ENTRY,         0,        {Q_NONE}}, 
   {"unsat",           C_UNSAT,         0,        {Q_NONE}}, 
   {"include",         C_INCLUDE,       0,        {Q_ALIGN,
						    Q_NONE}}, 
   {"map",             C_MAP,           0,        {Q_SUBSPACE,
				                    Q_SYMBOLS,
				                    Q_ALL,
						    Q_NONE}},
   {"options",         C_OBJECTS,       0,        {Q_NONE}}, 
   {"reserve",         C_RESERVE,       0,        {Q_NONE}}, 
   {"rename",          C_RENAME,        0,        {Q_NONE}}, 
   {"subspace",        C_SUBSPACE,      0,        {Q_ALIGN,
						    Q_NONE}},
   {"space",           C_SPACE,         0,        {Q_AT,
				                    Q_ALIGN,
				                    Q_RELOCATABLE,
				                    Q_COPYRAM,
				                    Q_NOCOMPRESS,
				                    Q_UNLOADABLE,
						    Q_NONE}},
   {NULL, 0, 0, 0}
   };


/* Alias List */
#define MAX_ALIAS_LIST_SIZE  10

#define ALIAS_NAME           -1

struct alias_table {
    char *name;               /* Name of alias or subspace to alias */
    char *file_name;          /* Name of file in alias name */
    char *module_name;        /* Name of module in alias name */
    int index;                /* Index into alias table (points to alias) */
};

struct alias_table *alias_list;
int alias_list_size = MAX_ALIAS_LIST_SIZE;
int num_aliases = 0;

/* Reserve List */
#define MAX_RESERVE_LIST_SIZE  10

#define RESERVE_NAME           -1

struct reserve_table {
    unsigned int start_addr;     /* start address of region to reserve */
    unsigned int end_addr;       /* end address of region to reserve */
};

struct reserve_table *reserve_list;
int reserve_list_size = MAX_RESERVE_LIST_SIZE;
int num_reserve_entries = 0;
Boolean reserve_regions_seen = FALSE;

/* extern*/ int line_num = 35;   /*counter used to give a unique name*/


#define ALIGN_16      16          /* used to check_align */
#define ALIGN_32      32

/* Locals for parsing */
#define NOT_SEEN       -1
static int cur_space = NOT_SEEN;           /* Index of current space seen    */
static char cur_space_name[100] = {'\0'};  /* Name of current space seen     */
static int cur_subspace = NOT_SEEN;        /* Index of current subspace seen */
static int cur_align = DEFAULT_SUBSP_ALIGN;     /* Current align seen   */
static int cur_subspace_order = -0x0fffffff;    /* Starting subsp order */
                                                /* 0 is normal subsp    */
static int copyram_space = FALSE;           /* Whether cur_space is copyram  */

static int reserve_counter = 0;   /* counter used to give a unique name  */
				  /* when a RESERVE is seen              */

static Boolean module_match_when_null = TRUE;   /* filename match will match */
                                                /* a null module name with   */
						/* anything. */

static unsigned int last_start_addr_seen = 0; /* save off last addr for space*/



void embed_initialize()
{
    if (alias_list == 0) {
        /* allocate alias list */
        alias_list = (struct alias_table *)
	    emalloc(alias_list_size * sizeof(struct alias_table));
    }
    num_aliases = 0;

    if (reserve_list == 0) {
        /* allocate reserve_list */
        reserve_list = (struct reserve_table *)
	    emalloc(reserve_list_size * sizeof(struct reserve_table));
    }
    num_reserve_entries = 0;
    reserve_regions_seen = FALSE;

    reserve_counter = 0;   /* reset reserve counter for unique subsp names */

    /* init starting addresses for spaces - used for sorting */
    last_start_addr_seen = 0; 
}

void process_command_file(type, char_ptr)
int type;
char *char_ptr;
{
    FILE *f;
    char **temp_argv;
    int temp_argc = 0;

    /* Save off original arglist */
    temp_argv = argv;
    temp_argc = argc;

    embed_initialize();

    cur_name = char_ptr;

    if ( type == AS_A_FILE) {
	read_cmd_file(char_ptr);
	if (verbose & V_DEBUG)
            printf("Filename '%s' Read in.\n", char_ptr);
    } else {    /* AS_A_STRING */
	/* Parse the string pointed to by char_ptr */
	if (verbose & V_DEBUG)
            printf("String '%s' Ignored - command files are not implemented.\n"
								   , char_ptr);
    }

    /* Restore original cmd line args */
    /* free(old argv) */
    argv = temp_argv;
    argc = temp_argc;
}


void read_cmd_file(fn)
char *fn;
{
    FILE *f;
    int end_condition;       /* EOF  for -F, '\n' for -k */

    char buf[1024];           /* the argument buffer */
    char temp_buf[1024];      /* a temp buffer */

    char *c;           /* temp char ptr */
    int temp_int;      /* temp integer */
    int temp_subsp_index;      /* temp integer */

    cmd_type_ptr cur_cmd;  /* the current command seen */
    int qual;          /* temp qualifier number */

    Boolean name_seen;     /* check if a name was given */
    Boolean at_seen;       /* check if the at qualifier was given */

    cmd_type_ptr match_command();
    int match_qualifier();
    char *file_name;     /* file_name portion of subspace specification */
    char *module_name;   /* module name portion of subspace specification */

    Boolean is_private;   /* used in space - if $PRIVATE$ is specified */
    Boolean is_text;      /* used in space - if $TEXT$ is specified */

    int target_index = 0;       /* used in rename */

    end_condition = '\n';    /* for now */


    /* open the file */
    f = efopen(fn,"r",CANT_OPEN);

    /* while there are more words in the file ... */
    while (fgetword(f, buf, EOF)) {
        if ((cur_cmd = match_command(buf)) != C_NOMATCH) {

	  switch (cur_cmd->name) {

            case C_ALIGN:
	        if (cur_space == NOT_SEEN) {
		    user_error(SPACE_NOT_SEEN, cur_cmd->string, fn, 0);
		}
	        if (get_num(f, &cur_align, "%x") != 0) {
		    check_align(ALIGN_16, cur_align, fn);

		    if (verbose & V_EMBED) 
	  	        printf("subspace Align set to %x\n", cur_align);
		} else
		    user_error(BAD_NUM_ARG, cur_cmd->string, 0);
	        break;

            case C_ENTRY:         
		if ((entry_name=get_next_word(f)) == NULL) 
	            entry_name = DEFAULT_ENTRY_NAME;

		if (verbose & V_EMBED)
	  	    printf("entry_name set to %s\n", entry_name);
                break;

            case C_INCLUDE:       
		if (cur_space == NOT_SEEN)
		    user_error(SPACE_NOT_SEEN, cur_cmd->string, fn, 0);
	        if (cur_subspace == NOT_SEEN)
		    user_error(SUBSPACE_NOT_SEEN, cur_cmd->string, fn, 0);
	        name_seen = FALSE;
		temp_subsp_index = NOT_SEEN; 

		while ((c = get_next_word(f)) != NULL) {
		    switch (qual=match_qualifiers(c, cur_cmd->qual_list)) {
			case Q_ALIGN:
		            if (get_num(f, &temp_int, "%x") != 0) {
		                check_align(ALIGN_16, temp_int, fn);

				if (verbose & V_EMBED)
				    printf("Align set to %x\n", temp_int);
			    } else
                                user_error(BAD_NUM_ARG, qual_strings[qual], 0);
			    if (temp_subsp_index == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
					   cur_cmd->string, 
					   qual_strings[qual], 
					   fn, 
					   0);
			    else {
                                Subsp_Misc(temp_subsp_index).force_new_align 
								    = TRUE;
			        Subsp_Dict(temp_subsp_index).alignment 
								    = temp_int;
			    }
			    break;

                        default:
			    if (qual || name_seen)
		                warning(UNRECOGNIZED_QUAL, 
					cur_cmd->string, 
					c, 
					fn, 
					0);
			    else {

			        /* must be a subspace name */
			        /* c gets subspace name file_name get file */
			        /* and obj name */
			        c = split_name(c, &file_name, &module_name);

		                temp_subsp_index = build_subspace(c,
			                cur_space_name, TEXT_SORTKEY, 
						WILDCARD_ACCESS, 
				    		CODE_QUADRANT, 10, 
				    		1, 
				    		cur_subspace_order++,0);
			
			        if ((file_name != NULL) || (module_name != NULL))
				    qualify_subsp(temp_subsp_index, 
						       file_name, module_name);

			        /* Put this subsp on first names list - ignore ret*/
			        module_match_when_null = FALSE;
			        find_subspace(temp_subsp_index);
			        module_match_when_null = TRUE;

			        Subsp_Misc(temp_subsp_index).input_subsp_only = 1;
			        /* mark out sub so it will appear */
			        Subsp_Misc(cur_subspace).input_subsp_only = 2;

			        /* Reset align for following subspaces */
			        name_seen = TRUE;
				}
			    break;
			}
		    }

		if (!name_seen)
		    user_error(NAME_NOT_SEEN, 
		                   cur_cmd->string, fn, 0);
		break;

            case C_MAP:       
		while ((c = get_next_word(f)) != NULL) {
		    switch (qual=match_qualifiers(c, cur_cmd->qual_list)) {
			case Q_SUBSPACE:
		            embed_subsp_map = TRUE;
			    break;
			case Q_SYMBOLS:
		            embed_sym_map = TRUE;
			    break;
			case Q_ALL:
		            embed_sym_map = TRUE;
		            embed_subsp_map = TRUE;
			    break;
			default:
		            warning(UNRECOGNIZED_QUAL, 
				    cur_cmd->string, 
				    c, 
				    fn, 
				    0);
			    break;
		    }
		}
		break;

            case C_UNSAT:         
		while ((c = get_next_word(f)) != NULL)
	            undef_list_add(c);
		break;

            case C_OBJECTS:       
		get_cmd_line_args(f, NULL);   
	        process_cmd_line_args();
	        break;

            case C_RESERVE:       
		if (cur_space == NOT_SEEN) {
                    /* Reserve seen outside a space - define an entry for */
                    /* the region the user wants to reserve */

                    /* do we have to expand the list? */
		    if (num_reserve_entries >= reserve_list_size) {
                        reserve_list_size += MAX_RESERVE_LIST_SIZE;
                        /* Expand reserve list */
                        reserve_list = (struct reserve_table *)
                             erealloc((char *) reserve_list,
                                  reserve_list_size * 
                                          sizeof(struct reserve_table));
                    }

                    /* get start addr */
		    if (get_num(f, &temp_int, "%x") == 0) 
			user_error(BAD_NUM_ARG, cur_cmd->string, 0);

                    reserve_list[num_reserve_entries].start_addr = temp_int;

                    /* get length */
		    if (get_num(f, &temp_int, "%x") == 0) 
			user_error(BAD_NUM_ARG, cur_cmd->string, 0);

                    reserve_list[num_reserve_entries].end_addr = 
                                 reserve_list[num_reserve_entries].start_addr +
                                 temp_int - 1;

		    num_reserve_entries++;    /* advance counter */

		    reserve_regions_seen = TRUE;

		    if (verbose & V_EMBED) {
		        printf("Reserve start set to %x\n", 
                                 reserve_list[num_reserve_entries].start_addr);
		        printf("Reserve end set to %x\n", 
                                 reserve_list[num_reserve_entries].end_addr);
                    }
	        } else {
		    if (get_num(f, &temp_int, "%x") != 0) {
                        sprintf(temp_buf, "$RESERVE$", reserve_counter++);

		        cur_subspace = 
			      build_subspace(temp_buf, 
					     cur_space_name, 
					     TEXT_SORTKEY, 
					     WILDCARD_ACCESS,
					     DATA_QUADRANT, 
					     20, 
					     cur_align, 
					     cur_subspace_order++, 
					     temp_int);
		        Subsp_Misc(cur_subspace).input_subsp_only = 2;
			if (verbose & V_EMBED)
		            printf("Reserve: Space seen\n");
		    } else
			user_error(BAD_NUM_ARG, "reserve", 0);
 	        }
	        break;

            case C_RENAME:      
		/* Set up alias_list for get_alias_list to use when called */
		/* from subspace_add */

                /* Skip an entry for the target alias */
		target_index = num_aliases++;

		while ((c = get_next_word(f)) != NULL) {
		    if (num_aliases >= alias_list_size) {
		        alias_list_size += MAX_ALIAS_LIST_SIZE;
			/* Expand alias list */
			alias_list = (struct alias_table *) 
			     erealloc((char *) alias_list, 
				  alias_list_size * sizeof(struct alias_table));
		    }
 
		    /* split name into file/obj and name */
		    c = split_name(c, &file_name, &module_name);

                    alias_list[num_aliases].name = c;
                    alias_list[num_aliases].file_name = file_name;
                    alias_list[num_aliases].module_name = module_name;

		    /* Index gets target name */
                    alias_list[num_aliases++].index = target_index;
   	        }
	
		/* take last entry and put it into target entry */
		/* decrement num_aliases so last entry doesn't count */
                alias_list[target_index].name = 
					 alias_list[--num_aliases].name;
                alias_list[target_index].file_name = 
					 alias_list[num_aliases].file_name;
                alias_list[target_index].module_name = 
					 alias_list[num_aliases].module_name;
                alias_list[target_index].index = ALIAS_NAME;

		if ((num_aliases - target_index) < 2)
		    user_error(NOT_ENOUGH_NAMES, fn, 0);
	        break;

            case C_SUBSPACE:       
	        name_seen = FALSE;
		cur_subspace = -1; 
		if (cur_space == NOT_SEEN) {
		    user_error(SPACE_NOT_SEEN, cur_cmd->string, fn, 0);
		}

		while ((c = get_next_word(f)) != NULL) {
		    switch (qual=match_qualifiers(c, cur_cmd->qual_list)) {
			case Q_ALIGN:
		            if (get_num(f, &temp_int, "%x") != 0) {
		                check_align(ALIGN_16, temp_int, fn);

				if (verbose & V_EMBED)
				    printf("Align set to %x\n", temp_int);
			    } else
                                user_error(BAD_NUM_ARG, qual_strings[qual], 0);
			    if (cur_subspace == -1)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    else  {
                                Subsp_Misc(cur_subspace).force_new_align = TRUE;
			        Subsp_Dict(cur_subspace).alignment = temp_int;
			    }
			    break;

                        default:
			    if (qual || name_seen)
		                warning(UNRECOGNIZED_QUAL, 
					cur_cmd->string, 
					c, 
					fn, 
					0);
			    else {

			        /* must be a subspace name */
		                cur_subspace = build_subspace(c,
			                cur_space_name, TEXT_SORTKEY, 
						WILDCARD_ACCESS,
				    		CODE_QUADRANT, 10, 
				    		cur_align, 
				    		cur_subspace_order++,0);

			        /* Put this subsp on first names list - ignore ret*/
			        find_subspace(cur_subspace);
    
                                /* If the align cmd was used make sure the new */
			        /* align is enforced */
			        if (cur_align != DEFAULT_SUBSP_ALIGN) {
                                  /*Subsp_Misc(cur_subspace).force_new_align = TRUE;*/
 		                  Subsp_Misc(cur_subspace).input_subsp_only = 2;
				  }

			        /* Reset align for following subspaces */
			        cur_align = DEFAULT_SUBSP_ALIGN;
			        name_seen = TRUE;
				}
			    break;
			}
		    }

		if (!name_seen)
		    user_error(NAME_NOT_SEEN, 
		               cur_cmd->string, 
			       fn, 
			       0);
		break;

            case C_SPACE:       
	        name_seen = FALSE;
	        at_seen = FALSE;
		/* Unimplemented */
	 	cur_space = NOT_SEEN;    /* new space */
	 	cur_subspace = NOT_SEEN;    /* new subspace will be needed  */
		copyram_space = FALSE;  /* assume false till proven otherwise */

		while (fgetword(f, temp_buf, '\n') != 0) {
		    switch (qual=match_qualifiers(temp_buf, 
						        cur_cmd->qual_list)) {
			case Q_ALIGN:
		            if (get_num(f, &temp_int, "%x") != 0) {
		                check_align(ALIGN_32, temp_int, fn);

				if (verbose & V_EMBED)
				    printf("Align set to %x\n", temp_int);
				}
			    else
                                user_error(BAD_NUM_ARG, qual_strings[qual], 0);
			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    else 
			        space_misc[cur_space].space_align = temp_int;
				
				
			    break;

			case Q_AT:
		            if (get_num(f, &temp_int, "%x") != 0) {
				if (verbose & V_EMBED)
				    printf("At set to %x\n", temp_int);
				}
			    else
                                user_error(BAD_NUM_ARG, qual_strings[qual], 0);

			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    else {
			        space_misc[cur_space].start_addr_specified = TRUE;
				space_misc[cur_space].start_virt_addr = temp_int;
				/* save off last for any spaces with no */
				/* addr specified so they will sort just */
				/* after the specified one */
				last_start_addr_seen = temp_int;
				}
                            at_seen = TRUE;
			    break;

			case Q_COPYRAM:
			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);

                            /* No longer need to get ram address...get it
			       from at keyword 

		            if (get_num(f, &temp_int, "%x") != NULL) {
				if (verbose & V_EMBED)
				    printf("Copyram set to %x\n", temp_int);
			        space_misc[cur_space].copyram_needed = TRUE;
			        space_misc[cur_space].ram_location = temp_int;
			    } else
                                user_error(BAD_NUM_ARG, qual_strings[qual], 0);
			    */

			    space_misc[cur_space].copyram_needed = TRUE;
			    /* If !at_seen, ram_loc set in allocate_virt_addr */
			    if (at_seen) {
			    	space_misc[cur_space].ram_location =
				       space_misc[cur_space].start_virt_addr;
                            }
				
			    if (verbose & V_EMBED)
				if (at_seen)
			           printf("Copyram set to %x\n", 
					   space_misc[cur_space].ram_location);
				else
			           printf("Copyram set to next unreserved address\n");

                            copyram_space = TRUE;
#if 0
			    if (!copyram_created) {
				copyram_subsp = 
					   find_subspace_by_name("$COPYRAM$");
				if (copyram_subsp == BAD_SUBSP) {
		                    copyram_subsp = 
				       build_subspace("$COPYRAM$",
				                      "$TEXT$", 
						      TEXT_SORTKEY, 
						      WILDCARD_ACCESS, 
						      CODE_QUADRANT, 
						      32, 
						      4, 
						      0, 
						      0);
                                }
				copyram_created = TRUE;
#endif /* 0 */
				/* done in crt0 */
				/* add_predef_sym(COPYRAM_SYMBOL, 
					      copyram_subsp, ST_DATA, 0, TRUE);
				}
					      */

			    break;

			case Q_NOCOMPRESS:
			    if (verbose & V_EMBED)
			        printf("nocompress qual on space\n");

			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    space_misc[cur_space].no_compression = TRUE;
			    break;

			case Q_RELOCATABLE:
			    if (verbose & V_EMBED)
			        printf("relocatable qual on space\n");

			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    space_misc[cur_space].relocatable = TRUE;
			    break;

			case Q_UNLOADABLE:
			    if (verbose & V_EMBED)
			        printf("unloadable qual on space\n");

			    if (cur_space == NOT_SEEN)
		                user_error(NAME_NOT_SEEN_BEF_QUAL, 
				    cur_cmd->string, qual_strings[qual], fn, 0);
			    /* This will force assign_virt_addr to */
			    /* NOT reset the origin to zero when this */
			    /* unloadable space is seen */
			    space_misc[cur_space].embed_unloadable = TRUE;
			    space_array[cur_space].is_loadable = FALSE;
			    break;

                        default:
			    if (qual || name_seen)
		                warning(UNRECOGNIZED_QUAL, cur_cmd->string, 
						     temp_buf, fn, 0);
			    else {
			        /* must be a subspace name */
			        /* cur_space_name is already set */
			        strcpy(cur_space_name, temp_buf);

			        is_private = FALSE;
			        is_text = FALSE;
                                if (strcmp(cur_space_name, "$PRIVATE$") == 0) {
				    is_private = TRUE;
			            cur_space = build_space(cur_space_name, 
							    PRIVATE_SORTKEY);
				} else {
				    if (strcmp(cur_space_name, "$TEXT$") == 0) 
				        is_text = TRUE;
			            cur_space = build_space(cur_space_name, 
							    TEXT_SORTKEY);
 			        }

				/* set start virt address but do NOT set */
				/* start_addr_specified flag. This allows */
				/* the new space to sort after any previously */
				/* specified space. The start address will */
				/* NOT be used in assign_virt_addresses. */
				space_misc[cur_space].start_virt_addr = 
				                          last_start_addr_seen;

				/* Mark space as specified in the cmd file */
				space_misc[cur_space].space_specified = TRUE;

			        name_seen = TRUE;
				}
			    break;
			}
		    }

                /* If $PRIVATE specified with no at qualifier data_mmap_addr */
		/* will NOT use the DEFAULT value */
		if (!at_seen) {
                    
		    if (cur_space != NOT_SEEN && 
                        !space_misc[cur_space].copyram_needed) {
 		        space_misc[cur_space].start_addr_specified = FALSE;
		    }
		    if (is_private) {
		        data_mmap_addr = UNSET_OFFSET;
		    } else if (is_text) {
		        code_mmap_addr = UNSET_OFFSET;
		    }
		}

		if (!name_seen)
		    user_error(NAME_NOT_SEEN, 
		               cur_cmd->string, 
			       fn, 
			       0);

	        break;

            default:              
		/* we did not recognize a case we should have */
		internal_error(CASE_MISSING, 0);
	        break;

            } /* end switch */

        } else {  /* end if cmd null */
	    /* This was not a valid keyword treat as a command line */
	    /* Make sure the word already read is used*/
	    get_cmd_line_args(f, buf);   
	    process_cmd_line_args();
	}
    }
    fclose(f);
}

/* match_command takes a word and tries to match it with the valid commands */
/* if the command is found it is returned, else NULL is returned. */
cmd_type_ptr match_command(word)
char *word;
{
    cmd_type_ptr p = commands;
    char buf[MAX_WORD_SIZE];
    char *c,*d;

    for (c=word, d=buf; *c != '\0'; c++, d++) 
	*d = _tolower(*c);

    *d = '\0';

    for ( ; p->string != NULL ; p++)
        if (strcmp(buf, p->string) == 0 ) {
            return(p);
	}
    return(C_NOMATCH);  /* no match was found */
}

int match_qualifiers(word, next_possible_qualifier)
char *word;
int *next_possible_qualifier;     /* ptr to null terminated list of ints */
{
    int this_qual;
    char buf[MAX_WORD_SIZE];
    char *c,*d;

    for (c=word, d=buf; *c != '\0'; c++, d++) 
	*d = _tolower(*c);

    *d = '\0';

    /* While there are still more qualifiers for this command */
    while (this_qual = *next_possible_qualifier) {
        if (strcmp(qual_strings[this_qual], buf) == 0 ) {
	    return(this_qual);
        }
        next_possible_qualifier++;   /* move to next qual in list */
    }

    return(Q_NONE);  /* no match was found */
}


int get_num(f, var, fmt)
FILE *f;
int *var;
char *fmt;
{
    char nextc[2];
    char fmtbuf[80];
    char *cp;

    if ((cp = get_next_word(f)) == NULL)
	return(0);

    if (cp[0] == '0' && (cp[1] == 'x' || cp[1] == 'X')) {
        strcpy(fmtbuf, "%x");
        cp += 2;
    } else if (cp[0] == '0') {
        strcpy(fmtbuf, "%o");
    } else
        strcpy(fmtbuf, fmt);

    strcat(fmtbuf, "%1s");
    if (sscanf(cp, fmtbuf, var, nextc) != 1)
	return(0);
    return(1);  /* non-NULL means OK read from string */
}

char *get_next_word(f)
FILE *f;
{
    char buf[1024];
    char *buf2;

    if (fgetword(f, buf, '\n') == FALSE) {
	return(NULL);
    } else {
        buf2 = (char *)emalloc((strlen(buf)+1)*sizeof(char));
	strcpy(buf2, buf);
        return(buf2);
    }
}


get_cmd_line_args(f, read_in)
FILE *f;
char *read_in;    /* if non null this is string already read in */
{
	/* Get args till end of file */
#if 0
	fil_get_args(f, EOF); 
void fil_get_args(f, end_condition)
FILE *f;
#endif /* 0 */
    int end_condition;       /* EOF  for -F, '\n' for -k */

    char buf[1024];      /* the argument buffer */


    end_condition = '\n';    /* for now */

    /* Any existing nargv will be gone */
    argv = (char **)emalloc(nargv_size * sizeof(char *));
    argv[0] = NULL;
    argc = 0;

    if (read_in != NULL) {
	/* Put already option already read in into argv */
        argv[++argc] = (char *)emalloc((strlen(read_in) + 1) * sizeof(char));
        strcpy(argv[argc],read_in);
	}

    /* f is a pointer to a file opened before this routine was called */

    while (fgetword(f, buf, end_condition)) {

        if (++argc >= argv_size) {
	    argv_size += MAX_ARGC;
	    argv = (char **)erealloc((char *) argv,
			        argv_size * sizeof(char *));
        }
    
        /* copy arguments from buffer */
        argv[argc] = (char *)emalloc((strlen(buf) + 1) * sizeof(char));
        strcpy(argv[argc],buf);

    }

    argc++;
}

char *get_alias_name(orig_name, file_name, module_name)
char *orig_name;           /* ptr to original name of the subspace */
char **file_name;          /* ptr to char ptr */
char **module_name;          /* ptr to char ptr */
{
    register struct alias_table *list_entry;
    register int i;

    if (! GET_ALIAS_NAME_POSSIBLE)
        return(NULL);    

    list_entry = alias_list;
    i = num_aliases;

    for ( ; (i > 0) ; list_entry++, i-- ) {
	if (list_entry->index == ALIAS_NAME)  /* this entry is the alias name */
	    continue;
	if (strcmp(orig_name, list_entry->name) == 0) {
	    if (list_entry->file_name != NULL) {

		/* must match file name too */
	        if (strcmp(*file_name, list_entry->file_name) == 0) {

		    /* and module name if there is one */
		    if ((list_entry->module_name == NULL)  ||
	               (strcmp(*module_name, list_entry->module_name) == 0)) {
		       *file_name = alias_list[list_entry->index].file_name;
		       *module_name = alias_list[list_entry->index].module_name;
	               return(alias_list[list_entry->index].name);
		    }

		}
	    } else
	        /* if file_name is null in alias list then it matches*/
	        return(alias_list[list_entry->index].name);
	}
    }

    /* No match was found */
    return(NULL);    
}

int filename_match(index1, index2)
int index1, index2;
{
    register struct subsp_misc_record *misc1;
    register struct subsp_misc_record *misc2;
    register Boolean either_is_null, both_are_null;
    char *module1, *module2;

    /* Already know one of these is qualified with a file or module name */
    /* Try and determind if the file and module names match as quickly as */
    /* possible. */

    misc1 = &Subsp_Misc(index1);
    misc2 = &Subsp_Misc(index2);

    /* if pointers are the same we have a quick file name match */
    /* otherwise a strcmp is necessary */
    if ((misc1->f_name == misc2->f_name) || 
        (strcmp(misc1->f_name, misc2->f_name) == 0)) {

        /* and module name if there is one */
	/* Within a cmd file it does not matter if either is null */
	/* The only time they match within a cmd file is if they are */
	/* the same string or they are both NULL! */

	module1 = misc1->module_name;
	module2 = misc2->module_name;

	either_is_null = (module1 == NULL || module2 == NULL);
	both_are_null  = (module1 == NULL && module2 == NULL);

        if ((module_match_when_null && either_is_null)  ||
	       both_are_null ||
               (!either_is_null && strcmp (module1, module2) == 0)) {
           return(TRUE);
        }
    }

    /* No match was found */
    return(FALSE);    
}


#define NUM_BITS (1 << 31)

check_align(max_bits, align, fn)
int max_bits;            /* number of bits this align must fit into */
unsigned int align;
char *fn;         /* filename (use for err message */
{
    int num_bits_on = 0;
    unsigned int bit;         /* only 16 bits are valid for an align */

    /* zero alignment is a no no */
    if (align == 0)
        user_error(ALIGN_WAS_ZERO, fn, 0);

    /* check to make sure the align is a power of two */
    for ( bit=1 ; (bit > 0 && num_bits_on <= 1 && max_bits > 0) 
						 ; bit = bit << 1, max_bits--)
	num_bits_on += ((align & bit) > 0);    /* count the on bits */
	
    if (num_bits_on != 1) {
	if (max_bits == 0)
            user_error(ALIGN_TOO_BIG, fn, 0);
	else 
            user_error(INVALID_ALIGN, fn, 0);
    }
    
    /* if only 1 bit must be a power of two */ 
}

char *split_name(c, file_name, module_name)
char *c;                /* full specification */
char **file_name;       /* put file part here */
char **module_name;     /* put module part here */
{
    char *subsp_name, *cptr;
    char *temp;             /* use as temp */
    extern char *strtok();

    /* Split name into parts, ex:  file(module_name):subspace_name */

    /* file_name is null if no : is found */
    if (MB_CUR_MAX == 1) {
	*file_name = strtok(c,":");             /* 1st call gives file_name */
	subsp_name = strtok(0,"");              /* return rest of string */
    } else {
	*file_name = mb_strtok(c,":");           /* 1st call gives file_name */
	subsp_name = mb_strtok(0,"");            /* return rest of string */
    }

    /* Now look for module name within parenthesis */
    if (subsp_name) {
	if (**file_name == '(') {
	    /* '(' is first - strtok will return the rest of the string! */
	    /* so instead we use the rest of the string */
	    *module_name = ((MB_CUR_MAX ==1) 
			        ? strtok((*file_name + 1), ")")
			        : mb_strtok((*file_name + 1), ")"));
	    *file_name = NULL;
        } else {
	    if (MB_CUR_MAX == 1) {
		*file_name = strtok(*file_name, "(");
		/* module_name gets null if no module_name */
		*module_name = strtok(0, ")");   
	    } else {
		*file_name = mb_strtok(*file_name, "(");
		/* module_name gets null if no module_name */
		*module_name = mb_strtok(0, ")");   
	    }
	}
    } else {
	/* file_name got the subspace name! Put it back the right way. */
	subsp_name = *file_name;
	*file_name = NULL;
	*module_name = NULL;     /* there was no module name */
    }
	    
    return(subsp_name);                   /* return rest of string */
}

int qualify_subsp(subsp_index, file_name, module_name)
int subsp_index;
char *file_name;
char *module_name;
{
    struct subsp_misc_record *misc;

    misc = &Subsp_Misc(subsp_index);

    /* this subsp name is qualified which means it has a file or module */
    /* name that must match along with the subspace name before the */
    /* subspaces can be considered the same */
    misc->is_qualified = TRUE;

    /* otherwise this was renamed earlier */
    if (misc->input_file_name == NULL)   
        misc->input_file_name = misc->file_name;
    misc->file_name = file_name;
    misc->f_name = file_name;
    misc->module_name = module_name;
}

unsigned int next_unreserved_address(start_addr)
unsigned int start_addr;
{
    struct reserve_table *r;
    int entry;
    
    /* move through all of reserve list checking for hits */
    for (entry = 0; entry < num_reserve_entries; entry++) {
	r = &reserve_list[entry];
	if (start_addr >= r->start_addr && start_addr <= r->end_addr) {
            start_addr = r->end_addr;
	}
    }
    return(start_addr);
}

Boolean check_reserve_list(str_ind, start_addr, end_addr)
int str_ind;
unsigned int start_addr;
unsigned int end_addr;
{
    struct reserve_table *r;
    int entry;
    char str[512], str2[512];
    
    /* move through reserve list checking for hits */
    for (entry = 0; entry < num_reserve_entries; entry++) {
	r = &reserve_list[entry];
	if ((start_addr >= r->start_addr && start_addr <= r->end_addr) ||
	   (end_addr >= r->start_addr && end_addr <= r->end_addr) ||
	   (start_addr <= r->start_addr && end_addr >= r->end_addr)) {
	    sprintf(str, 
		    "Subspace %s(0x%x)", 
		    string_ptr_linear(str_ind, &space_strings), 
		    start_addr);
	    sprintf(str2, "0x%x", r->start_addr);
            warning(RESERVE_REGION_WARN, str, str2);
	    continue;
	}
    }
} /* end check_reserve_list */
