/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *  
 *  Linkmap-related operations
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Header: /home/cvs/cvsroot/linker/ld_linkmap.c,v 1.1.1.1 1999/10/18 19:53:02 pschwan Exp $
 */

#include <stdlib.h>
#include <limits.h>
#include <assert.h>

#include "filehdr.h"
#include "aouthdr.h"
#include "spacehdr.h"
#include "scnhdr.h"
#include "syms.h"
#include "std.h"
#include "ldlimits.h"
#include "ldnls.h"
#include "spaces.h"
#include "subspaces.h"
#include "symbols.h"
#include "libraries.h"
#include "errors.h"
#include "util.h"
#include "linker.h"
#include "output.h"
#include "driver.h"
#include "ld_strings.h"

#include "linkmap.h"
#include "ld_linkmap.h"
#include "ld_linkmap_int.h"

/* options					*/
static Boolean 	options_tools		= FALSE;
static Boolean 	options_noobjdebug	= FALSE;
static Boolean 	options_nocopyobjdebug	= FALSE;
static Boolean 	options_copyobjdebug	= FALSE;
static Boolean 	options_nolinkmap	= FALSE;
static Boolean 	options_linkmap		= FALSE;
static Boolean 	options_strip_dash_s	= FALSE;
static Boolean 	options_strip_dash_G	= FALSE;
static Boolean 	copied_debug		= FALSE;

/* tables and such 				*/
static lm_st	st	= {NULL, 0, 0, NULL};
static lm_st	st_in	= {NULL, 0, 0, NULL};
static lm_bst	bst	= {0, NULL};
static lm_bsst	bsst	= {0, NULL};
static lm_oft	oft	= {0, NULL};

/* subspace indices 				*/
static unsigned int 	subsp_lm		= BAD_SUBSP;
static unsigned int 	subsp_lm_str		= BAD_SUBSP;
static unsigned int 	subsp_lm_bss	 	= BAD_SUBSP;
static unsigned int	subsp_lm_file 		= BAD_SUBSP;
static unsigned int 	out_subsp_lm		= BAD_SUBSP;
static unsigned int 	out_subsp_lm_str	= BAD_SUBSP;
static unsigned int 	out_subsp_lm_bss 	= BAD_SUBSP;
static unsigned int	out_subsp_lm_file 	= BAD_SUBSP;

/* debug/linkmap modes				*/
static enum lm_mode 	objdebug_mode		= DEFAULT;
static enum lm_mode 	linkmap_mode		= DEFAULT;

/* buffer management				*/
static unsigned int 	read_subspace		= BAD_SUBSP;
static unsigned int 	write_subspace		= BAD_SUBSP;
static unsigned int	read_error		= FALSE;
static unsigned int	write_error		= FALSE;
static char 		read_data     [LM_READBUF_SIZE];
static char 		read_data_map [LM_READBUF_SIZE];
static char 		write_data    [LM_WRITEBUF_SIZE];
static unsigned int 	cur_subsp_offset	= 0;
static unsigned int 	cur_subsp_offset_map	= 0;

/* misc structures				*/
static unsigned int 	link_dir_offset   	= 0;
static lm_stn           *str_node               = NULL;
static lm_isc           *input_contribs         = NULL;

/* not static due to function -> macro speed-up conversion */
unsigned int	        strip_debug_sp_idx 	= BAD_SPACE;
unsigned int	        *bss_remap		= NULL;
unsigned int 	        cur_objfile_index   	= 0;

/* used to initialize subspace data */
char 	                lm_subsp_data4[4]	= {0, 0, 0, 0};

/*
 * Internal functionality ==========================================
 */

#ifdef CNX_TOOLS

/*
 * create new string table node.
 *
 * - return a pointer to the node
 * = nodes are allocated in LM_STRNODE_ALLOC chunks
 */
static
struct linkmap_str_node *lmi_get_str_node() 
{
  static unsigned int index = 0;


  if ((index == LM_STRNODE_ALLOC) || !str_node) {
    str_node = (lm_stn *)emalloc(LM_STRNODE_ALLOC*sizeof(lm_stn)); 
    index = 0;
  } 
  
  return (&str_node[index++]);
}

/*
 * initialize output string table
 */
static 
void lmi_init_string_table(void) 
{
  int i;

  /* first time initialization */
  for (i=0; i<LM_HASHTBL_SIZE; i++) {
    st.hash_table[i] = NULL;
  }	
  st.data = (char *)emalloc(MIN_STRBLK_ALLOCATION);
  st.data_size = MIN_STRBLK_ALLOCATION;
  /* 0th index in the string table is NULL */
  st.data[0] = '\0';
  st.cur_offset = 1;
}

/*
 * add string to the output string table
 *
 * - return index into output string table
 * = internal string pointer is realloced in 
 * MIN_STRBLK_ALLOCATION chunks.
 */
static 
unsigned int lmi_add_string(char *str) 
{
  int i=0, len=0;
  unsigned int hashval=0;
  struct linkmap_str_node *node=NULL;
  
  if (!str || !(*str)) {
    return (0);
  }
  
  hashval = hash_string(str);
  for (node=st.hash_table[hashval%LM_HASHTBL_SIZE]; 
       node; node=node->next) {
    if ((node->hashval == hashval) && 
	(strcmp(st.data+node->offset, str) == 0)) {
      
      /* found a node, return offset */
#ifdef DEBUG
      if (verbose & V_DOOM) {
	fprintf(stderr, 
		"lmi_add_string(): looked up %s at offset %d\n", 
		str, 
		node->offset);
      }
#endif /* DEBUG */

      return (node->offset);
    }
  } 
    
  /* add a hash node */
  node = lmi_get_str_node();
  node->offset = st.cur_offset;
  node->hashval = hashval;
  node->next = st.hash_table[hashval%LM_HASHTBL_SIZE];
  st.hash_table[hashval%LM_HASHTBL_SIZE] = node;
  
  /* add a string */
  len = strlen(str)+1;
  if ((st.cur_offset+len) > st.data_size) {
    st.data = (char *)erealloc(st.data, 
			       st.data_size+MIN_STRBLK_ALLOCATION);
    st.data_size += MIN_STRBLK_ALLOCATION;
  }
  strcpy(st.data+st.cur_offset, str);
  st.cur_offset += len;
  
#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_add_string(): added %s at offset %d\n", 
	    str, 
	    node->offset);
  }
#endif /* DEBUG */
  
  /* return offset */
  return (node->offset);
}

/*
 * allocate space for a new entry in the object file table 
 *
 * - return an index into the object file table 
 * = realloc the array in LM_OBJFTBL_ALLOC chunks
 */
static 
unsigned int lmi_new_file_entry(void)
{
  static unsigned int num_entries = 0;

  if (num_entries == oft.num_entries) {
    num_entries += LM_OBJFTBL_ALLOC;
    oft.entries = (lm_ofte *)erealloc((char *)oft.entries,
				      sizeof(lm_ofte)*num_entries);
  }
  
#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_new_file_entry(): allocated entry at index %d\n", 
	    oft.num_entries);
  }
#endif /* DEBUG */
  
  return (oft.num_entries++);
}

/* 
 * create object file table entry from arguments.  add
 * the strings to the output string table.
 *
 * - return an index into the object file table
 */
static
unsigned int lmi_create_file_entry(char *dir_name, char *file_name)
{
  static unsigned int out_dir_name_offset = 0;
  static char in_dir_name[PATH_MAX] = "";
  LM_file_entry_info *finfo 	= NULL;
  unsigned int index		= 0;

  index = lmi_new_file_entry();
  finfo = &(oft.entries[index].file_entry);
  
  /* optimize for libraries by comparing pointers directly */
  finfo->link_dir 	= link_dir_offset;
  if (strcmp(in_dir_name, dir_name) == 0) {
    finfo->dir_name     = out_dir_name_offset;
  } else {
    out_dir_name_offset = lmi_add_string(dir_name);
    finfo->dir_name 	= out_dir_name_offset;
    if (dir_name) strcpy(in_dir_name, dir_name);
  }
  finfo->file_name 	= lmi_add_string(file_name);

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_create_file_entry(): added %s(%s) at index %d\n", 
	    (dir_name)?dir_name:"NULL", 
	    (file_name)?file_name:"NULL", 
	    index);
  }
#endif /* DEBUG */
  
  return (index);
}

/*
 * process an entry obtained from the input linkmap.  
 * Remap and add input strings to the output string 
 * table.
 *
 * = speed up the routine by keeping track of the link
 * directory, so that no needless hashing will be done.
 */
static
void lmi_process_file_entry(LM_file_entry_info *entry) 
{
  static unsigned int in_link_dir_offset  = 0;
  static unsigned int out_link_dir_offset = 0;
  static unsigned int in_dir_name_offset  = 0;
  static unsigned int out_dir_name_offset = 0;
  unsigned int in_offset = 0;

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_process_file_entry(): begin entry %d/%d/%d\n",
	    entry->link_dir, 
	    entry->dir_name, 
	    entry->file_name);
  }
#endif /* DEBUG */

  /* optimize processing of the link and dir names */
  in_offset = entry->link_dir;
  if (in_link_dir_offset == in_offset) {
    entry->link_dir 	= out_link_dir_offset;
  } else {
    in_link_dir_offset  = in_offset;
    out_link_dir_offset = lmi_add_string(lmi_string_at(in_offset));
    entry->link_dir     = out_link_dir_offset;
  }

  in_offset = entry->dir_name;
  if (in_dir_name_offset == in_offset) {
    entry->dir_name 	= out_dir_name_offset;
  } else {
    in_dir_name_offset  = in_offset;
    out_dir_name_offset = lmi_add_string(lmi_string_at(in_offset));
    entry->dir_name     = out_dir_name_offset;;
  }

  /* add the strings to the output string table and remap indices */
  entry->file_name 	= lmi_add_string(lmi_string_at(entry->file_name));

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_process_file_entry(): end entry %d/%d/%d\n",
	    entry->link_dir, 
	    entry->dir_name, 
	    entry->file_name);
  }
#endif /* DEBUG */

}

/*
 * sort bss symbol table based on the output file index
 * of the object file from which they originated.  
 */
static 
int lmi_bss_symbol_sort(const void *p1, const void *p2)
{
  lm_bste *e1 = (lm_bste *)p1;
  lm_bste *e2 = (lm_bste *)p2;
  
  return (oft.entries[Sym_Misc(e1->sym_index).objfile_index].file_index<
	  oft.entries[Sym_Misc(e2->sym_index).objfile_index].file_index)?(-1):(1);
}

/*
 * sort the (temporary) bss symbol address table based on the 
 * increasing address of the symbol.  
 */
static 
int lmi_bss_symaddr_sort(const void *p1, const void *p2)
{
  lm_bsae *e1 = (lm_bsae *)p1;
  lm_bsae *e2 = (lm_bsae *)p2;

  if (e1->subsp_index < e2->subsp_index) {
     return -1;
  } else if ((e1->subsp_index == e2->subsp_index) && 
	     (e1->out_addr < e2->out_addr)) {
     return -1;
  } else {
     return 1;
  }
}

/*
 * sort bss subspace table based on the increasing address
 * of the subspace.  The table is needed since the sizes 
 * of BSS subspaces tend to change during the link.
 */
static
int lmi_bss_subsp_sort(const void *p1, const void *p2)
{
  lm_bsste *e1 = (lm_bsste *)p1;
  lm_bsste *e2 = (lm_bsste *)p2;

  if (e1->subsp_index < e2->subsp_index) {
     return -1;
  } else if ((e1->subsp_index == e2->subsp_index) && 
	     (e1->out_addr < e2->out_addr)) {
     return -1;
  } else {
     return 1;
  }
}

/*
 * create new input subspace contribution.
 *
 * - return a pointer to the contrib
 * = nodes are allocated in LM_ISCONTR_ALLOC chunks
 */
static
struct input_subsp_contrib *lmi_get_input_contrib() 
{
  static unsigned int index = 0;


  if ((index == LM_ISCONTR_ALLOC) || !input_contribs) {
    input_contribs = (lm_isc *)emalloc(LM_ISCONTR_ALLOC*sizeof(lm_isc)); 
    index = 0;
  } 
  
  return (&input_contribs[index++]);
}

/* 
 * callback function for the reading of the linkmap data.
 * Use "read_subspace" to decide which subspace should be
 * read to obtain the information.
 *
 * = read errors are not handled locally, see lm_linkmap_read()
 */
static
char *lmi_buffer_reader(unsigned int *size)
{
  struct subspace_dictionary_record	*dict = NULL;
  unsigned int 				bufsize = 0;
  char                                  *ptr = NULL;
  unsigned int 				*offset = NULL;
  
  /* 
   * because we must initialize linkmap before reading any other 
   * subspaces (such as _file), we use 2 different buffers, one 
   * for linkmap, and another for _file and _bss
   */
  ptr = (read_subspace == subsp_lm)?&(read_data_map[0]):(&read_data[0]);
  offset = (read_subspace == subsp_lm)?&cur_subsp_offset_map:&cur_subsp_offset;

  dict = &Subsp_Dict(read_subspace);
  bufsize = min(LM_READBUF_SIZE, dict->subspace_length-(*offset));
  if (!dict->subspace_length || (bufsize == 0)) {
    *offset = 0;
    *size = 0;
    return (NULL);
  }
  ffetch(cur_fildes,
	 dict->file_loc_init_value + cur_ofs + *offset,
	 ptr,
	 sizeof(char),
	 bufsize);
  *size = bufsize;
  *offset += bufsize;
  
#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_buffer_reader(): read %d bytes from input subspace %d\n",
	    bufsize, 
	    read_subspace);
  }
#endif /* DEBUG */

  return (ptr);
}

/* 
 * callback function for the writing of the linkmap data.
 * Use "write_subspace" to decide which subspace should be
 * written to obtain the information.
 *
 * = write errors are not handled locally, see lm_linkmap_create()
 */
static
char *lmi_buffer_writer(unsigned int *size)
{
  struct subspace_dictionary_record	*dict		= NULL;
  struct subsp_misc_record		*misc		= NULL;
  static int				last_subsp      = BAD_SUBSP;
  unsigned int 				length          = 0;

  /* 
   * if subspace is getting written for the first time
   * just return the empty buffer to the support library
   */
  if (write_subspace != last_subsp) {
    last_subsp = write_subspace;
    *size = LM_WRITEBUF_SIZE;
    return (write_data);
  }

  dict = &Subsp_Dict(write_subspace);
  misc = &Subsp_Misc(write_subspace);
  length = dict->subspace_length;
  fdump(out_som_fd, 
	misc->exec_file_offset+length, 
	write_data, 
	1, 
	LM_WRITEBUF_SIZE);
  lmi_set_subspace_length(write_subspace, length+LM_WRITEBUF_SIZE);
  *size = LM_WRITEBUF_SIZE;
  
#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_buffer_writer(): wrote %d bytes to output subspace %d\n",
	    LM_WRITEBUF_SIZE, 
	    write_subspace);
  }
#endif /* DEBUG */

  return (write_data);
}

/*
 * flush the remains of the buffer to the file.  Make
 * sure to pad the contents to 4byte alignment.  Use
 * "write_subspace" to figure out which subspace the 
 * data belongs to.
 *
 * = write errors are not handled locally
 */
static 
void lmi_buffer_writer_finish(unsigned int size)
{
  struct subspace_dictionary_record	*dict		= NULL;
  struct subsp_misc_record		*misc		= NULL;
  unsigned int 				pad_bytes	= 0;
  unsigned int 				buf_bytes	= 0;
  unsigned int 				length	        = 0;
  
  dict = &Subsp_Dict(write_subspace);
  misc = &Subsp_Misc(write_subspace);
  length = dict->subspace_length;
  buf_bytes = LM_ALIGN_4(length+size);
  pad_bytes = buf_bytes-length+size;
  
  if (pad_bytes && LM_pad_op_null(pad_bytes, write_data+size) != LM_NO_ERROR) {
    write_error = TRUE;
  }
  fdump(out_som_fd, 
	misc->exec_file_offset+length, 
	write_data, 
	1, 
	buf_bytes);
  
  lmi_set_subspace_length(write_subspace, length+buf_bytes);

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_buffer_writer_finish(): "
	    "wrote %d total (%d pad) to output subspace %d\n",
	    buf_bytes, 
	    pad_bytes, 
	    write_subspace);
  }
#endif /* DEBUG */

}

/* 
 * write the contents of the output string table
 *
 * - return the size of the string table written.
 * = write errors are not handled locally
 */
static
unsigned int lmi_create_string_table(unsigned int offset)
{
  struct subspace_dictionary_record	*dict		= NULL;
  struct subsp_misc_record		*misc		= NULL;
  int					size		= 0;

  dict 	= &Subsp_Dict(out_subsp_lm_str);
  misc 	= &Subsp_Misc(out_subsp_lm_str);
  misc->exec_file_offset += offset;

  size 	= LM_ALIGN_4(st.cur_offset); 
  fdump(out_som_fd, misc->exec_file_offset, st.data, 1, size);
  lmi_set_subspace_length(out_subsp_lm_str, size);

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lmi_create_string_table: "
	    "wrote %d (unaligned %d) bytes at %d offset\n",
	    size, 
	    st.cur_offset, 
	    misc->exec_file_offset);
  }
#endif /* DEBUG */

  return (dict->subspace_length);
}

/* 
 * write the contents of the output object file table
 *
 * - return the size of the object file table written
 * = write errors are not handled locally
 */
static
unsigned int lmi_create_objfile_table(unsigned int offset)
{
  struct subspace_dictionary_record	*dict		= NULL;
  struct subsp_misc_record		*misc		= NULL;
  unsigned int				file_index	= 0;
  int					size		= 0;
  int					i		= 0;

  dict 	= &Subsp_Dict(out_subsp_lm_file);
  misc 	= &Subsp_Misc(out_subsp_lm_file);
  misc->exec_file_offset += offset;
  
  write_subspace = out_subsp_lm_file;
  if (LM_begin_write_linkmap_file(lmi_buffer_writer) != LM_NO_ERROR) {
    /* if we can't generate info, do not generate subspace */
    lmi_strip_subspace(out_subsp_lm_file);
  } else {
    file_index = 0;
    for (i=0; i<oft.num_entries; i++) {
      /* 
       * output object files seen on link line.  if compsize is 0,
       * object file has linkmap in it and should not have an entry.
       */
      if ((oft.entries[i].file_index == (-1)) &&
	  (oft.entries[i].compsize)) {
	oft.entries[i].file_index = file_index++;

#ifdef DEBUG
	if (verbose & V_DOOM) {
	  fprintf(stderr, 
		  "lmi_create_objfile_table: "
		  "explicit entry %d with: %d/%d/%d/%d/%d\n",
		  oft.entries[i].file_index,
		  oft.entries[i].file_entry.link_dir,
		  oft.entries[i].file_entry.dir_name,
		  oft.entries[i].file_entry.file_name,
		  oft.entries[i].file_entry.is_archive,
		  oft.entries[i].file_entry.is_basename);
	}
#endif /* DEBUG */

	if (LM_put_file_entry(&(oft.entries[i].file_entry)) != LM_NO_ERROR) {
	  write_error = TRUE;
	};
      }
    }

    /* if don't have any embedded entries, no need to write them */
    if (file_index < oft.num_entries) {
      for (i=0; i<oft.num_entries; i++) {
	/* output object files seen in input linkmap */
	if (oft.entries[i].file_index == (-2)) {
	  oft.entries[i].file_index = file_index++;
	  
#ifdef DEBUG
	  if (verbose & V_DOOM) {
	    fprintf(stderr, 
		    "lmi_create_objfile_table: "
		    "implicit entry %d with: %d/%d/%d/%d/%d\n",
		    oft.entries[i].file_index,
		    oft.entries[i].file_entry.link_dir,
		    oft.entries[i].file_entry.dir_name,
		    oft.entries[i].file_entry.file_name,
		    oft.entries[i].file_entry.is_archive,
		    oft.entries[i].file_entry.is_basename);
	  }
#endif /* DEBUG */
	  
	  if (LM_put_file_entry(&(oft.entries[i].file_entry)) != LM_NO_ERROR) {
	    write_error = TRUE;
	  };
	}
      }
    }

    if (LM_end_write_linkmap_file(&size) != LM_NO_ERROR) {
      write_error = TRUE;
    } else {
      lmi_buffer_writer_finish(size);
    }
  }
  return (dict->subspace_length);
}

/* 
 * write  the contents of the linkmap subspace
 *
 * - return the size of the linkmap written.
 * = write errors are not handled locally
 */
static
unsigned int lmi_create_linkmap(unsigned int offset)
{
  /* subspace dictionary/misc(s) and their indices */
  struct subspace_dictionary_record	*dict, *subdict	= NULL;
  struct subsp_misc_record		*misc, *submisc = NULL;
  unsigned int 				old_index	= 0;
  unsigned int 				new_index	= 0;

  /* various mappings that are used */
  LM_mapping_info			mapinfo         = {LM_INVALID_IDX,
                                                           LM_INPUT_SECTION,
                                                           LM_INVALID_IDX,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  LM_mapping_info			linkinfo        = {LM_INVALID_IDX,
                                                           LM_LINKER_SECTION,
                                                           LM_INVALID_IDX,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  LM_mapping_info			bsyminfo        = {LM_INVALID_IDX,
                                                           LM_LINKER_BSS,
                                                           LM_INVALID_IDX,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  LM_mapping_info			datainfo        = {LM_INVALID_IDX,
                                                           LM_DATA_STRUCTURE,
                                                           DATA_STRUCT_COMPILATION_UNIT_DICT,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  LM_mapping_info			undefinfo       = {LM_INVALID_IDX,
                                                           LM_UNDEF_REGION,
                                                           LM_INVALID_IDX,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  /* control which one of the above gets selected */
  LM_mapping_info			*mapinfo_ptr	= NULL;
  Boolean				do_mapinfoptr	= TRUE;

  /* need to fill in the alignment holes due to BSS subspaces */
  LM_mapping_info			holeinfo        = {LM_INVALID_IDX,
                                                           LM_UNDEF_REGION,
                                                           LM_INVALID_IDX,
                                                           LM_INVALID_IDX,
                                                           0, 0};
  Boolean				do_holeinfo	= FALSE;


  /* bss symbol handling */
  struct bss_symaddr_entry		*symaddr	= NULL;
  struct bss_symaddr_entry		*bsym_entry	= NULL;
  struct symbol_misc_record	 	*bsymmisc	= NULL;
  unsigned int 				bsym_index	= 0;

  /* bss subspace handling */
  struct bss_subspace_table_entry	*bsub_entry	= NULL;
  struct subsp_misc_record		*bsubmisc	= NULL;
  unsigned int 				bsub_index	= 0;
  unsigned int 				in_subsp_index	= 0;

  /* linkmap entry boundaries as they are written out */
  unsigned int				cur_out_subsp	= BAD_SUBSP;
  unsigned int				cur_start_addr	= (-1);
  unsigned int				prev_end_addr	= (-1);
  unsigned int				new_subspace_start = (-1);
  unsigned int				subspace_length = 0;
  int					subspace_objfile_index = (-1);

  /* control over next (non-bss)contribution or subspace */
  struct input_subsp_contrib 		*cur_contrib	= NULL;
  Boolean				change_index	= FALSE;

  /* misc */
  int 					objfile_index   = 0;
  int 				        size		= 0;
  int					i	        = 0;


  dict 	= &Subsp_Dict(out_subsp_lm);
  misc 	= &Subsp_Misc(out_subsp_lm);
  misc->exec_file_offset += offset;

  /* remap bss symbols and sort according to symbol address */
  if (!relocatable && bst.num_entries) {
    symaddr = (lm_bsae *)emalloc(sizeof(lm_bsae)*bst.num_entries);
    for (i=0; i<bst.num_entries; i++) {
      symaddr[i].subsp_index = 
	Subsp_Misc(Sym_Subsp(bst.entries[i].sym_index)).r_c_x;
      symaddr[i].out_addr      = symbol_value(bst.entries[i].sym_index);
      symaddr[i].bst_index     = i;
    }
    qsort((void *)symaddr, 
	  (size_t)bst.num_entries,
	  sizeof(lm_bsae),
	  lmi_bss_symaddr_sort);
  }

  /* remap bss subspaces and sort according to output address */
  if (bsst.num_entries) {
    for (i=0; i<bsst.num_entries; i++) {
      in_subsp_index = Subsp_Misc(Subsp_Misc(bsst.entries[i].in_subsp_index).r_o_x).r_n_x;
      /*
      ** set out_addr for BSS subspaces only; 
      ** out_addr of TBSS subspace is already set at lm_bss_subspace_add().
      */
      submisc = &Subsp_Misc(in_subsp_index);
      if (bsst.entries[i].out_addr == -1) {
         bsst.entries[i].out_addr = submisc->new_subspace_start;
      } else {
	 /* For simplifying the support for -r link of the TBSS subspaces,
	  * we assume that the linker does not merge TBSS subspaces in
	  * a -r link. i.e., a TBSS subspace has at most one linkmap
	  * contrib record.  
	  */
	 assert(submisc->contrib == NULL || submisc->contrib->next == NULL);
	 assert(submisc->contrib == NULL || 
		submisc->contrib->map.size == bsst.entries[i].length);
      }
      bsst.entries[i].subsp_index = Subsp_Misc(in_subsp_index).r_c_x; 
    }
    qsort((void *)bsst.entries,
	  (size_t)bsst.num_entries,
	  sizeof(lm_bsste),
	  lmi_bss_subsp_sort);
  }
  
  /* generate linkmap information */
  write_subspace = out_subsp_lm;
  if (LM_begin_write_linkmap(lmi_buffer_writer) != LM_NO_ERROR) {
    /* if we can't generate info, do not generate subspace */
    lmi_strip_subspace(out_subsp_lm);
  } else {
    /* first, generate data structure opcodes */
    for (i=0; i<oft.num_entries; i++) {
      if (!oft.entries[i].compsize) {
        continue;
      }

      datainfo.size       = oft.entries[i].compsize;
      datainfo.file_index = oft.entries[i].file_index;
      
#ifdef DEBUG
      if (verbose & V_DOOM) {
	fprintf(stderr, 
		"data_structs: writing %d structure "
		"for %d index with %d size\n",
		i, oft.entries[i].file_index, oft.entries[i].compsize);
      }
#endif /* DEBUG */
      
      if (LM_put_mapping(&datainfo) != LM_NO_ERROR) {
	write_error = TRUE;
      }
    }
    
    /* second, generate subspace information */
    bsub_index = 0; 
    bsym_index = 0;
    new_index = 0;
    change_index = TRUE;
    while (new_index < subsp_dict_size) {
      /* control next for-loop iteration */
      if (change_index) {
        old_index = Subsp_Misc(new_index).r_n_x;
        subdict   = &Subsp_Dict(old_index);
        submisc   = &Subsp_Misc(old_index);
	subspace_objfile_index = submisc->objfile_index;
        if (subdict->is_tspecific) {
	   new_subspace_start = 0;
	   subspace_length = tbss_size;
	   /* contributions of the input subspace are already recorded
	    * in bsst. */
           cur_contrib = NULL; 
        } else {
	   new_subspace_start = submisc->new_subspace_start;
	   subspace_length = subdict->subspace_length;
	   /* first contribution of the input subspace */
           cur_contrib = submisc->contrib; 
	}
	change_index = FALSE;
      }
      
      /* skip subspaces that will not be outputted */
      if ((strip_debug && !subdict->is_loadable) ||
	  (subdict->is_comdat && submisc->comdat_nulled) ||
	  (subdict->subspace_length == 0) ||
	  (submisc->first_common_def != (-1)) ||
	  submisc->is_doom_stripped ||
	  !submisc->is_linkmap_entry ||
	  submisc->delete_subspace) {
	change_index = TRUE;
	new_index++;
	continue;
      }
      
      /* 
       * track current output subspace, as well as address of the last
       * byte to which last entry had written (prev_end_addr).  If starting
       * address of an entry to be written (cur_start_addr) in the same 
       * output subspace is not where the last one left off, we have an 
       * alignment hole, which is filled using UNDEF_REGION.
       */
      if (cur_out_subsp != submisc->r_c_x) {
	cur_out_subsp = submisc->r_c_x;
	prev_end_addr = submisc->new_subspace_start;
      }


      /* consider each subspace type */
      do_mapinfoptr = TRUE;
      do_holeinfo = FALSE;
      if (submisc->is_undefined_region) {
	/* undefined region - skip (if any) contributions */
	mapinfo_ptr = &undefinfo;
        mapinfo_ptr->output_info = submisc->r_c_x;
        mapinfo_ptr->size = subspace_length;
	change_index = TRUE;
	cur_start_addr = new_subspace_start;
      } else if (submisc->is_linker_generated) {
	/* linker generated - there can not be any contribs */
	mapinfo_ptr = &linkinfo;
        mapinfo_ptr->output_info = submisc->r_c_x;
        mapinfo_ptr->size = subspace_length;
	change_index = TRUE;
	cur_start_addr = new_subspace_start;
      } else if (!submisc->is_bss_subspace) {
	/* handle regular input sections */
	if (submisc->contrib) {
	  if (!cur_contrib) {
	     do_mapinfoptr = FALSE;
	     change_index = TRUE;
	  } else {
	     mapinfo_ptr = &(cur_contrib->map);
	     objfile_index = subspace_objfile_index+mapinfo_ptr->file_index+1;
	     mapinfo_ptr->file_index = 
		(mapinfo_ptr->file_index != LM_INVALID_IDX)?
	            (oft.entries[objfile_index].file_index) :
		    LM_INVALID_IDX;
	     mapinfo_ptr->output_info= submisc->r_c_x;
	     cur_contrib = cur_contrib->next;
	  }
	} else {
	  mapinfo_ptr = &mapinfo;
	  mapinfo_ptr->file_index = oft.entries[subspace_objfile_index].file_index;
	  mapinfo_ptr->input_info = old_index-oft.entries[subspace_objfile_index].subsp_bias;
	  mapinfo_ptr->output_info = submisc->r_c_x;
	  mapinfo_ptr->size = subspace_length;
	  change_index = TRUE;
	}
	cur_start_addr = new_subspace_start+mapinfo_ptr->offset;
      } else {
	/* ahh, bss... in here we have a number of variables to consider: (a) we have a
	** bss subspace as recorded in the bss subspace table, (b) we have a bss symbol,
	** as recorded in the bss symbol table, (c) we have a bss subspace as derived 
	** from the input linkmap (and not recorded in the bss subspace table), and (d)
	** we have (possible, but not now), multiple input contributions within (c).
	**
	** general approach is to obtain the lowest possible address in the same output
	** subspace among the above, and then output the appropriate data structure, etc.
	** input contribs are tracked internally, via bcontrib, so that change_index
	** is always FALSE for BSS subspaces, until there are no more entries to the 
	** current output subspace (cur_out_subsp).
	*/

	/* bss symbol in same input/output subspace */
	if (!relocatable && 
	    (bsym_index < bst.num_entries) &&
	    (symaddr[bsym_index].subsp_index == cur_out_subsp) &&
	    (symaddr[bsym_index].out_addr >= new_subspace_start) &&
	    (symaddr[bsym_index].out_addr < new_subspace_start+subspace_length)) {
	  bsym_entry = &symaddr[bsym_index];
	} else {
	  bsym_entry = NULL;
	}
	
	/* bss subspace in bss subspace table in same input/output subspace */
	if ((bsub_index < bsst.num_entries) &&
	    (bsst.entries[bsub_index].subsp_index == cur_out_subsp) &&
	    (bsst.entries[bsub_index].out_addr >= new_subspace_start) &&
 	    (bsst.entries[bsub_index].out_addr < new_subspace_start+subspace_length)) {
	  bsub_entry = &bsst.entries[bsub_index];
	} else {
	  bsub_entry = NULL;
	}
	
        /* if all are NULL, move on to the next input subspace */
	change_index = FALSE;
	if (!cur_contrib && !bsub_entry && !bsym_entry) {
	  change_index = TRUE;
	  do_mapinfoptr = FALSE;
	} else {
	  if ((cur_contrib) &&
	      (!bsym_entry || (new_subspace_start+cur_contrib->map.offset<
			       bsym_entry->out_addr)) &&
	      (!bsub_entry || (new_subspace_start+cur_contrib->map.offset<
			       bsub_entry->out_addr))) {
	    /* pick cur_contrib */
	    cur_start_addr = new_subspace_start+cur_contrib->map.offset;
	    mapinfo_ptr = &(cur_contrib->map);
	    objfile_index = subspace_objfile_index+mapinfo_ptr->file_index+1;
	    mapinfo_ptr->file_index = 
	      (mapinfo_ptr->file_index != LM_INVALID_IDX)?
	      (oft.entries[objfile_index].file_index):LM_INVALID_IDX;
	    mapinfo_ptr->output_info= submisc->r_c_x;
	    cur_contrib = cur_contrib->next;
	  } else if ((bsub_entry) &&
		     (!bsym_entry || (bsub_entry->out_addr <= bsym_entry->out_addr)) &&
		     (!cur_contrib || (bsub_entry->out_addr <= new_subspace_start+
				       cur_contrib->map.offset))) {
	    /* pick bsub_contrib */
	    cur_start_addr = bsub_entry->out_addr;
	    in_subsp_index = Subsp_Misc(Subsp_Misc(bsub_entry->in_subsp_index).r_o_x).r_n_x;
            bsubmisc = &Subsp_Misc(in_subsp_index);
	    objfile_index = bsubmisc->objfile_index;
	    if (Subsp_Dict(in_subsp_index).is_tspecific && bsubmisc->contrib) {
	       /* If a TBSS subspace and has a linkmap contrib, i.e., TBSS
		* subspace is contributed from a -r link object file,
		* the object file contributed to this subspace is in the 
	        * the linkmap contrib.
		*/ 
	       objfile_index = bsubmisc->objfile_index +
				  bsubmisc->contrib->map.file_index+1;
	       mapinfo_ptr = &(bsubmisc->contrib->map);
	    } else {
	       mapinfo_ptr = &mapinfo;
	       mapinfo_ptr->input_info = 
	          (bsub_entry->in_subsp_index - 
	          oft.entries[objfile_index].subsp_bias);
	    }
	    mapinfo_ptr->file_index = oft.entries[objfile_index].file_index;
	    mapinfo_ptr->output_info = submisc->r_c_x;
	    mapinfo_ptr->size = bsub_entry->length;
	    bsub_index++;
	  } else {
	    /* pick bsym_entry */
	    cur_start_addr = bsym_entry->out_addr;
	    bsymmisc = &Sym_Misc(bst.entries[bsym_entry->bst_index].sym_index);
	    mapinfo_ptr = &bsyminfo;
	    mapinfo_ptr->file_index = 
	      oft.entries[bsymmisc->objfile_index].file_index;
	    mapinfo_ptr->output_info = submisc->r_c_x;
	    mapinfo_ptr->size = bst.entries[bsym_entry->bst_index].length;
	    bsym_index++;
	  }
	}
      }

      /* handle alignment holes */
      if (do_mapinfoptr) {
	if (prev_end_addr < cur_start_addr) {
	  do_holeinfo = TRUE;
	  holeinfo.output_info = mapinfo_ptr->output_info;
	  holeinfo.size = (cur_start_addr-prev_end_addr);
	}
	/* since we are outputting an entry, adjust ending address */
	prev_end_addr = cur_start_addr+mapinfo_ptr->size;
      } else {
	/*
	** this is usually set when we finished processing all bss-related
	** information for current output subspace.  As the next subspace would
	** require an increment of new_index, there is no need to modify 
	** prev_end_addr.
	*/
      }
      
#ifdef DEBUG
      if (verbose & V_DOOM) {
	if (do_holeinfo) {
	  fprintf(stderr, 
		  "lmi_create_linkmap : alignment hole filler section %d "
		  "contrib %d/%d/%d/%d/%lld/%lld\n",
		  old_index, 
		  holeinfo.file_index, 
		  holeinfo.mapping_type,
		  holeinfo.output_info,
		  holeinfo.input_info,
		  holeinfo.size,
		  holeinfo.offset);
	} 
	if (do_mapinfoptr) {
	  fprintf(stderr, 
		  "lmi_create_linkmap : writing section %d "
		  "contrib %d/%d/%d/%d/%lld/%lld\n",
		  old_index, 
		  mapinfo_ptr->file_index, 
		  mapinfo_ptr->mapping_type,
		  mapinfo_ptr->output_info,
		  mapinfo_ptr->input_info,
		  mapinfo_ptr->size,
		  mapinfo_ptr->offset);
	}
      }
#endif /* DEBUG */

      /* control next for-loop iteration */
      if (change_index) {
	 new_index++;
      }
      
      /* write out the linkmap entries */
      if (do_holeinfo) {
	if (LM_put_mapping(&holeinfo) != LM_NO_ERROR) {
	  write_error = TRUE;
	}
      }

      if (do_mapinfoptr) {
	if (LM_put_mapping(mapinfo_ptr) != LM_NO_ERROR) {
	  write_error = TRUE;
	}
      }
    }
    
    if (LM_end_write_linkmap(&size) != LM_NO_ERROR) {
      write_error = TRUE;
    } else {
      lmi_buffer_writer_finish(size);
    }
  }
  
  return (dict->subspace_length);
}

/* 
 * write the contents of the bss symbol table (-r only)
 *
 * - return the size of the bss symbol table written.
 * = write errors are not handled locally
 */
static
unsigned int lmi_create_bss(unsigned int offset)
{
  struct subspace_dictionary_record	*dict		= NULL;
  struct subsp_misc_record		*misc		= NULL;
  LM_sym_info				syminfo		= {0};
  unsigned int 			        sym_index	= 0;
  int					size		= 0;
  int					i		= 0;
  

  dict 	= &Subsp_Dict(out_subsp_lm_bss);
  misc 	= &Subsp_Misc(out_subsp_lm_bss);
  misc->exec_file_offset += offset;
  
  /* sort according to output file for most efficient stream */
  if (bst.num_entries) {
    qsort((void *)bst.entries, 
	  (size_t)bst.num_entries,
	  sizeof(lm_bste),
	  lmi_bss_symbol_sort);
  }

  write_subspace = out_subsp_lm_bss;
  if (LM_begin_write_linkmap_bss(lmi_buffer_writer) != LM_NO_ERROR) {
    /* if we can't generate info, do not generate subspace */
    lmi_strip_subspace(out_subsp_lm_bss);
  } else {
    for (i=0; i<bst.num_entries; i++) {
      sym_index = bst.entries[i].sym_index;
      syminfo.file_index = 
	oft.entries[Sym_Misc(sym_index).objfile_index].file_index;
      syminfo.sym_index  = Sym_Out_Remap(sym_index);
      syminfo.is_tls = Sym_Misc(sym_index).tls;
      
#ifdef DEBUG
      if (verbose & V_DOOM) {
	fprintf(stderr, 
		"lmi_create_bss(): writing symbol info for %s %d/%d/%d\n",
		Sym_Name(bst.entries[i].sym_index), 
		syminfo.file_index,
		syminfo.sym_index,
		syminfo.is_tls);
      }
#endif /* DEBUG */

      if (LM_put_bss_symbol(&syminfo) != LM_NO_ERROR) {
	write_error = TRUE;
      }
    }
    
    if (LM_end_write_linkmap_bss(&size) != LM_NO_ERROR) {
      write_error = TRUE;
    } else {
      lmi_buffer_writer_finish(size);
    }
  }
  
  return (dict->subspace_length);
}

#endif /* CNX_TOOLS */

/* 
** Externally visible interfaces ===================================
*/
void lm_options_noobjdebug(void) 
{
  options_noobjdebug		= TRUE;	
}

void lm_options_nocopyobjdebug(void) 
{
  options_nocopyobjdebug 	= TRUE;
}

void lm_options_copyobjdebug(void) 
{
  options_copyobjdebug 		= TRUE;
}

void lm_options_nolinkmap(void) 
{
  options_nolinkmap		= TRUE;
}

void lm_options_linkmap(void) 
{
  options_linkmap		= TRUE;
}

void lm_options_tools(void) 
{
  options_tools 		= TRUE;
}

void lm_options_strip_s(void) 
{
  options_strip_dash_s		= TRUE;
}

void lm_options_strip_G(void) 
{
  options_strip_dash_G		= TRUE;
}

/*
 * options-based decision making on whether or not
 * to create a linkmap.  also, one time initialization.
 */
void lm_finish_options(void)
{
#ifdef CNX_TOOLS
  char link_dir[PATH_MAX];

  /* add current link directory to the output string table */
  getcwd(link_dir, PATH_MAX);
  lmi_init_string_table();
  link_dir_offset = lmi_add_string(link_dir);

  /* allocate this one here, so that free() in lm_som_add() works */
  bss_remap = (unsigned int *)emalloc(4);

  /* handle options */
  if (options_tools || options_strip_dash_s ||options_strip_dash_G) {
    objdebug_mode		= SUPPRESS;
    linkmap_mode		= SUPPRESS;
    return;
  }

  if (options_noobjdebug && (!relocatable || options_nocopyobjdebug)) {
    objdebug_mode		= SUPPRESS;
    linkmap_mode		= SUPPRESS;
  }

  if (options_nolinkmap) {
    linkmap_mode		= SUPPRESS;
  }

  if (options_linkmap) {
    linkmap_mode		= CREATE;
  }
#endif /* CNX_TOOLS */
}

/*
 * add current object file to the output object file table
 *
 * = set the value for cur_objfile_index.
 */
void lm_som_add(void)
{
#ifdef CNX_TOOLS
  static int bss_remap_size = 0;
  char dir_name[PATH_MAX], file_name[PATH_MAX];
  char *ptr = NULL;
  int i;

  /* add current object file */
  if (module_name == NULL) {
    ptr = mb_strrchr(f_name, '/');
    if (ptr) {
      memset(dir_name, 0, PATH_MAX-1);
      memset(file_name, 0, PATH_MAX-1);
      strncpy(dir_name, f_name, (ptr-f_name));
      strcpy(file_name, ptr+1);
      cur_objfile_index = lmi_create_file_entry(dir_name, file_name);
      oft.entries[cur_objfile_index].file_entry.is_archive  = FALSE;
      oft.entries[cur_objfile_index].file_entry.is_basename = TRUE;
    } else {
      cur_objfile_index = lmi_create_file_entry(NULL, f_name);
      oft.entries[cur_objfile_index].file_entry.is_archive  = FALSE;
      oft.entries[cur_objfile_index].file_entry.is_basename = FALSE;
    }
  } else {
    cur_objfile_index = lmi_create_file_entry(f_name, module_name);
    oft.entries[cur_objfile_index].file_entry.is_archive  = TRUE;
    oft.entries[cur_objfile_index].file_entry.is_basename = TRUE;
  }    
  /* object was seen on the link line */
  oft.entries[cur_objfile_index].file_index        = (-1); 

  /* record subspace bias to map internal subspace index to input index */
  oft.entries[cur_objfile_index].subsp_bias        = cur_subsp_index_bias;

  /* need to remap bss symbols - only if bigger (to save on malloc()) */
  if (bss_remap_size<cur_header.symbol_total) {
    free(bss_remap);
    bss_remap_size = cur_header.symbol_total;
    bss_remap = (unsigned int *)emalloc
      (sizeof(unsigned int)*bss_remap_size);    
  }
  memset(bss_remap, 0, sizeof(unsigned int)*bss_remap_size);
  
#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lm_som_add(): added (explicit) entry for %s at index %d\n"
	    "\tcreated a bss remap for %d symbols\n",
	    f_name, 
	    cur_objfile_index,
	    cur_header.symbol_total);
  }
#endif /* DEBUG */

#endif /* CNX_TOOLS */
}

/* 
 * set the size of the compunits for the current object file 
 */
void lm_compunit_read(unsigned int compsize)
{
#ifdef CNX_TOOLS
  oft.entries[cur_objfile_index].compsize = compsize;
#endif /* CNX_TOOLS */
}

/* 
 * process space entries for the current object file
 * figure out what kind of link is being performed
 * strip extraneous spaces that should not make it
 * record input linkmap's subspace indices
 */
void lm_spaces_add(unsigned int debug_sp_idx,
		   unsigned int objdebug_sp_idx,
		   unsigned int linkmap_sp_idx,
		   unsigned int igndbg_sp_idx)
{
#ifdef CNX_TOOLS
  struct subspace_dictionary_record	*dict	= NULL;
  struct subsp_misc_record		*misc	= NULL;
  char 					*name	= NULL;
  int i;
  
  /* reinitialize indices */
  subsp_lm                = BAD_SUBSP;
  subsp_lm_str            = BAD_SUBSP;
  subsp_lm_bss            = BAD_SUBSP;
  subsp_lm_file           = BAD_SUBSP;

  /* 
  ** Multiple spaces with the same name(unlike subspaces) get 
  ** merged into a single space record.  In $DEBUG$ handling, we
  ** may have to strip $DEBUG$ in the current object file, but 
  ** not in the whole link, so strip_debug_sp_idx will mark whether
  ** $DEBUG$ in the current object file should be stripped or not 
  */
  strip_debug_sp_idx	  = BAD_SPACE;

  /* objdebug handling */
  if (objdebug_sp_idx != BAD_SPACE) {
    if (objdebug_mode == SUPPRESS) {
      /* strip all input $OBJDEBUG$ spaces */
      space_misc[objdebug_sp_idx].is_doom_stripped = TRUE;
    } else {
      objdebug_mode = CREATE;
      if (linkmap_mode == DEFAULT) {
	linkmap_mode = CREATE;
      }    
    }
  } 
  
  /* debug handling */
  if (debug_sp_idx != BAD_SPACE) {
    /* objdebug file with no linkmap */
    if ((objdebug_mode == CREATE) && 
	(objdebug_sp_idx != BAD_SPACE) &&
	(linkmap_sp_idx == BAD_SPACE)) {
      strip_debug_sp_idx = debug_sp_idx;
    } else {
      /* only have debug */
      if(igndbg_sp_idx != BAD_SPACE) {
        strip_debug_sp_idx = debug_sp_idx;
      } else {
	copied_debug = TRUE;
      }
    }
  }

  /* 
   * if we copied debug, don't strip space,
   * but may still strip some subspaces, as
   * determined by strip_debug_sp_idx
   */
  if (debug_sp_idx != BAD_SPACE) {
    if (copied_debug) {
      space_misc[debug_sp_idx].is_doom_stripped = FALSE;
    } else {
      space_misc[debug_sp_idx].is_doom_stripped = TRUE;
    }
  }

  /* igndbg handling - always strip out */
  if (igndbg_sp_idx != BAD_SPACE) {
    space_misc[igndbg_sp_idx].is_doom_stripped = TRUE;
  }

  /* linkmap handling */
  if (linkmap_sp_idx != BAD_SPACE) {
    if (linkmap_mode == CREATE) {
      for (i=0; i<cur_header.subspace_total; i++) {
	dict = &Subsp_Dict(i+cur_subsp_index_bias);
	name = (char *)(dict->name.n_strx+space_str_buffer.block_ptr);
	if (strcmp(name,"$LINKMAP$") == 0) {
	  subsp_lm 		= i+cur_subsp_index_bias;
	} else if (strcmp(name,"$LINKMAP_BSS$") == 0) {
	  subsp_lm_bss 		= i+cur_subsp_index_bias;
	} else if (strcmp(name,"$LINKMAP_STR$") == 0) {
	  subsp_lm_str 		= i+cur_subsp_index_bias;
	} else if (strcmp(name,"$LINKMAP_FILE$") == 0) {
	  subsp_lm_file 		= i+cur_subsp_index_bias;
	}
      }

#ifdef DEBUG
      if (verbose & V_DOOM) {
	fprintf(stderr, 
		"lm_spaces_add(): detected following subspace indices:\n"
		"\t%12s\t%d\n\t%12s\t%d\n\t%12s\t%d\n\t%12s\t%d\n",
		"linkmap_str" , subsp_lm_str, 
		"linkmap_file", subsp_lm_file,
		"linkmap"     , subsp_lm,
		"linkmap_bss" , subsp_lm_bss);
      }
#endif /* DEBUG */
      
    } 
    /* strip the space, lm_subspace_add() will strip subspace */
    space_misc[linkmap_sp_idx].is_doom_stripped = TRUE;
  }

  /* warning messages and such */
  if (verbose & V_WHY) {
    static int emitted_msg = FALSE;
    if (!emitted_msg && copied_debug && (objdebug_mode == CREATE)) {
      emitted_msg = TRUE;
      printf(ld_gets(1, POSSIBLE_MISSING_DEBUG, "(Warning) Mixing new "
		     "and old style debugging is not supported.  Some "
		     "debug information may be missing in the SOM file.\n"));
    }
  }
#endif /* CNX_TOOLS */
}

/* 
 * add a subspace entry to BSS subspace table.
 *
 * =  realloc the array in LM_BSUBTBL_ALLOC chunks
 */
void lm_bss_subspace_add(unsigned int subsp_idx, unsigned int start)
{
#ifdef CNX_TOOLS
  static unsigned int num_entries = 0;
  
  /* 
  ** if this object contains embedded linkmap, do not
  ** add its BSS subspaces to BSS subspace table, as we 
  ** will process contributions directly.
  ** if it is a TBSS subspace, we still need to add it in the table
  ** so that we will get the starting TP-relative offset of the TBSS
  ** subspace.
  */
  if (subsp_lm != BAD_SPACE && ! Subsp_Dict(subsp_idx).is_tspecific) {
    return;
  }

  if (num_entries == bsst.num_entries) {
    num_entries += LM_BSUBTBL_ALLOC;
    bsst.entries = (lm_bsste *)erealloc((char *)bsst.entries,
					sizeof(lm_bsste)*num_entries);
  }
  
  (bsst.entries[bsst.num_entries]).length = Subspace_Length(subsp_idx);
  (bsst.entries[bsst.num_entries]).out_addr = start;
  (bsst.entries[bsst.num_entries]).in_subsp_index= subsp_idx;
  (bsst.entries[bsst.num_entries]).subsp_index= BAD_SUBSP;
  bsst.num_entries++;

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lm_bss_subspace_add(): added subspace %d "
	    "(idx %d in file %d) with length %d start %d\n", 
	    subsp_idx,
	    subsp_idx-oft.entries[Subsp_Misc(subsp_idx).objfile_index].subsp_bias,
            cur_objfile_index,
	    Subspace_Length(subsp_idx),
	    start);
  }
#endif /* DEBUG */

#endif /* CNX_TOOLS */
}

/*
 * add a symbol entry to BSS symbol table.
 *
 * = realloc the array in LM_BSYMTBL_ALLOC chunks
 */
void lm_bss_symbol_add(unsigned int sym_idx, unsigned int length)
{
#ifdef CNX_TOOLS
  static unsigned int num_entries = 0;
  
  if (num_entries == bst.num_entries) {
    num_entries += LM_BSYMTBL_ALLOC;
    bst.entries = (lm_bste *)erealloc((char *)bst.entries,
					   sizeof(lm_bste)*num_entries);
  }
  
  bst.entries[bst.num_entries].sym_index = sym_idx;
  bst.entries[bst.num_entries++].length	 = length;

#ifdef DEBUG
  if (verbose & V_DOOM) {
    fprintf(stderr, 
	    "lm_bss_symbol_add(): added symbol (input_idx) %d len: %d\n", 
	    sym_idx,
	    length);
  }
#endif /* DEBUG */

#endif /* CNX_TOOLS */
}

/* 
 * add all unsatisfied ST_[T]STORAGE symbols to the 
 * bss symbol table (used by linkmap_bss)
 */
void lm_process_stor_unsats(void)
{
#ifdef CNX_TOOLS
  int i;
  struct symnode *n;
  
  for (i=0; i<UNSATHASHSIZE; i++) {
    for (n=unsat_hash_table[i]; n; n = n->next) {
      if (n->type == ST_STORAGE || n->type == ST_TSTORAGE) {
	lm_bss_symbol_add(n->index, n->len);

#ifdef DEBUG
	if (verbose & V_DOOM) {
	  fprintf(stderr, 
		  "lm_process_stor_unsats(): added sym %s "
		  "with index %d len %d\n",
		  n->name, 
		  n->index,
		  n->len);
	}
#endif /* DEBUG */

      }
    }
  }
#endif /* CNX_TOOLS */
}

/* 
 * invoke pxdb if any debug spaces have been copied
 */
int lm_should_invoke_pxdb(void)
{
#ifdef CNX_TOOLS
  return (copied_debug);
#else
  return 0;
#endif /* CNX_TOOLS */
}

/* 
 * disable line table generation for objdebug link
 */
int lm_should_disable_linetables(void)
{
#ifdef CNX_TOOLS
  return (objdebug_mode == CREATE);
#else
  return 0;
#endif /* CNX_TOOLS */
}

/* 
 * read the input linkmap (embedded in the object file).
 *
 * = any read errors are handled by setting all subspaces
 * that came from the current object file to produce 
 * undefined regions in the output linkmap.
 */
void lm_linkmap_read(void)
{
#ifdef CNX_TOOLS
  struct subspace_dictionary_record	*dict	= NULL;
  struct subsp_misc_record		*misc	= NULL;
  struct input_subsp_contrib		*contrib=NULL, *last_contrib=NULL;
  LM_sym_info				syminfo = {0};
  LM_mapping_info			mapinfo = {0};
  LM_errcode				read_ret= LM_NO_ERROR;
  unsigned int				subsp_index = BAD_SUBSP;
  unsigned int				last_index = BAD_SUBSP;
  unsigned int				numfiles= 0;
  int 					i, idx 	= 0;
  char 					*ptr    = NULL;
  

  if (linkmap_mode != CREATE) {
    /* nothing to do */
    return;
  } else if ((subsp_lm_str  == BAD_SUBSP) && 
	     (subsp_lm_file == BAD_SUBSP) &&
	     (subsp_lm      == BAD_SUBSP)) {
    /* no input linkmap */
    return;
  } else if ((subsp_lm_str  == BAD_SUBSP) ||
             (subsp_lm_file == BAD_SUBSP) ||
             (subsp_lm      == BAD_SUBSP)) {
    /* 
     * some are present and some are absent, something had gone 
     * wrong, so mark all subspaces as undefined_regions.
     */
    read_error = TRUE;
  }
  
  /* read in string table */
  if (!read_error) {
    dict = &Subsp_Dict(subsp_lm_str);
    misc = &Subsp_Misc(subsp_lm_str);
    if (dict->subspace_length) {
      ptr = (char *)emalloc(dict->subspace_length);
      ffetch(cur_fildes,
	     dict->file_loc_init_value + cur_ofs,
	     ptr,
	     sizeof(char),
	     dict->subspace_length);
      st_in.data = ptr;
      st_in.data_size = dict->subspace_length;
    } else {
      st_in.data = NULL;
      st_in.data_size = 0;
    }
  }
	
  /* read in and process object file table */
  if (!read_error) {
    /* read in part of the linkmap for version numbers and such */
    read_subspace = subsp_lm;
    if (LM_begin_read_linkmap(lmi_buffer_reader) != LM_NO_ERROR) {
      read_error = TRUE;
    }else {
      misc = &Subsp_Misc(subsp_lm_file);
      read_subspace = subsp_lm_file;
      if (LM_begin_read_linkmap_file(lmi_buffer_reader) != LM_NO_ERROR) {
	read_error = TRUE;
      }else {
	if (LM_get_num_files(&numfiles) != LM_NO_ERROR) {
	  read_error = TRUE;
	} else {
	  for (i=0; i<numfiles; i++) {
	    idx = lmi_new_file_entry();
	    if (LM_get_file_entry(i, &(oft.entries[idx].file_entry)) != 
		LM_NO_ERROR) {
	      read_error = TRUE;
	    }
	    
#ifdef DEBUG
	    if (verbose & V_DOOM) {
	      fprintf(stderr, 
		      "lm_linkmap_read(): added (implicit) entry at index %d\n",
		      idx);
	    }
#endif /* DEBUG */
	    
	    /* mark this object file entry as implicit */
	    oft.entries[idx].file_index = (-2);
	    oft.entries[idx].subsp_bias = 
	      oft.entries[cur_objfile_index].subsp_bias;
	    lmi_process_file_entry(&(oft.entries[idx].file_entry));
	  }
	}

	if (LM_end_read_linkmap_file() != LM_NO_ERROR) {
	  read_error = TRUE;
	}	
      }
    }
  }

  /* 
   * if current object file contains a linkmap, we must not 
   * output its compsize to the output linkmap.  Reset the 
   * values, so that creation will skip them (it's however
   * too late for the strings, which are now part of the 
   * output string table (and are never referenced).
   */
  if (!read_error) {
    oft.entries[cur_objfile_index].compsize = 0;
  }

  /* read in and process input linkmap */
  if (!read_error) {
    misc = &Subsp_Misc(subsp_lm);
    read_subspace = subsp_lm;
    /* linkmap has already been initialized, so start reading */
    contrib = lmi_get_input_contrib();
    lmi_init_contrib(contrib);
    while ((read_ret = LM_get_next_mapping(&(contrib->map))) == 
	   LM_NO_ERROR) {
      /* data structure */
      if (contrib->map.mapping_type == LM_DATA_STRUCTURE) {
	oft.entries[cur_objfile_index+contrib->map.file_index+1].compsize = 
	  contrib->map.size;
	
#ifdef DEBUG
	if (verbose & V_DOOM) {
	  fprintf(stderr, 
		  "lm_linkmap_read(): assigned compsize %lld for entry at index %d\n",
		  contrib->map.size, (cur_objfile_index+contrib->map.file_index+1));
	}
#endif /* DEBUG */
	
      } else {
	/* attach subspace contrib asis, just remap subspace index */
	subsp_index = contrib->map.output_info;
	/* because input is sorted, we can append record to the end */
	if (last_index == subsp_index) {
	  last_contrib->next 	= contrib;
	  last_contrib 	= contrib;
	} else {
	  last_index = subsp_index;
	  last_contrib = contrib;	
	  Subsp_Misc(subsp_index+cur_subsp_index_bias).contrib = contrib;
	}	
	
#ifdef DEBUG
	if (verbose & V_DOOM) {
	  fprintf(stderr, 
		  "lm_linkmap_read(): added contrib to "
		  "subsp %d (original %d)\n",
		  subsp_index+cur_subsp_index_bias,
		  subsp_index);
	}
#endif /* DEBUG */
	
	/* allocate space for the new contrib */
	contrib = lmi_get_input_contrib();
	lmi_init_contrib(contrib);
      }
    }

    if ((read_ret != LM_NO_ERROR) &&
	(read_ret != LM_USER_CALLBACK_NULL)) {
      read_error = TRUE;
    }
      
    if (LM_end_read_linkmap() != LM_NO_ERROR) {
      read_error = TRUE;
    }
  }
  
  /* read in and process bss symbols */
  if (!read_error) {
    misc = &Subsp_Misc(subsp_lm_bss);
    read_subspace = subsp_lm_bss;
    if (LM_begin_read_linkmap_bss(lmi_buffer_reader) != LM_NO_ERROR) {
      read_error = TRUE;
    } else {
      while (LM_get_next_bss_symbol(&syminfo) != LM_USER_CALLBACK_NULL) {
	bss_remap[syminfo.sym_index] = cur_objfile_index+syminfo.file_index+1;

#ifdef DEBUG
	if (verbose & V_DOOM) {
	  fprintf(stderr, 
		  "lm_linkmap_read(): remap bss symbol "
		  "%s(%d) from file entry %d\n",
		  _sym_record[syminfo.sym_index+cur_sym_index_bias]->dict.name.n_strx+
                     sym_str_buffer.block_ptr,
		  syminfo.sym_index,
		  cur_objfile_index+syminfo.file_index+1);
	}
#endif /* DEBUG */

      }	

      if (LM_end_read_linkmap_bss() != LM_NO_ERROR) {
	read_error = TRUE;
      }
    }
  }

  /* if there was a read error, mark all subspaces undef */
  if (read_error) {
    read_error = FALSE;
    for (i=0; i<cur_header.subspace_total; i++) {
      idx = i+cur_subsp_index_bias;
      Subsp_Misc(idx).is_undefined_region = TRUE;
    }
  }

  /* 
  ** lm_spaces_add() marked input $LINKMAP$ for stripping.  However,
  ** since we will be creating output $LINKMAP$ and linker merges
  ** all spaces with the same name, we need to "unmark" $LINKMAP$
  ** space from doom stripping
  */
  Sub_Space_Misc(subsp_lm_str).is_doom_stripped = FALSE;
#endif /* CNX_TOOLS */
}

/*
 * build linkmap subspaces and remember output indices 
 *
 * = initialize size to be 4 bytes to make relocate_subspaces()
 * function assign starting addresses to these subspaces.
 * = do not create linkmap entries for linkmap subspaces.
 */
void lm_linkmap_build_subspaces(void)
{
#ifdef CNX_TOOLS
  if (linkmap_mode != CREATE) {
    return;
  }

  /* create subspaces */
  out_subsp_lm_str = build_subspace("$LINKMAP_STR$", "$LINKMAP$",
				    77, WILDCARD_ACCESS, 
				    0, 20, 4, 0, 4);
  out_subsp_lm_file = build_subspace("$LINKMAP_FILE$", "$LINKMAP$",
				     77, WILDCARD_ACCESS, 
				     0, 25, 4, 0, 4);
  out_subsp_lm = build_subspace("$LINKMAP$", "$LINKMAP$",
				77, WILDCARD_ACCESS, 
				0, 30, 4, 0, 4);

  /* initialize the data structures */
  lmi_init_subspace(out_subsp_lm_str);
  lmi_init_subspace(out_subsp_lm_file);
  lmi_init_subspace(out_subsp_lm);
  
  /* point subsp_data to some 0-filled struct, in case of write error */
  Subsp_Misc(out_subsp_lm_str).subsp_data = lm_subsp_data4;
  Subsp_Misc(out_subsp_lm_file).subsp_data = lm_subsp_data4;
  Subsp_Misc(out_subsp_lm).subsp_data = lm_subsp_data4;

  if (relocatable) {
    out_subsp_lm_bss = build_subspace("$LINKMAP_BSS$", "$LINKMAP$",
			              77, WILDCARD_ACCESS, 
				      0, 35, 4, 0, 4);

    lmi_init_subspace(out_subsp_lm_bss);
    Subsp_Misc(out_subsp_lm_bss).subsp_data = lm_subsp_data4;
  }
#endif /* CNX_TOOLS */
}

/*
 * create the output linkmap by calling helper
 * functions.  adjust the offsets after creation.
 * make sure to account for initial size of each
 * subspace of 4 bytes.
 *
 * = if write error has occurred, do not output
 * linkmap subspaces.
 */
void lm_linkmap_create(void)
{
#ifdef CNX_TOOLS
  int idx1, idx2, off1, off2, size;
  unsigned int lsize = 0;
  
  if (linkmap_mode != CREATE) {
    return;
  }

  /* reset sizes of each subspace */
  lmi_set_subspace_length(out_subsp_lm_str, 0);
  lmi_set_subspace_length(out_subsp_lm_file, 0);
  lmi_set_subspace_length(out_subsp_lm, 0);
  if (relocatable) {
    lmi_set_subspace_length(out_subsp_lm_bss, 0);
  }

  /* output string table */
  if (!write_error) {
    lsize += lmi_create_string_table(0);
  } 

  /* output object file table */
  if (!write_error) {
    lsize += lmi_create_objfile_table(lsize-4);
  }
  
  /* output linkmap */
  if (!write_error) {
    lsize += lmi_create_linkmap(lsize-8);
  }

  /* output linmap's bss */
  if (!write_error) {
    if (relocatable) {
      lsize += lmi_create_bss(lsize-12);
    }
  }

  /* check that the linkmap subspaces are consecutive */
  idx1 = Subsp_Misc(out_subsp_lm_str).r_c_x;
  idx2 = Subsp_Misc(out_subsp_lm_file).r_c_x;
  off1 = Subsp_Misc(out_subsp_lm_str).exec_file_offset;
  off2 = Subsp_Misc(out_subsp_lm_file).exec_file_offset;
  size = Subsp_Dict(out_subsp_lm_str).subspace_length;
  if ((idx1+1 != idx2) || (LM_ALIGN_4(off1+size) != off2)) {
     write_error = TRUE;
  }
  idx1 = Subsp_Misc(out_subsp_lm_file).r_c_x;
  idx2 = Subsp_Misc(out_subsp_lm).r_c_x;
  off1 = Subsp_Misc(out_subsp_lm_file).exec_file_offset;
  off2 = Subsp_Misc(out_subsp_lm).exec_file_offset;
  size = Subsp_Dict(out_subsp_lm_file).subspace_length;
  if ((idx1+1 != idx2) || (LM_ALIGN_4(off1+size) != off2)) {
     write_error = TRUE;
  }
  if (relocatable) {
     idx1 = Subsp_Misc(out_subsp_lm).r_c_x;
     idx2 = Subsp_Misc(out_subsp_lm_bss).r_c_x;
     off1 = Subsp_Misc(out_subsp_lm).exec_file_offset;
     off2 = Subsp_Misc(out_subsp_lm_bss).exec_file_offset;
     size = Subsp_Dict(out_subsp_lm).subspace_length;
     if ((idx1+1 != idx2) || (LM_ALIGN_4(off1+size) != off2)) {
        write_error = TRUE;
     }
  }

  if (!write_error) {
    /* reassign subspace locations */
    assign_file_locations();

    /* mark subspaces as having their data outputted */
    Subsp_Misc(out_subsp_lm).subsp_data_out = TRUE;
    Subsp_Misc(out_subsp_lm_str).subsp_data_out = TRUE;
    Subsp_Misc(out_subsp_lm_file).subsp_data_out = TRUE;
    if (relocatable) {
      Subsp_Misc(out_subsp_lm_bss).subsp_data_out = TRUE;
    }
  } else {
     /* reset sizes of each subspace */
     lmi_set_subspace_length(out_subsp_lm_str, 4);
     lmi_set_subspace_length(out_subsp_lm_file, 4);
     lmi_set_subspace_length(out_subsp_lm, 4);
     if (relocatable) {
        lmi_set_subspace_length(out_subsp_lm_bss, 4);
     }
     /* we already assigned subsp_data to some 0-filled struct */
  }
#endif /* CNX_TOOLS */
}

