/*
 *  HP 9000 Series 800 Linker, Copyright Hewlett-Packard Co. 1985-1999  
 *  Subspace Dictionary Manipulation Routines  
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Header: /home/cvs/cvsroot/linker/subspaces.c,v 1.1.1.1 1999/10/18 19:53:03 pschwan Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/times.h>
#include <memory.h>
#include <assert.h>

#include "filehdr.h"
#include "aouthdr.h"
#include "spacehdr.h"
#include "scnhdr.h"
#include "reloc.h"

#include "std.h"
#include "driver.h"
#include "errors.h"
#include "fixups.h"
#include "libraries.h"
#include "ldlimits.h"
#include "linker.h"
#include "spaces.h"
#include "subspaces.h"
#include "ld_strings.h"
#include "syms.h"
#include "symbols.h"
#include "util.h"
#include "embed.h"

#ifdef CNX_TOOLS
#include "cnx_link_info.h"
#endif /* CNX_TOOLS */
  
/* doom support */
#include "ld_linkmap.h"

/* Globals */

struct subspace_dictionary_record **_subsp_dict = NULL;
struct subsp_misc_record **_subsp_misc = NULL;
int    subsp_dict_size = 0;
int    new_subsp_dict_size = 0;
int    subsp_segs = 0;
int    subsp_segs_max = MAX_SEGS;
int    next_subsp_index_bias = 0;
int    cur_subsp_index_bias = 0;
int    data_file_offset = 0;
int    subsp_data_area_max = SUBSP_AREA_MAX;
int    data_fildes = -1;
Boolean mr_seen = FALSE;  /* has a Memory Resident subspace been seen? */
char   *subsp_data_area = 0;
char   *data_file;

extern Boolean do_procedure_fdp;
extern char *fdp_counter_file_name; 
extern void get_FDP_list();

extern Boolean obj_has_dbg;

/* Macros */

/* HP-UX shared libraries */			
#define INIT_MISC_SHL_VERSION(misc)                                          \
    (misc)->shlib_version = shlib_version; /* version num for this module */ 

#define INIT_MISC_ENTRY_SYM(misc)	(misc)->entry_sym = BAD_SYM;

#define INITIALIZE_SUBSP_MISC(misc, subsp, index)                            \
    /* common subspace misc record initialization routine used by            \
       subspace_add() and build_subspace(). If an assignment is not          \
       made to a specific field, that field will have the value of 0 */      \
    {									     \
    /* clear the subspace_misc_record to zero/NULL */		             \
    memset((char *) (misc), '\0', sizeof(struct subsp_misc_record));         \
									     \
    (misc)->r_o_x = (index);				                     \
    (misc)->r_n_x = -1;							     \
    (misc)->r_c_x = -1;							     \
    (misc)->file_name = cur_name;					     \
    (misc)->f_name = f_name;						     \
    (misc)->module_name = module_name;					     \
    (misc)->input_name = NULL;						     \
    (misc)->symbol_index_bias = cur_sym_index_bias;			     \
    (misc)->new_subspace_start = (subsp)->subspace_start;		     \
    (misc)->first_common_def = -1;					     \
    (misc)->module_id = module_seq_num;                                      \
    INIT_MISC_SHL_VERSION(misc)                                              \
    INIT_MISC_ENTRY_SYM(misc)						     \
    (misc)->obj_has_dbg = FALSE;                                             \
									     \
    /* doom support */							     \
    (misc)->is_doom_stripped = FALSE;				     	     \
    (misc)->is_bss_subspace = FALSE;				     	     \
    (misc)->is_linkmap_entry = TRUE;				     	     \
    }									     

void subsp_initialize()
{
    int seg_size;

    subsp_dict_size = 0;
    mr_seen = FALSE;
    new_subsp_dict_size = 0;
    next_subsp_index_bias = 0;
    cur_subsp_index_bias = 0;
    data_file_offset = 0;

    if (_subsp_dict == 0) {

        /* allocate subspace dictionary segment pointers */
        _subsp_dict = (struct subspace_dictionary_record **)
		emalloc(subsp_segs_max *
			sizeof(struct subspace_dictionary_record *));
        _subsp_misc = (struct subsp_misc_record **)
		emalloc(subsp_segs_max * sizeof(struct subsp_misc_record *));
     }
} /* end subsp_initialize */

subspace_init_after_driver()
{
    FILE *f;

    if (two_pass)
        subsp_data_area_max = 0;

    if (save_data) {
	/* open temporary data file to store code & data */
	if (data_fildes == -1) {
	    data_file = tempnam(DEFAULT_TEMP_DIR, "LDD");
	    f = efopen(data_file,"w+", CANT_OP_TMP_DATA_FL);
	    data_fildes = fileno(f);
	}
	if (subsp_data_area == 0)
   	    subsp_data_area = (char *) emalloc(subsp_data_area_max);
    } 
} /* end subspace_init_after_driver */

#define NOT_FOUND   (-1)

/* general subspace management routines */
int build_subspace(subspace_name,
		   space_name,
		   space_sort_key,
		   access_control_bits,
		   quadrant,
		   sort_key,
		   alignment,
		   subspace_order,
		   subsp_length)

char *subspace_name;  	  /* name of the subspace to build */
char *space_name;  	  /* name of the space to build this subspace for */
int space_sort_key;
int access_control_bits;
int quadrant;  
int sort_key;
int alignment;
int subspace_order;
unsigned int subsp_length;

    /* builds and initializes data structures for one new subspace to be 
       created from scratch (not read in).  Return value is subspace index. 
       */

{
    int seg, off, index;
    struct subspace_dictionary_record *subsp;
    struct subsp_misc_record *misc;
    int space_index;  			/* index for the space */
    int i;  				/* loop counter */

    cur_subsp_index_bias = next_subsp_index_bias;
    index = subsp_dict_size;
    seg = Subsp_Seg(index);
    off = Subsp_Off(index);
    if (seg >= subsp_segs) {  /* This segment full, build another */
	if (seg >= subsp_segs_max) {  /* This ptr array full, build another */
	    subsp_segs_max += MAX_SEGS;
	    _subsp_dict = (struct subspace_dictionary_record **)
			erealloc((char *) _subsp_dict, subsp_segs_max *
				sizeof(struct subspace_dictionary_record *));
	    _subsp_misc = (struct subsp_misc_record **)
			erealloc((char *) _subsp_misc, subsp_segs_max *
				sizeof(struct subsp_misc_record *));
	}  /* END This pointer array full, build another */
	_subsp_dict[seg] = (struct subspace_dictionary_record *)
		emalloc(subsp_dict_max *
			sizeof(struct subspace_dictionary_record));
	_subsp_misc[seg] = (struct subsp_misc_record *)
		emalloc(subsp_dict_max *
			sizeof(struct subsp_misc_record));
	subsp_segs++;
    }  /* END This segment full, build another */

    next_subsp_index_bias = ++subsp_dict_size;

    subsp = & Subsp_Dict(index);
    misc = & Subsp_Misc(index);

    /* 
    ** clear the subspace_dictionary_record to zero/FALSE - corrections below 
    */
    memset((char *)subsp, '\0', sizeof(struct subspace_dictionary_record));
	
    /* find the space to insert record in */
    space_index = NOT_FOUND;
    for (i = 0; i < cur_space_total; i++)
	if (strcmp(space_name, space_array[i].NAME_PT) == 0) {
	    space_index = i;
	    break;
	}
    if (space_index == NOT_FOUND) {  /* the space wasn't found */
	/* was : user_error(REQ_SPACE_NOT_FOUND, space_name); */
	space_index = build_space(space_name, space_sort_key);
    }

    space_array[space_index].subspace_quantity++;
    subsp->space_index = space_index;

    subsp->access_control_bits = access_control_bits;
    subsp->quadrant = quadrant;
    subsp->sort_key = sort_key;

    /*
    ** If the subspace alignment is larger than the space alignment, 
    ** increase the space alignment so that it is always at least as
    ** large as all of the subspaces it contains.  This happens specifically
    ** when $SHLIB_INFO$ is added (alignment 8) by the linker to the $TEXT$
    ** space (alignment 4).
    */

    subsp->alignment = alignment;
    if (alignment > space_misc[space_index].space_align) {
       space_misc[space_index].space_align = alignment;
    }
    subsp->is_loadable = TRUE;

    subsp->NAME_PT = add_string(&space_strings, 
				subspace_name, 
				hash_string(subspace_name), 
				FALSE, 
				TRUE); 

    subsp->fixup_request_index = -1;  /* don't leave uninit */

    subsp->subspace_length = subsp_length;  /* init length */
#ifdef TSD /* TSD */
    subsp->is_tspecific = 0;
#endif /* TSD */

    /* initialize misc record */
    INITIALIZE_SUBSP_MISC(misc, subsp, index);
    /* let shared libs set cur_name to null---misc->file_name = NULL; */
 
 
    misc->data_file_offset = IN_MEMORY; /* subsp data will be left in memory */
    misc->subsp_order = subspace_order; /*subsp order specified in cmd file*/

    /* doom support */
    lm_linker_gened_subspace(index);

    return(index);
} /* build_subspace */

void subspaces_read(fil_ofs, nsubsp)
int fil_ofs, nsubsp;
{
    int index, seg, off, count;
    struct subspace_dictionary_record *subsp;

    cur_subsp_index_bias = next_subsp_index_bias;
    while (nsubsp > 0) {
        index = subsp_dict_size;
        seg = Subsp_Seg(index);
        off = Subsp_Off(index);
        if (seg >= subsp_segs) {  /* This segment full, build another */
            /* this ptr array full, build another */
	    if (seg >= subsp_segs_max){ 
	        subsp_segs_max += MAX_SEGS;
	        _subsp_dict = (struct subspace_dictionary_record **)
			erealloc((char *) _subsp_dict, subsp_segs_max *
				sizeof(struct subspace_dictionary_record *));
	        _subsp_misc = (struct subsp_misc_record **)
			erealloc((char *) _subsp_misc, subsp_segs_max *
				sizeof(struct subsp_misc_record *));
	    }  /* END This pointer array full, build another */
	    _subsp_dict[seg] = (struct subspace_dictionary_record *)
		    emalloc(subsp_dict_max *
			sizeof(struct subspace_dictionary_record));
	    _subsp_misc[seg] = (struct subsp_misc_record *)
		    emalloc(subsp_dict_max *
			sizeof(struct subsp_misc_record));
	    subsp_segs++;
	}  /* END This segment full, build another */

        subsp = &_subsp_dict[seg][off];
        count = min(subsp_dict_max - off, nsubsp);
        ffetch(cur_fildes, 
	       fil_ofs, 
	       subsp,
	       sizeof(struct subspace_dictionary_record), 
	       count);

        subsp_dict_size += count;
        nsubsp -= count;
        fil_ofs += count * sizeof(struct subspace_dictionary_record);
    }

    next_subsp_index_bias = subsp_dict_size;
} /* end subspaces_read */

void subspace_add(int index)
/* add 1 record at a time to subsp_array, update subsp_array_size */
{
    int same;
    struct subspace_dictionary_record *subsp, *subsp2;
    struct subsp_misc_record *misc, *misc2;
    int len;
    char *c;
    char *file_name;
    char *mod_name;

    subsp = & Subsp_Dict(index);
    misc = & Subsp_Misc(index);

    /* adjust subspace record fields and initialize misc record */
    subsp->name.n_name = subsp->name.n_strx + space_str_buffer.block_ptr;
    subsp->space_index = sp_remap[subsp->space_index];

    /* 
    ** if the alignment for any subspace is greater than the current page
    ** size we will have difficulty rounding both the file offset locations
    ** and virtual memory locations so that the are equivalent. Since we
    ** only guarentee page alignment for both the file offsets and the start
    ** of data, we have a problem with rounding when one is also aligned 
    ** to the value of "alignment" and the other is not. 
    */
    if (subsp->alignment > page_size) 
	external_error(BAD_SUBSP_ALIGNMENT, subsp->NAME_PT, cur_name, 0);

    /* if linking for dynamic loading -- leave subspaces alone */
    if (cur_file_name == base_file_name && !relinkable) {
	subsp->fixup_request_quantity = 0;
	subsp->subspace_length = 0;
	subsp->initialization_length = 0;
    }

    /* record if memory resident subspace has been seen. */
    mr_seen |= (Boolean) subsp->memory_resident;  

    if (subsp->fixup_request_quantity != 0) {
        if (save_fixups)
            subsp->fixup_request_index += fixup_area_first;
        else
            /* relies on character pointer behaving nicely as integer */
            /* and vice versa */
    	    subsp->fixup_request_index += (int) subsp_fixup_area;
    }

    INITIALIZE_SUBSP_MISC(misc, subsp, index);

#ifdef CNX_TOOLS
    if ((CNX_SUBSPACE_MAP_subsp_index != BAD_SUBSP) && cur_name)
       misc->obj_fname = add_string(&space_strings, cur_name,
                                    hash_string(cur_name), FALSE, TRUE);
#endif /* CNX_TOOLS */

    /*  Indicate in the subspace whether the SOM that contains it has
     *  debug information. The obj_has_dbg flag is set while reading
     *  in the space information. 
     */
    
    misc->obj_has_dbg = obj_has_dbg;

    /* doom support */
    lm_subspace_add(index);
  
    /* Check for an alias of this subspace name */
    file_name = misc->f_name;
    mod_name = misc->module_name;

    /* Embedded systems code */
    if (GET_ALIAS_NAME_POSSIBLE && 
	(c=get_alias_name(subsp->NAME_PT, &file_name, &mod_name)) != NULL) {
	/* save off original names */
	misc->input_name = (char *) emalloc(
				    strlen(subsp->NAME_PT) * sizeof(char)+1);
	misc->input_file_name = (char *) emalloc(
				    strlen(misc->file_name) * sizeof(char)+1);
	strcpy(misc->input_name, subsp->NAME_PT);
	strcpy(misc->input_file_name, misc->file_name);
        subsp->NAME_PT = c;
        misc->file_name = cur_name;      /* file and obj in string */
        misc->f_name = file_name;        /* just file name */
        misc->module_name = mod_name;    
        misc->module_id = module_seq_num;
    } else {
	/* set input name to NULL if subsp name is original name */
	misc->input_name = NULL;
    }

    /* look for first instance of this subspace name */
    same = find_subspace(index);

    if (index == 0 || same < 0)
        /* this is the first:  add new subspace name to string table */
        subsp->NAME_PT = add_string(&space_strings, 
				    subsp->NAME_PT, 
				    hash_string(subsp->NAME_PT), 
				    FALSE, 
				    TRUE);
    else {
        /* copy previous subspace name and sort key */
        subsp2 = &Subsp_Dict(same);
        misc2 = &Subsp_Misc(same);
        subsp->NAME_PT = subsp2->NAME_PT;
        subsp->sort_key = subsp2->sort_key;
        misc->include_order = misc2->include_order;
        misc->subsp_order = misc2->subsp_order;

	/* change spaces */
	space_array[subsp->space_index].subspace_quantity--;
        subsp->space_index = subsp2->space_index;
	space_array[subsp->space_index].subspace_quantity++;

        if (misc2->force_new_align) {
            /* This subspace had an align qualifier in the command file */
	    /* Force the new alignment on the input subspaces */
	    if (subsp->alignment > subsp2->alignment) {
		printf(ld_gets(1, 
			       1210, 
			      "Warn - %s alignment forced to smaller value\n"),
		       subsp->name.n_name);
	    }
            subsp->alignment = subsp2->alignment;
	}

        misc->input_subsp_only = misc2->input_subsp_only;
    }

    /* space alignment will be the most restrictive subspace alignment */
    if (space_misc[subsp->space_index].space_align < subsp->alignment)
        space_misc[subsp->space_index].space_align = subsp->alignment;
	
    len = subsp->initialization_length;

    if (len > 0) {
	/* get subspace data */
	if (two_pass) {
	    if (len > subsp_data_area_max)
                subsp_data_area_max = len;
	    misc->subsp_data = cur_file_name;
	    misc->data_file_offset = subsp->file_loc_init_value + cur_ofs;
	} else if (save_data) {
	    if (len > subsp_data_area_max) {
		efree(subsp_data_area);
		subsp_data_area_max = len + len/4;
		subsp_data_area = (char *) emalloc(subsp_data_area_max);
	    }
	    ffetch(cur_fildes,
		   subsp->file_loc_init_value + cur_ofs,
		   subsp_data_area, 
		   sizeof(char),
		   len);
	    fdump(data_fildes,
		  data_file_offset,
	          subsp_data_area,
		  1,
		  len);
	    misc->data_file_offset = data_file_offset;
	    data_file_offset += len;
	    misc->subsp_data = data_file;
	} else {
	    misc->subsp_data = (char *) emalloc(len);
	    ffetch(cur_fildes,
		   subsp->file_loc_init_value + cur_ofs,
		   misc->subsp_data, 
		   sizeof(char), 
		   len);
	    misc->data_file_offset = IN_MEMORY;
	}
    }
} /* end subspace_add */

/*
*****************************************************************************
** Find the first instance of a subspace with the same name.  During som
** collection, the r_n_x field of the subspace misc record is used to link
** together the first instances of each subspace name
*****************************************************************************
*/

int find_subspace(index)
int index;
{
    register int index2;
    register struct subspace_dictionary_record *subsp;
    register struct subsp_misc_record *misc, *misc2;
    extern int filename_match();
    register int first_found;

    first_found = -1;      /* init */
    subsp = &Subsp_Dict(index);
    misc = &Subsp_Misc(index);
    for (index2 = 0; index2 >= 0; index2 = misc2->r_n_x) {
        misc2 = &Subsp_Misc(index2);
        if (strcmp(subsp->NAME_PT, Subsp_Dict(index2).NAME_PT) == 0) {
            if ((first_found == -1) && 
		     !(misc2->is_qualified || misc->is_qualified)) 
	        first_found = index2;   /* 1st name only match */

            /* only match filenames if either subspace name is qualified */
	    if ((misc2->is_qualified ||
	        misc->is_qualified) && 
			filename_match(index, index2))
	        return (index2);
	}
    }

    /* if a first was found then subsp name matches but filename doesn't */
    if (first_found != -1)
        return(first_found);

    /* not found:  append to list of first instances */
    misc2->r_n_x = index;
    return (-1);
} /* end find_subspace */

int find_subspace_by_name(subsp_name)
char *subsp_name;
{
    /* 
    ** searches through the subspace records looking for the first 
    ** subspace with the name "subsp_name". This routine is mainly 
    ** used to find subspaces to gain sort key information. 
    */

    register index;

    for (index = 0; index < subsp_dict_size; index++) 
        if (strcmp(subsp_name, Subsp_Dict(index).NAME_PT) == 0)
	    return (index);

    return(BAD_SUBSP);
} /* find_subspace_by_name */

void subspaces_sort()
{
    int space_index, prev_space_index, new_index, old_index;
    time_t rstart, rstop;
    struct tms pstart, pstop;

    extern void sort_by_key();

    if (verbose & V_TIMES) {
	printf(ld_gets(1, 1211, "Starting subspace sort...\n"));
	rstart = times(&pstart);
    }

    sort_by_key();

    /* Initialize new index remapping array. */
    for (old_index = 0; old_index < subsp_dict_size; old_index++) {
	new_index = Subsp_Misc(old_index).r_o_x;
	Subsp_Misc(new_index).r_n_x = old_index;
    }

    /* Update subspace index in space records. */
    prev_space_index = -1;
    for (new_index = 0; new_index < subsp_dict_size; new_index++) {
	old_index = Subsp_Misc(new_index).r_n_x;
	if (Subsp_Dict(old_index).space_index != prev_space_index) {
	    space_index = Subsp_Dict(old_index).space_index;
	    space_array[space_index].subspace_index = new_index;
	    prev_space_index = space_index;
	}
    }

/* 
** The Unified header files do not describe the units returned
** by the times() function other than to say that they are in
** hardware dependent units. Defining HZ to be 1000 will convert
** all our timing print statements on the Unified release to be
** in units of clock ticks.
*/

    if (verbose & V_TIMES) {
	rstop = times(&pstop);
	printf(ld_gets(1, 
		       1212, 
		       "End subspace sort (%d ms. real, %d ms. user)\n"),
	       (int) 1000*(rstop-rstart)/HZ,
	       (int) 1000*(pstop.tms_utime-pstart.tms_utime)/HZ);
    }

} /* end subspaces_sort */

void sort_by_key()
{
    int i;
    int *subptrs;
    extern int space_compare(), subspace_compare();

    /* sort the sp_remap array */
    for (i = 0; i < cur_space_total; i++)
        sp_remap[i] = i;
    qsort(sp_remap, cur_space_total, sizeof(int), space_compare);

     /*  - don't call fdp on -P -r link, as we just waste 
     * time executing fdp tool.  Wait until the final link to produce
     * the list of functions to be layed out
     */
     if (do_procedure_fdp && !relocatable)
        /* 
        ** if isom_exist (TRUE if isoms exist in object files and during 
        ** pre-link phase), do not call get_FDP_list. Only call this during
        ** the actual link after isoms are compiled
	*/
       if (!isom_exist)
           get_FDP_list();

    /* sort an array of subspace indices */
    subptrs = (int *) emalloc(sizeof(int) * subsp_dict_size);
    for (i = 0; i < subsp_dict_size; i++)
        subptrs[i] = i;
    qsort(subptrs, subsp_dict_size, sizeof(int), subspace_compare);

    /* set the remap index for each subspace */
    for (i = 0; i < subsp_dict_size; i++)
        Subsp_Misc(subptrs[i]).r_o_x = i;

    efree(subptrs);
} /* end sort_by_key */

/* reads in the FDP (feedback directed positioning) list that has the 
   form: 	optional_qualifier_name.func_name  
   example:	module_x.function 

   or, in the case of C where the qualifier name is a file name 
   used to differentiate local functions, and your file name contains a "."
   form: 	optional_qualifier.name:func_name
   example:     file.o:foo

   This function only called in FDP.  So we ifdef it. 
*/

int space_compare(sp1, sp2)
int *sp1, *sp2;
{
    int key1, key2;
    unsigned int ukey1, ukey2;

    key1 = space_misc[*sp1].space_specified;
    key2 = space_misc[*sp2].space_specified;

    /* 
    ** Sort by virt address if specified.  Note here that the 
    ** start_virt_address was set regardless of whether the virt_addr_specified
    ** bit is set. This is to allow spaces that had no virtual address but
    ** were defined after a space that has a virtual address to sort in the
    ** order seen. Since the virt_addr_specified bit is unset the space will
    ** not be tied to that address. 
    */

    if (key1 && key2) {
	ukey1 = space_misc[*sp1].start_virt_addr;
	ukey2 = space_misc[*sp2].start_virt_addr;
	if (ukey1 != ukey2) {
	    /* two unsigned values */
	    return( ((ukey1 > ukey2)? 1 : -1) );
	} else {
            /* both have same virt address, use order seen in cmd file */
            return(*sp1 - *sp2);
	}
    } else if (key1) 
	return(-1);          /* specified always takes precedence */
    else if (key2) 
	return(1);           /* specified always takes precedence */

    /* nothing was specified use normal sortkey */

    key1 = space_array[*sp1].sort_key;
    key2 = space_array[*sp2].sort_key;
    if (key1 == 0)
        key1 = 80;       /* this is for backwards */
    if (key2 == 0)       /*   compatibility       */
        key2 = 80;
    if (key1 != key2)
	return (key1 - key2);
    return (*sp1 - *sp2);
} /* end space_compare */

int subspace_compare(subp1, subp2)
int *subp1, *subp2;
{
    int sub_num1, sub_num2;
    int sp1, sp2;
    register struct subspace_dictionary_record *sub1, *sub2;
    int name1, name2;
    struct space_dictionary_record *space;
    int fru1, fru2;
    struct subsp_misc_record *misc1, *misc2;
    int i;

    sub_num1 = *subp1;
    sub_num2 = *subp2;

    sub1 = &Subsp_Dict(sub_num1);
    sub2 = &Subsp_Dict(sub_num2);
    sp1 = sp_remap[sub1->space_index];
    sp2 = sp_remap[sub2->space_index];
    if (sp1 != sp2)
	return (sp1 - sp2);

    misc1 = &Subsp_Misc(sub_num1);
    misc2 = &Subsp_Misc(sub_num2);

    /* Rearrange any embedded systems subspaces */
    i = misc1->subsp_order - misc2->subsp_order;
    if (i != 0)
        return (i);

    fru1 = (sub_num1 < fru_subsp_count);
    fru2 = (sub_num2 < fru_subsp_count);

    if (fru1 != fru2) {
        /* DOING A FRU RELINK, and one of the subspaces is from */
        /* the executable and the other subspace is from the    */
        /* relocatable                                          */

        /* normally, the executable one comes first...  but not */
        /* if one subspace is UNWIND/RECOVER and the other is   */
        /* a CODE subspace                                      */

        space = & space_array[sp1];
        if (space->is_loadable && !space->is_private) {
            /* only look at $TEXT$ */
            if ((sub1->sort_key >= UNWIND_SORT &&
                 sub2->sort_key < UNWIND_SORT) ||
                (sub2->sort_key >= UNWIND_SORT &&
                 sub1->sort_key < UNWIND_SORT))
        	return (sub1->sort_key - sub2->sort_key);
        }
        return (fru2 - fru1);
    }

    if (fru1) {
        /* know that the status of fru1 == fru2 */
        /* therefore, if we get here, then we   */
        /* are in a FRU relink                  */
        /* maintain original subspace order     */
        return (sub_num1 - sub_num2);
    }

    if (sub1->sort_key != sub2->sort_key)
	return (sub1->sort_key - sub2->sort_key);
    if (sub1->memory_resident != sub2->memory_resident)
	return (sub2->memory_resident - sub1->memory_resident);

    /* Changed FDP_file to fdp_link_order_file, FDP_CODE */
    if (fdp_link_order_file != NULL) {
        register int i;

        i = misc1->FDP_order_specified - misc2->FDP_order_specified;
        if (i != 0)
            return (i);
    }

    /* 
    ** sort by access rights and then initially frozen.  Check for
    ** wildcards on access rights 
    */

    /* group subspaces with the same access control properties together */
    if ((sub1->access_control_bits != sub2->access_control_bits) &&
          (sub1->access_control_bits != 0) &&
	  (sub2->access_control_bits != 0)) 
        return (sub2->access_control_bits - sub1->access_control_bits);

    /* separate subspaces that are initially frozen from those that aren't */
    if (sub1->initially_frozen != sub2->initially_frozen)
        return (sub2->initially_frozen - sub1->initially_frozen);

    /* 
    ** Sort subspaces by name - any predicate before this can make subspaces
    ** with the same name NOT sort together and hence not merge. 
    */
    if (space_strings.open_segment == 0) {
        /* 
        ** if only one space string segment in memory, they have to have been
        ** allocated in the same segment, so their string addresses directly
        ** give the order the subspace names were read in/created.
        ** Short-circuits 103,000 calls to "space_str_index" when building
	*/
	if (sub1->NAME_PT != sub2->NAME_PT)
	    return (sub1->NAME_PT - sub2->NAME_PT);  
    } else {
    	name1 = string_offset_linear(sub1->NAME_PT, &space_strings);
    	name2 = string_offset_linear(sub2->NAME_PT, &space_strings);
    	if (name1 != name2)
	    return (name1 - name2);
    }

    /* optimize the data references made by dreloc records in shlibs */
    if (building_shlib && 
	sub1->is_loadable && sub2->is_loadable &&
	(sub1->quadrant  == 1) && (sub2->quadrant == 1) &&
	(sub1->fixup_request_quantity != sub2->fixup_request_quantity)) {
	    Boolean sub1_drelocs, sub2_drelocs;  /* dreloc records likely? */

	    /* 
            ** heurstic to make data subspaces likely to have dreloc records
	    ** sort near each other in a shared library, to minimize pages
	    ** touched when fixing up dreloc recs at dynamic load time.
	    ** This is judged more maintainable than resorting after the
	    ** first look at the fixups tells us exactly which subspaces DO
	    ** have dreloc records.  
	    */

	    switch(sub1->fixup_request_quantity) {
		case 0:
		case 1:
		case 2:
		case 8:
	        sub1_drelocs = FALSE;  /* dreloc records unlikely. */
		break;
		default:
	        sub1_drelocs = TRUE;  /* dreloc records likely. */
		break;
	    }

	    switch(sub2->fixup_request_quantity) {
		case 0:
		case 1:
		case 2:
		case 8:
	        sub2_drelocs = FALSE;  /* dreloc records unlikely. */
		break;
		default:
	        sub2_drelocs = TRUE;  /* dreloc records likely. */
		break;
	    }
	    
	    if (sub2_drelocs != sub1_drelocs)
		return (sub1_drelocs ? 1 : -1);
    }

    if (sub1->code_only && sub2->code_only &&
	sub1->dup_common && !sub2->dup_common)
	return 1;
    else if  (sub1->code_only && sub2->code_only &&
	      !sub1->dup_common && sub2->dup_common)
	return -1;

    /* if all else is the same, sort them in input (subspace number) order */
    return (sub_num1 - sub_num2);
} /* end subspace_compare */

void extend_common(index,len)
int index, len;
{
    while (index > 0 && Subsp_Dict(index).continuation)
	index--;
    len -= Subsp_Dict(index).subspace_length;
    index++;
    while (index < subsp_dict_size && Subsp_Dict(index).continuation) {
	len -= Subsp_Dict(index).subspace_length;
	index++;
    }
    Subsp_Dict(index-1).subspace_length += len;
}

void delete_dup_common(index)
int index;
{
    while (index > 0 && Subsp_Dict(index).continuation)
	index--;
    Subsp_Dict(index).subspace_length = 0;
    Subsp_Dict(index).initialization_length = 0;
    /*
    **  If doing ld -r with dupe common symbols,
    ** do NOT output more than one subspace--only output the first
    ** one.  We will check the fields when determining if we need to output
    ** subspaces.
    */

    if (relocatable) {
        Subsp_Misc(index).symbol_count = 0; 
        Subsp_Dict(index).fixup_request_quantity = 0;
    }
    index++;
    while (index < subsp_dict_size && Subsp_Dict(index).continuation) {
	Subsp_Dict(index).subspace_length = 0;
	Subsp_Dict(index).initialization_length = 0;
        if (relocatable) {
            Subsp_Misc(index).symbol_count = 0; 
            Subsp_Dict(index).fixup_request_quantity = 0;
	}
	index++;
    }
} /* end delete_dup_common */

int get_common_length(index)
int index;
{
    int len;

    while (index > 0 && Subsp_Dict(index).continuation)
	index--;
    len = Subsp_Dict(index).subspace_length;
    index++;
    while (index < subsp_dict_size && Subsp_Dict(index).continuation) {
	len += Subsp_Dict(index).subspace_length;
	index++;
    }
    return (len);
} /* end get_common_length */

find_symbol_subspace(sym_value)
int sym_value;
{
    /* given the symbol value, what subspace does it lie in? for FRU */

    int index;
    struct subspace_dictionary_record *subsp;

    for (index = 0; index < next_subsp_index_bias; index++) {
        subsp = & Subsp_Dict(index);
        if (subsp->subspace_start <= sym_value &&
            sym_value < subsp->subspace_start+subsp->subspace_length)
            return (index);
    }

    internal_error(BAD_SYMBOL_ADDRESS, 0);
} /* end symbol_subspace */

/* PROC_ELIM */
void delete_subsp(subsp_index)
int subsp_index;
{

    struct symbol_dictionary_record *sym;
    struct subspace_dictionary_record *subsp;
    struct subsp_misc_record *misc;
    int sym_count, symbol_index;

        subsp = &Subsp_Dict(subsp_index);
        misc = &Subsp_Misc(subsp_index);

        symbol_index = misc->symbol_index_bias;
        sym_count = misc->symbol_count;

        /* zero out the subspace */
        subsp->subspace_length = 0;
        subsp->initialization_length = 0;
        subsp->fixup_request_index = -1;
        subsp->fixup_request_quantity = 0;
        misc->symbol_count = 0;

        while(sym_count && symbol_index < sym_dict_size) {
            if (Sym_Subsp(symbol_index) == subsp_index &&
                Sym_Scope(symbol_index) != SS_UNSAT) {
                Sym_Dict(symbol_index).symbol_type = ST_NULL;
                sym_count--;
            }
            symbol_index++;
        }
        if (sym_count)
            internal_error(WRNG_NUM_NON_NULL_SYMS, 0);
} /* delete_subsp */

void subprog_elim()
{

    char **list, *name;
    int ind;
    struct symbol_dictionary_record *sym;
    struct subspace_dictionary_record *subsp;
    struct subsp_misc_record *misc;
    int subsp_index;
    int sym_count, symbol_index;

    list = subprog_list;
    if (!list) 
	return;

    while ((name = *list++) != NULL) {
	ind = univ_find(name, 0, NULL, hash_string(name), ST_CODE);
    	if (ind == BAD_SYM) {
	    warning(CANT_ELIM_SPROG, name, 0);
	    continue;
        }

	subsp_index = Sym_Dict(ind).symbol_info;
	delete_subsp(subsp_index);
	subsp = &Subsp_Dict(subsp_index);
	misc = &Subsp_Misc(subsp_index);

	symbol_index = misc->symbol_index_bias;
	sym_count = misc->symbol_count;

	/* zero out the subspace */
	subsp->subspace_length = 0;
	subsp->initialization_length = 0;
	subsp->fixup_request_index = -1;
	subsp->fixup_request_quantity = 0;
	misc->symbol_count = 0;

	while(sym_count && symbol_index < sym_dict_size) {
            if (Sym_Subsp(symbol_index) == subsp_index &&
		Sym_Scope(symbol_index) != SS_UNSAT) {
		Sym_Dict(symbol_index).symbol_type = ST_NULL;
		sym_count--;
            }
	    symbol_index++;
	}
	if (sym_count) 
	    internal_error(WRNG_NUM_NON_NULL_SYMS, 0);
    }
    /* reclaim/reset for following use */
    efree(subprog_list);
    subprog_list = NULL;
    subprog_list_size = 0;

} /* end subprog_elim */

int subspace_match(subsp1, subsp2)
struct subspace_dictionary_record *subsp1, *subsp2;
{
    /* 
    ** Return true if attributes are the same.  Used from import_stub_needed
    */

    /* should locality be checked (strcmp on names?) */
    if (subsp1->space_index == subsp2->space_index &&
           subsp1->is_loadable == subsp2->is_loadable &&
           subsp1->access_control_bits == 
				 subsp2->access_control_bits &&
           subsp1->dup_common == subsp2->dup_common &&
           subsp1->is_common == subsp2->is_common &&
           subsp1->quadrant == subsp2->quadrant &&
#ifdef TSD /* TSD */
	   subsp1->is_tspecific == subsp2->is_tspecific &&
#endif /* TSD */
           subsp1->initially_frozen == subsp2->initially_frozen &&
           subsp1->memory_resident == subsp2->memory_resident) {

        /* If not memory resident check locality names */
        if (!subsp1->memory_resident) {
	    if (!same_locality(subsp1->name.n_name, subsp2->name.n_name))
		return(FALSE);         /* different localities */
	}

        return(TRUE);
    }

    /* If we get here, they do not match so... */
    return(FALSE);
} /* end subspace_match */

#ifdef TSD
/*
 * Get the subspace index of $TBSS$.  
 * This is called when an unsat is resolved by TLS shlib exported 
 * symbols and we don't know the subspace index of $TBSS$.
 */
int get_tbss_subsp_index()
{
   int tbss_subsp_index;
   int space_index;
   int i;
   struct space_dictionary_record *tspecific_sp;

   /* first check if there is a TBSS subspace appended in 
    * the subspace array.  If there is one, we use this subspace to
    * contain those TLS shlib exported symbols that are used to resolve
    * unsats and do data copying.
    */
   tbss_subsp_index = find_subspace_by_name("$TBSS$");

   if (tbss_subsp_index == BAD_SUBSP) {

      /* If not found, build a $TBSS$ subspace to contain TLS symbols
       * imported from a shared library when we do data copying for them.
       * (These unsats are resolved by TLS shared library exports.
       */

      /* We should never build a $THREAD_SPECIFIC$ space because 
       * if the unsat is TLS, this space should be added already;
       * otherwise, the unsat is non-TLS but resolved by a TLS symbol.
       * We quit if such a space does not exist.
       */
      space_index = -1;
      for (i = 0; i < cur_space_total; i++) {
	  if (strcmp(TSPECIFIC_SPACENAME, space_array[i].NAME_PT) == 0) {
	     space_index = i;
	     break;
	  }
      }
      if (space_index == -1) {
	 user_error(TSPECIFIC_SPACE_NOT_FOUND, TSPECIFIC_SPACENAME);
      }
      tbss_subsp_index = build_subspace("$TBSS$",
                                        TSPECIFIC_SPACENAME,
                                        TSPECIFIC_SORTKEY,
                                        0x1f,
                                        DATA_QUADRANT,
                                        SORT_TBSS,
                                        8, 0, 0);

      Subsp_Dict(tbss_subsp_index).is_tspecific = 1;

      assert(space_index == Subsp_Dict(tbss_subsp_index).space_index);

   }
   return tbss_subsp_index;
}

#endif /* TSD */

