/******************************************************************************
 *                  QLOGIC LINUX SOFTWARE
 *
 * QLogic ISP2x00 device driver for Linux 2.4.x
 * Copyright (C) 2002 Qlogic Corporation
 * (www.qlogic.com)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 ******************************************************************************/


#define	QLA_PT_CMD_TOV			(30 * 2)
#define QLA_IOCTL_ACCESS_WAIT_TIME	30 /* seconds */
#define QLA_INITIAL_IOCTLMEM_SIZE	(2 * PAGE_SIZE)

#define IOCTL_INVALID_STATUS    0xffff

#if defined(INTAPI)
#include "inioct.h"
/* from qla_inioct.c */
extern int qla2x00_read_nvram(scsi_qla_host_t *, EXT_IOCTL *, int);
extern int qla2x00_update_nvram(scsi_qla_host_t *, EXT_IOCTL *, int);
extern int qla2x00_write_nvram_word(scsi_qla_host_t *, uint8_t, uint16_t);
extern int qla2x00_send_loopback(scsi_qla_host_t *, EXT_IOCTL *, int);
extern int qla2x00_read_option_rom(scsi_qla_host_t *, EXT_IOCTL *, int);
extern int qla2x00_update_option_rom(scsi_qla_host_t *, EXT_IOCTL *, int);
#endif


STATIC int qla2x00_alloc_ioctl_mem(scsi_qla_host_t *);
STATIC int qla2x00_get_new_ioctl_dma_mem(scsi_qla_host_t *, uint32_t);
STATIC void qla2x00_free_ioctl_mem(scsi_qla_host_t *);

/*
 * Local prototypes
 */
STATIC int qla2x00_find_curr_ha(int, scsi_qla_host_t **);

STATIC int qla2x00_aen_reg(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_aen_get(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC void qla2x00_enqueue_aen(scsi_qla_host_t *, uint16_t, void *);

STATIC int qla2x00_query(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_hba_node(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_hba_port(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_disc_port(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_disc_tgt(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_chip(scsi_qla_host_t *, EXT_IOCTL *, int);

STATIC int qla2x00_get_data(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_get_statistics(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_get_fc_statistics(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_get_port_summary(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_driver(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_query_fw(scsi_qla_host_t *, EXT_IOCTL *, int);

STATIC int qla2x00_send_fcct(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_start_ms_cmd(scsi_qla_host_t *, EXT_IOCTL *, srb_t *);

STATIC int qla2x00_wwpn_to_scsiaddr(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_scsi_passthru(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_sc_scsi_passthru(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_sc_fc_scsi_passthru(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_sc_scsi3_passthru(scsi_qla_host_t *, EXT_IOCTL *, int);

STATIC int qla2x00_send_els_rnid(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_get_rnid_params(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_set_host_data(scsi_qla_host_t *, EXT_IOCTL *, int);
STATIC int qla2x00_set_rnid_params(scsi_qla_host_t *, EXT_IOCTL *, int);

STATIC void qla2x00_waitq_sem_timeout(unsigned long);
STATIC uint8_t qla2x00_get_ioctl_access(scsi_qla_host_t *, uint32_t);
STATIC uint8_t qla2x00_release_ioctl_access(scsi_qla_host_t *);

STATIC void qla2x00_wait_q_memb_alloc(scsi_qla_host_t *, wait_q_t **);
STATIC void qla2x00_wait_q_memb_free(scsi_qla_host_t *, wait_q_t *);
STATIC uint8_t qla2x00_wait_q_add(scsi_qla_host_t *, wait_q_t **);
STATIC void qla2x00_wait_q_remove(scsi_qla_host_t *, wait_q_t **);


/*
 * qla2x00_ioctl_sleep_done
 *
 * Description:
 *   This is the callback function to wakeup ioctl completion semaphore
 *   for the ioctl request that is waiting.
 *
 * Input:
 *   sem - pointer to the ioctl completion semaphore.
 *
 * Returns:
 */
STATIC void
qla2x00_ioctl_sleep_done(struct semaphore * sem)
{
	DEBUG9(printk("ioctl_sleep: entered.\n");)

	if (sem != NULL){
		DEBUG9(printk("ioctl_sleep: wake up sem.\n");)
		up(sem);
	}

	DEBUG9(printk("ioctl_sleep: exiting.\n");)
}

/*
 * qla2x00_ioctl_sem_init
 *
 * Description:
 *   Initialize the ioctl timer and semaphore used to wait for passthru
 *   completion.
 *
 * Input:
 *   ha - pointer to scsi_qla_host_t structure used for initialization.
 *
 * Returns:
 *   None.
 */
STATIC void
qla2x00_ioctl_sem_init(scsi_qla_host_t *ha)
{
	init_MUTEX_LOCKED(&ha->ioctl->cmpl_sem);
	init_timer(&(ha->ioctl->cmpl_timer));
	ha->ioctl->cmpl_timer.data = (unsigned long)&ha->ioctl->cmpl_sem;
	ha->ioctl->cmpl_timer.function =
	    (void (*)(unsigned long))qla2x00_ioctl_sleep_done;

}

/*
 * qla2x00_scsi_pt_done
 *
 * Description:
 *   Resets ioctl progress flag and wakes up the ioctl completion semaphore.
 *
 * Input:
 *   pscsi_cmd - pointer to the passthru Scsi cmd structure which has completed.
 *
 * Returns:
 */
STATIC void
qla2x00_scsi_pt_done(Scsi_Cmnd *pscsi_cmd)
{
	struct Scsi_Host *host;
	scsi_qla_host_t  *ha;

	host = pscsi_cmd->host;
	ha = (scsi_qla_host_t *) host->hostdata;

	DEBUG9(printk("qla2x00_scsi_pt_done post function called OK\n");)

	/* save detail status for IOCTL reporting */
	ha->ioctl->SCSIPT_InProgress = 0;
	ha->ioctl->ioctl_tov = 0;

	up(&ha->ioctl->cmpl_sem);

	DEBUG9(printk("qla2x00_scsi_pt_done: exiting.\n");)

	return;
}

/*
 * qla2x00_fcct_done
 *
 * Description:
 *   Resets FCCT ioctl progress flag and wakes up the ioctl completion
 *   semaphore.
 *
 * Input:
 *   cmd - pointer to the passthru Scsi cmd structure which has completed.
 *
 * Returns:
 */
STATIC void
qla2x00_fcct_done(Scsi_Cmnd *pscsi_cmd)
{
	struct Scsi_Host *host;
	scsi_qla_host_t  *ha;

	host = pscsi_cmd->host;
	ha = (scsi_qla_host_t *) host->hostdata;

	DEBUG9(printk("qla2x00_fcct_done post function called OK\n");)

	ha->ioctl->FCCT_InProgress = 0;
	ha->ioctl->ioctl_tov = 0;

	up(&ha->ioctl->cmpl_sem);

	DEBUG9(printk("qla2x00_fcct_done: exiting.\n");)
		
	return;
}

/*************************************************************************
 * qla2x00_ioctl
 *
 * Description:
 *   Performs additional ioctl requests not satisfied by the upper levels.
 *
 * Returns:
 *   ret  = 0    Success
 *   ret != 0    Failed; detailed status copied to EXT_IOCTL structure
 *               if applicable
 *************************************************************************/
int
qla2x00_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
	int		mode = 0;
	int		rval = 0;
	int		ret = EINVAL;

	uint8_t		*temp;
	uint8_t		tempbuf[8];
	uint32_t	i;
	uint32_t	status;

	static EXT_IOCTL	ext;
	PEXT_IOCTL		pext = &ext;

	scsi_qla_host_t		*ha;
	struct Scsi_Host	*host;


	DEBUG9(printk("qla2x00_ioctl: entry to command (%x), arg (%p)\n",
		cmd, arg);)

	/* Catch any non-exioct ioctls */
	if (_IOC_TYPE(cmd) != QLMULTIPATH_MAGIC) {
		return (-EINVAL);
	}

	host = dev->host;
	ha = (scsi_qla_host_t *) host->hostdata; /* midlayer chosen instance */

	ret = verify_area(VERIFY_READ, (void *)arg, sizeof(EXT_IOCTL));
	if (ret) {
		DEBUG9_10(printk("qla2x00_ioctl: ERROR VERIFY_READ EXT_IOCTL "
		    "sturct. ha=%p.\n", ha);)
		return ret;
	}

	/* copy in application layer EXT_IOCTL */
	ret = copy_from_user(pext, arg, sizeof(EXT_IOCTL));
	if (ret) {
		DEBUG9_10(printk("qla2x00_ioctl: ERROR COPY_FROM_USER "
				       "EXT_IOCTL sturct. ha=%p.\n",
				       ha);)
		return ret;
	}

	/* check signature of this ioctl */
	temp = (uint8_t *) &pext->Signature;

	for (i = 0; i < 4; i++, temp++)
		tempbuf[i] = *temp;

	if ((tempbuf[0] == 'Q') && (tempbuf[1] == 'L') &&
	    (tempbuf[2] == 'O') && (tempbuf[3] == 'G'))
		status = 0;
	else
		status = 1;

	if (status != 0) {
		DEBUG9_10(printk("qla2x00_ioctl: signature did not match. "
		    "ha=%p\n", ha);)
		pext->Status = EXT_STATUS_INVALID_PARAM;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));
		return EXT_STATUS_ERR;
	}

	/* check version of this ioctl */
	if (pext->Version > EXT_VERSION) {
		printk(KERN_WARNING
		    "qla2x00: interface version not supported = %d.\n",
				pext->Version);
		pext->Status = EXT_STATUS_UNSUPPORTED_VERSION;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));
		return EXT_STATUS_ERR;
	}

	/* check for API setting HBA Instance for subsequent operations */
	if (cmd == (int)EXT_CC_STARTIOCTL) {
		DEBUG9(printk("qla2x00_ioctl: got startioctl command.\n");)

		pext->Instance = num_hosts;
		pext->Status = EXT_STATUS_OK;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));
		return EXT_STATUS_OK;

	} else if (cmd == (int)EXT_CC_SETINSTANCE) {
		/*
		 * Since API opens devices once and uses handle for
		 * subsequent calls, we keep a parameter to designate
		 * the "active HBA" for ioctls.
		 */
		if (pext->Instance < num_hosts) {
			apiHBAInstance = pext->Instance;
			/*
			 * Return host number in pext->HbaSelect for
			 * reference by IOCTL caller.
			 */
			if (qla2x00_find_curr_ha(apiHBAInstance, &ha) != 0) {
				DEBUG9_10(printk("qla2x00_ioctl: ERROR finding "
				    "ha in EXT_SETINSTANCE. Instance=%d "
				    "num_hosts=%d ha=%p.\n",
				    pext->Instance, num_hosts, ha);)
				pext->Status = EXT_STATUS_DEV_NOT_FOUND;
				copy_to_user(arg, pext, sizeof(EXT_IOCTL));

				return EXT_STATUS_ERR;
			}

			pext->HbaSelect = ha->host_no;
			pext->Status = EXT_STATUS_OK;
			copy_to_user(arg, pext, sizeof(EXT_IOCTL));

			DEBUG9(printk("qla2x00_ioctl: Setting instance to "
			    "%d.\n", apiHBAInstance);)

			ret = EXT_STATUS_OK;
		} else {
			DEBUG9_10(printk("qla2x00_ioctl: ERROR EXT_SETINSTANCE."
			    " Instance=%d num_hosts=%d ha=%p.\n",
			    pext->Instance, num_hosts, ha);)

			pext->Status = EXT_STATUS_DEV_NOT_FOUND;
			copy_to_user(arg, pext, sizeof(EXT_IOCTL));
			ret = EXT_STATUS_ERR;
		}

		return ret;
	}

	/*
	 * Check for valid apiHBAInstance (set previously by EXT_SETINSTANCE
	 * or default 0)  and set ha context for this IOCTL.
	 */
	if (qla2x00_find_curr_ha(apiHBAInstance, &ha) != 0) {

		DEBUG9_10(printk("qla2x00_ioctl: ERROR matching apiHBAInstance "
		    "%d to an HBA Instance.\n", apiHBAInstance);)

		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		copy_to_user(arg, pext, sizeof(EXT_IOCTL));
		return EXT_STATUS_ERR;
	}

	/*
	 * Get permission to process ioctl command. Only one will proceed
	 * at a time.
	 */
	if (qla2x00_get_ioctl_access(ha, QLA_IOCTL_ACCESS_WAIT_TIME) != 0) {
		/* error timed out */
		DEBUG9_10(printk("qla2x00_ioctl: ERROR timeout getting "
		    "ioctl access. Inst=%d.\n", apiHBAInstance);)

		pext->Status = EXT_STATUS_BUSY;
		copy_to_user(arg, pext, sizeof(EXT_IOCTL));
		return EXT_STATUS_ERR;
	}

	/*
	 * Set EXT_IOCTL.HbaSelect to host number for reference by IOCTL
	 * caller.
	 */
	pext->HbaSelect = ha->host_no;

	DEBUG9(printk("qla2x00_ioctl: active apiHBAInstance=%d CC=%x SC=%x.\n",
	    apiHBAInstance, cmd, pext->SubCode);)


	while( test_bit(CFG_ACTIVE, &ha->cfg_flags) || ha->dpc_active ) {
		if( signal_pending(current) )
			break;   /* get out */

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ);
	};

	switch (cmd) { /* switch on EXT IOCTL COMMAND CODE */

	case EXT_CC_QUERY:
		DEBUG9(printk("qla2x00_ioctl: got query command.\n");)

		rval = qla2x00_query(ha, pext, 0);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_GET_DATA:
		DEBUG9(printk("qla2x00_ioctl: got get_data command.\n");)

		rval = qla2x00_get_data(ha, pext, 0);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_SEND_FCCT_PASSTHRU:
		DEBUG9(printk("qla2x00_ioctl: got CT passthru cmd.\n"));

		rval = qla2x00_send_fcct(ha, pext, 0);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_SEND_SCSI_PASSTHRU:
		DEBUG9(printk("qla2x00_ioctl: got SCSI passthru cmd.\n"));

		rval = qla2x00_scsi_passthru(ha, pext, mode);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_REG_AEN:
		rval = qla2x00_aen_reg(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_GET_AEN:
		rval = qla2x00_aen_get(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case EXT_CC_WWPN_TO_SCSIADDR:
		rval = qla2x00_wwpn_to_scsiaddr(ha, pext, 0);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user(arg, pext, sizeof(EXT_IOCTL));
		break;

	case EXT_CC_SEND_ELS_RNID:
		rval = qla2x00_send_els_rnid(ha, pext, mode);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user(arg, pext, sizeof(EXT_IOCTL));
		break;

	case EXT_CC_SET_DATA:
		rval = qla2x00_set_host_data(ha, pext, mode);
		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user(arg, pext, sizeof(EXT_IOCTL));
		break;                                                          

#if defined(INTAPI)
	case INT_CC_READ_NVRAM:
		rval = qla2x00_read_nvram(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case INT_CC_UPDATE_NVRAM:
		rval = qla2x00_update_nvram(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case INT_CC_LOOPBACK:
		rval = qla2x00_send_loopback(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case INT_CC_READ_OPTION_ROM:
		rval = qla2x00_read_option_rom(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;

	case INT_CC_UPDATE_OPTION_ROM:
		rval = qla2x00_update_option_rom(ha, pext, mode);

		ret = (rval == QL_STATUS_SUCCESS) ?
		    EXT_STATUS_OK : EXT_STATUS_ERR;
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));

		break;
#endif /* INTAPI */

	/* all others go here */
	/*
	   case EXT_CC_ELS_RNID_SEND:
	   break;
	   case EXT_CC_ELS_RTIN_SEND:
	   break;
	   case EXT_CC_PLATFORM_REG:
	   break;
	 */

	/* Failover IOCTLs */
	case FO_CC_GET_PARAMS:
	case FO_CC_SET_PARAMS:
	case FO_CC_GET_PATHS:
	case FO_CC_SET_CURRENT_PATH:
	case FO_CC_RESET_HBA_STAT:
	case FO_CC_GET_HBA_STAT:
	case FO_CC_GET_LUN_DATA:
	case FO_CC_SET_LUN_DATA:
	case FO_CC_GET_TARGET_DATA:
	case FO_CC_SET_TARGET_DATA:
		DEBUG9(printk("qla2x00_ioctl: failover arg (%p):\n", arg);)
		    qla2x00_fo_ioctl(ha,cmd, arg, mode);
		copy_to_user((void *)arg, (void *)pext, sizeof(EXT_IOCTL));
		break;

	default:
		ret = EXT_STATUS_ERR;
		break;
	} /* end of CC decode switch */

	DEBUG9(printk("qla2x00_ioctl: exiting. rval(%d) ret(%d)\n", rval, ret);)

	qla2x00_release_ioctl_access(ha);

	return ret;
}

/*
 * qla2x00_alloc_ioctl_mem
 *	Allocates memory needed by IOCTL code.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla2x00 local function return status code.
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_alloc_ioctl_mem(scsi_qla_host_t *ha)
{
	DEBUG9(printk("qla2x00_alloc_ioctl_mem entered.\n");)

	if (qla2x00_get_new_ioctl_dma_mem(ha, QLA_INITIAL_IOCTLMEM_SIZE) !=
	    QL_STATUS_SUCCESS) {
		printk(KERN_WARNING
		    "qla2x00: ERROR in ioctl physical memory allocation\n");

		return QL_STATUS_RESOURCE_ERROR;
	}

	/* Allocate context memory buffer */
	ha->ioctl = KMEM_ZALLOC(sizeof(hba_ioctl_context), 11);
	if (ha->ioctl == NULL) {
		/* error */
		printk(KERN_WARNING
		    "qla2x00: ERROR in ioctl context allocation.\n");
		return QL_STATUS_RESOURCE_ERROR;
	}

	/* Allocate AEN tracking buffer */
	ha->ioctl->aen_tracking_queue =
	    KMEM_ZALLOC(EXT_DEF_MAX_AEN_QUEUE * sizeof(EXT_ASYNC_EVENT), 12);
	if (ha->ioctl->aen_tracking_queue == NULL) {
		printk(KERN_WARNING
		    "qla2x00: ERROR in ioctl aen_queue allocation.\n");
		return QL_STATUS_RESOURCE_ERROR;
	}

	ha->ioctl->ioctl_tq = KMEM_ZALLOC(sizeof(os_tgt_t), 13);
	if (ha->ioctl->ioctl_tq == NULL) {
		printk(KERN_WARNING
		    "qla2x00: ERROR in ioctl tgt queue allocation.\n");
		return QL_STATUS_RESOURCE_ERROR;
	}

	ha->ioctl->ioctl_lq = KMEM_ZALLOC(sizeof(os_lun_t), 14);
	if (ha->ioctl->ioctl_lq == NULL) {
		printk(KERN_WARNING
		    "qla2x00: ERROR in ioctl lun queue allocation.\n");
		return QL_STATUS_RESOURCE_ERROR;
	}
	/*INIT_LIST_HEAD(&(ha->ioctl->ioctl_lq->cmd));*/
	ha->ioctl->ioctl_lq->q_state = LUN_STATE_READY;
#ifdef __VMWARE__
    spin_lock_init(&ha->ioctl->ioctl_lq->q_lock);
#else
	ha->ioctl->ioctl_lq->q_lock = SPIN_LOCK_UNLOCKED;
#endif

	/* Init wait_q fields */
#ifdef __VMWARE__
    spin_lock_init(&ha->ioctl->wait_q_lock);
#else
	ha->ioctl->wait_q_lock = SPIN_LOCK_UNLOCKED;
#endif

	DEBUG9(printk("qla2x00_alloc_ioctl_mem exiting.\n");)

	return QLA2X00_SUCCESS;
}

/*
 * qla2x00_get_new_ioctl_dma_mem
 *	Allocates dma memory of the specified size.
 *	This is done to replace any previously allocated ioctl dma buffer.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla2x00 local function return status code.
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_get_new_ioctl_dma_mem(scsi_qla_host_t *ha, uint32_t size)
{
	if (ha->ioctl_mem) {
		DEBUG9(printk("qla2x00_get_new_ioctl_dma_mem: ioctl_mem "
		    "was previously allocated. Dealloc old buffer.\n");)

	 	/* free the memory first */
	 	pci_free_consistent(ha->pdev, ha->ioctl_mem_size, ha->ioctl_mem,
		    ha->ioctl_mem_phys);
	}

	/* Get consistent memory allocated for ioctl I/O operations. */
	ha->ioctl_mem = pci_alloc_consistent(ha->pdev,
	    size, &ha->ioctl_mem_phys);

	if (ha->ioctl_mem == NULL) {
		printk(KERN_WARNING
		    "qla2x00_get_new_ioctl_dma_mem: ERROR in ioctl physical "
		    "memory allocation. Requested length=%x.\n", size);

		ha->ioctl_mem_size = 0;
		return QL_STATUS_RESOURCE_ERROR;
	}
	ha->ioctl_mem_size = size;

	return QL_STATUS_SUCCESS;
}

/*
 * qla2x00_free_ioctl_mem
 *	Frees memory used by IOCTL code for the specified ha.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Context:
 *	Kernel context.
 */
STATIC void
qla2x00_free_ioctl_mem(scsi_qla_host_t *ha)
{
	DEBUG9(printk("qla2x00_free_ioctl_mem entered.\n");)

	if (ha->ioctl != NULL) {

		if (ha->ioctl->ioctl_tq != NULL) {
			KMEM_FREE(ha->ioctl->ioctl_tq, sizeof(os_tgt_t));
			ha->ioctl->ioctl_tq = NULL;
		}

		if (ha->ioctl->ioctl_lq != NULL) {
#ifdef __VMWARE__
			spin_lock_destroy(&ha->ioctl->ioctl_lq->q_lock);
			spin_lock_destroy(&ha->ioctl->wait_q_lock);
#endif
			KMEM_FREE(ha->ioctl->ioctl_lq, sizeof(os_lun_t));
			ha->ioctl->ioctl_lq = NULL;
		}

		if (ha->ioctl->aen_tracking_queue != NULL) {
			KMEM_FREE(ha->ioctl->aen_tracking_queue,
			    EXT_DEF_MAX_AEN_QUEUE * sizeof(EXT_ASYNC_EVENT));
			ha->ioctl->aen_tracking_queue = NULL;
		}

		KMEM_FREE(ha->ioctl, sizeof(hba_ioctl_context));
		ha->ioctl = NULL;
	}

	/* free memory allocated for ioctl operations */
	pci_free_consistent(ha->pdev, ha->ioctl_mem_size, ha->ioctl_mem,
	    ha->ioctl_mem_phys);
	ha->ioctl_mem = NULL;

	DEBUG9(printk("qla2x00_free_ioctl_mem exiting.\n");)
}

/*
 * qla2x00_find_curr_ha
 *	Searches and returns the pointer to the adapter instance specified.
 *
 * Input:
 *	inst = adapter instance number to search.
 *	ha = adapter state pointer of the instance requested.
 *
 * Returns:
 *	qla2x00 local function return status code.
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_find_curr_ha(int inst, scsi_qla_host_t **ret_ha)
{
	int	rval = QL_STATUS_SUCCESS;
	scsi_qla_host_t *search_ha = NULL;

	/*
	 * Check for valid apiHBAInstance (set previously by EXT_SETINSTANCE 
	 * or default 0)  and set ha context for this IOCTL
	 */
	for (search_ha = qla2x00_hostlist;
	    (search_ha != NULL) && search_ha->instance != inst;
	    search_ha = search_ha->next)
		continue;

	if (search_ha == NULL) {
		DEBUG10(printk("qla2x00_ioctl: ERROR matching apiHBAInstance "
		    "%d to an HBA Instance.\n", apiHBAInstance);)
		rval = QL_STATUS_ERROR;
	} else {
		*ret_ha = search_ha;
	}

	return rval;
}

/*
 * qla2x00_aen_reg
 *	IOCTL management server Asynchronous Event Tracking Enable/Disable.
 *
 * Input:
 *	ha = pointer to the adapter struct of the adapter to register.
 *	cmd = pointer to EXT_IOCTL structure containing values from user.
 *	mode = flags. not used.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_aen_reg(scsi_qla_host_t *ha, EXT_IOCTL *cmd, int mode)
{
	int		rval = 0;
	EXT_REG_AEN	reg_struct;

	DEBUG9(printk("qla2x00_aen_reg: entered.\n");)

	rval = copy_from_user(&reg_struct, cmd->RequestAdr, cmd->RequestLen);
	if (rval == 0) {
		cmd->Status = EXT_STATUS_OK;
		if (reg_struct.Enable) {
			ha->ioctl->flags |= IOCTL_AEN_TRACKING_ENABLE;
		} else {
			ha->ioctl->flags &= ~IOCTL_AEN_TRACKING_ENABLE;
		}
	} else {
		cmd->Status = EXT_STATUS_COPY_ERR;
		rval = EFAULT;
	}
	DEBUG9(printk("qla2x00_aen_reg: reg_struct. Enable(%d) "
	    "ha->ioctl_flag(%x) cmd->Status(%d) cmd->DetailStatus (%d).",
	    reg_struct.Enable, ha->ioctl->flags, cmd->Status,
	    cmd->DetailStatus);)

	DEBUG9(printk("qla2x00_aen_reg: exiting.\n");)

	return rval;
}

/*
 * qla2x00_aen_get
 *	Asynchronous Event Record Transfer to user.
 *	The entire queue will be emptied and transferred back.
 *
 * Input:
 *	ha = pointer to the adapter struct of the specified adapter.
 *	pext = pointer to EXT_IOCTL structure containing values from user.
 *	mode = flags.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 *
 * NOTE: Need to use hardware lock to protect the queues from updates
 *	 via isr/enqueue_aen after we get rid of io_request_lock.
 */
STATIC int
qla2x00_aen_get(scsi_qla_host_t *ha, EXT_IOCTL *cmd, int mode)
{
	int		rval = 0;
	EXT_ASYNC_EVENT	*tmp_q;
	static  EXT_ASYNC_EVENT	aen[EXT_DEF_MAX_AEN_QUEUE];
	uint8_t		i;
	uint8_t		queue_cnt;
	uint8_t		request_cnt;
	uint32_t	stat = EXT_STATUS_OK;
	uint32_t	dstat = EXT_STATUS_OK;
	uint32_t	ret_len = 0;
	unsigned long   cpu_flags = 0;

	DEBUG9(printk("qla2x00_aen_get: entered.\n");)

	request_cnt = (uint8_t)(cmd->ResponseLen / sizeof(EXT_ASYNC_EVENT));

	if (request_cnt < EXT_DEF_MAX_AEN_QUEUE) {
		/* We require caller to alloc for the maximum request count */
		cmd->Status       = EXT_STATUS_BUFFER_TOO_SMALL;
		rval = 0;
		DEBUG9_10(printk(
		    "qla2x00_aen_get: Buffer too small. Exiting normally.");)
		return rval;
	}

	/* 1st: Make a local copy of the entire queue content. */
	tmp_q = (EXT_ASYNC_EVENT *)ha->ioctl->aen_tracking_queue;
	queue_cnt = 0;

	spin_lock_irqsave(&ha->hardware_lock, cpu_flags);
	i = ha->ioctl->aen_q_head;

	for (; queue_cnt < EXT_DEF_MAX_AEN_QUEUE;) {
		if (tmp_q[i].AsyncEventCode != 0) {
			memcpy(&aen[queue_cnt], &tmp_q[i],
					sizeof(EXT_ASYNC_EVENT));
			queue_cnt++;
			tmp_q[i].AsyncEventCode = 0; /* empty out the slot */
		}

		if (i == ha->ioctl->aen_q_tail) {
			/* done. */
			break;
		}

		i++;

		if (i == EXT_DEF_MAX_AEN_QUEUE) {
			i = 0;
		}
	}

	/* Empty the queue. */
	ha->ioctl->aen_q_head = 0;
	ha->ioctl->aen_q_tail = 0;

	spin_unlock_irqrestore(&ha->hardware_lock, cpu_flags);

	/* 2nd: Now transfer the queue content to user buffer */
	/* Copy the entire queue to user's buffer. */
	ret_len = (uint32_t)(queue_cnt * sizeof(EXT_ASYNC_EVENT));
	if (queue_cnt != 0) {
		copy_to_user(cmd->ResponseAdr, aen, ret_len);
	}
	cmd->ResponseLen = ret_len;

	if (rval != 0) {
		stat = EXT_STATUS_COPY_ERR;
		rval = EFAULT;
		DEBUG10(printk("qla2x00_aen_get: FAILED. error = %d\n", stat);)
	} else {
		stat = EXT_STATUS_OK;
		rval = 0;
		DEBUG9(printk("qla2x00_aen_get: exiting normally.\n");)
	}

	cmd->Status       = stat;
	cmd->DetailStatus = dstat;

	DEBUG9(printk("qla2x00_aen_get: exiting. rval= %d\n", rval);)

	return rval;
}

/*
 * qla2x00_enqueue_aen
 *
 * Input:
 *	ha = adapter state pointer.
 *	event_code = async event code of the event to add to queue.
 *	payload = event payload for the queue.
 *
 * Context:
 *	Interrupt context.
 * NOTE: Need to hold the hardware lock to protect the queues from
 *	 aen_get after we get rid of the io_request_lock.
 */
void
qla2x00_enqueue_aen(scsi_qla_host_t *ha, uint16_t event_code, void *payload)
{
	uint8_t			new_entry; /* index to current entry */
	uint16_t		*mbx;
	EXT_ASYNC_EVENT		*aen_queue;

	DEBUG9(printk("qla2x00_enqueue_aen: entered.\n");)

	aen_queue = (EXT_ASYNC_EVENT *)ha->ioctl->aen_tracking_queue;
	if (aen_queue[ha->ioctl->aen_q_tail].AsyncEventCode != 0) {
		/* Need to change queue pointers to make room. */

		/* Increment tail for adding new entry. */
		ha->ioctl->aen_q_tail++;
		if (ha->ioctl->aen_q_tail == EXT_DEF_MAX_AEN_QUEUE) {
			ha->ioctl->aen_q_tail = 0;
		}

		if (ha->ioctl->aen_q_head == ha->ioctl->aen_q_tail) {
			/*
			 * We're overwriting the oldest entry, so need to
			 * update the head pointer.
			 */
			ha->ioctl->aen_q_head++;
			if (ha->ioctl->aen_q_head == EXT_DEF_MAX_AEN_QUEUE) {
				ha->ioctl->aen_q_head = 0;
			}
		}
	}

	DEBUG(printk("qla2x00_enqueue_aen: Adding code 0x%x to aen_q %p @ %d\n",
	    event_code, aen_queue, ha->ioctl->aen_q_tail);)
	new_entry = ha->ioctl->aen_q_tail;
	aen_queue[new_entry].AsyncEventCode = event_code;

	DEBUG(printk("qla2x00_enqueue_aen: Adding code 0x%8x\n",
	    aen_queue[new_entry].AsyncEventCode);)

		/* Update payload */
	switch (event_code) {
	case MBA_LIP_OCCURRED:
	case MBA_LOOP_UP:
	case MBA_LOOP_DOWN:
	case MBA_LIP_RESET:
	case MBA_PORT_UPDATE:
		/* empty */
		break;

	case MBA_SCR_UPDATE:
		mbx = (uint16_t *)payload;
		/* al_pa */
		aen_queue[new_entry].Payload.RSCN.RSCNInfo[0] = LSB(mbx[2]);
		/* area */
		aen_queue[new_entry].Payload.RSCN.RSCNInfo[1] = MSB(mbx[2]);
		/* domain */
		aen_queue[new_entry].Payload.RSCN.RSCNInfo[2] = LSB(mbx[1]);
		/* save in big endian */
		BIG_ENDIAN_24(aen_queue[new_entry].Payload.RSCN.RSCNInfo[0]);

		aen_queue[new_entry].Payload.RSCN.AddrFormat = MSB(mbx[1]);

		break;

	default:
		/* Not supported */
		aen_queue[new_entry].AsyncEventCode = 0;
		break;
	}

	DEBUG9(printk("qla2x00_enqueue_aen: exiting.\n");)
}

STATIC int
qla2x00_query(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int rval;

	DEBUG9(printk("qla2x00_query: entered.\n");)

	/* All Query type ioctls are done here */
	switch(pext->SubCode) {

	case EXT_SC_QUERY_HBA_NODE:
		/* fill in HBA NODE Information */
		rval = qla2x00_query_hba_node(ha, pext, mode);
		break;

	case EXT_SC_QUERY_HBA_PORT:
		/* return HBA PORT related info */
		rval = qla2x00_query_hba_port(ha, pext, mode);
		break;

	case EXT_SC_QUERY_DISC_PORT:
		/* return discovered port information */
		rval = qla2x00_query_disc_port(ha, pext, mode);
		break;

	case EXT_SC_QUERY_DISC_TGT:
		/* printk("[Start SC_QUERY_DISC_TGT active ha=%x]\n",ha); */
		rval = qla2x00_query_disc_tgt(ha, pext, mode);
		break;

	case EXT_SC_QUERY_CHIP:
		rval = qla2x00_query_chip(ha, pext, mode);
		break;

	case EXT_SC_QUERY_DISC_LUN:
		pext->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		rval = pext->Status;
		break;

	default:
		DEBUG10(printk("qla2x00_query: unknown SubCode %d.\n",
		    pext->SubCode);)
		pext->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		rval = pext->Status;
		break;
	}

	DEBUG9(printk("qla2x00_query: exiting.\n");)
	return rval;
}

STATIC int
qla2x00_query_hba_node(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EXT_STATUS_OK;
	uint32_t	i, transfer_size;
	static  EXT_HBA_NODE	tmp_hba_node;
	qla_boards_t	*bdp;

	DEBUG9(printk("qla2x00_query_hba_node: entered.\n");)

	memset(&tmp_hba_node, 0, sizeof(EXT_HBA_NODE));

	/* fill all available HBA NODE Information */
	bdp = &QLBoardTbl_fc[ha->devnum];
	for (i = 0; i < 8 ; i++)
		tmp_hba_node.WWNN[i] = ha->node_name[i];

	sprintf((char *)(tmp_hba_node.Manufacturer),"Qlogic Corp.");
	sprintf((char *)(tmp_hba_node.Model),(char *)&bdp->bdName[0]);

	tmp_hba_node.SerialNum[0] = ha->serial0;
	tmp_hba_node.SerialNum[1] = ha->serial1;
	tmp_hba_node.SerialNum[2] = ha->serial2;
	sprintf((char *)(tmp_hba_node.DriverVersion),QLA2100_VERSION);
	sprintf((char *)(tmp_hba_node.FWVersion),"%2d.%02d.%02d",
	    bdp->fwver[0], bdp->fwver[1], bdp->fwver[2]);

	sprintf((char *)(tmp_hba_node.OptRomVersion),"%d.%d",
	    ha->optrom_major, ha->optrom_minor);

	tmp_hba_node.InterfaceType = EXT_DEF_FC_INTF_TYPE;
	tmp_hba_node.PortCount = 1;


	tmp_hba_node.DriverAttr = (ha->flags.failover_enabled) ?
	    DRVR_FO_ENABLED : 0;

	ret = verify_area(VERIFY_WRITE, (void  *)pext->ResponseAdr,
	    sizeof(EXT_HBA_NODE));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_query_hba_node: ERROR verify write "
		    "rsp buffer.\n");)
		return pext->Status;
	}

	/* now copy up the HBA_NODE to user */
	if (pext->ResponseLen < sizeof(EXT_HBA_NODE))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_HBA_NODE);

	copy_to_user((uint8_t *)pext->ResponseAdr, (uint8_t *)&tmp_hba_node,
	    transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_hba_node: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_query_hba_port(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int	ret = EXT_STATUS_OK;
	uint32_t	tgt_cnt, tgt, transfer_size;
	uint32_t	port_cnt;
	fc_port_t	*fcport;
	EXT_HBA_PORT	tmp_hba_port;

	DEBUG9(printk("qla2x00_query_hba_port: entered.\n");)

	memset(&tmp_hba_port, 0, sizeof(EXT_HBA_PORT));

	/* reflect all HBA PORT related info */
	tmp_hba_port.WWPN[7] = ha->init_cb->port_name[7];
	tmp_hba_port.WWPN[6] = ha->init_cb->port_name[6];
	tmp_hba_port.WWPN[5] = ha->init_cb->port_name[5];
	tmp_hba_port.WWPN[4] = ha->init_cb->port_name[4];
	tmp_hba_port.WWPN[3] = ha->init_cb->port_name[3];
	tmp_hba_port.WWPN[2] = ha->init_cb->port_name[2];
	tmp_hba_port.WWPN[1] = ha->init_cb->port_name[1];
	tmp_hba_port.WWPN[0] = ha->init_cb->port_name[0];
	tmp_hba_port.Id[0] = 0;
	tmp_hba_port.Id[1] = ha->d_id.r.d_id[2];
	tmp_hba_port.Id[2] = ha->d_id.r.d_id[1];
	tmp_hba_port.Id[3] = ha->d_id.r.d_id[0];
	tmp_hba_port.Type =  EXT_DEF_INITIATOR_DEV;

	switch (ha->current_topology) {
	case ISP_CFG_NL:
	case ISP_CFG_FL:
		tmp_hba_port.Mode = EXT_DEF_LOOP_MODE;
		break;

	case ISP_CFG_N:
	case ISP_CFG_F:
		tmp_hba_port.Mode = EXT_DEF_P2P_MODE;
		break;

	default:
		tmp_hba_port.Mode = EXT_DEF_UNKNOWN_MODE;
		break;
	}

	port_cnt = 0;
	for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
		/* if removed or missing */
		if (atomic_read(&fcport->state) != FC_ONLINE) {
			DEBUG9(printk("query_hba_port: port %06x not online.\n",
			    fcport->d_id.b24);)
			continue;
		}
		port_cnt++;
	}

	tgt_cnt  = 0;
	for (tgt = 0; tgt < MAX_TARGETS; tgt++) {
		if (ha->otgt[tgt] == NULL) {
			continue;
		}
		tgt_cnt++;
	}

	tmp_hba_port.DiscPortCount   = port_cnt;
	tmp_hba_port.DiscTargetCount = tgt_cnt;

	if (ha->loop_state == LOOP_DOWN) {

		tmp_hba_port.State = EXT_DEF_HBA_LOOP_DOWN;

	} else if (ha->loop_state != LOOP_READY ||
	    test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags) ||
	    test_bit(CFG_ACTIVE, &ha->cfg_flags) || ABORTS_ACTIVE) {

		tmp_hba_port.State = EXT_DEF_HBA_SUSPENDED;

	} else {

		tmp_hba_port.State = EXT_DEF_HBA_OK;

	}

	tmp_hba_port.DiscPortNameType = EXT_DEF_USE_PORT_NAME;

	/* Return supported FC4 type depending on driver support. */
	tmp_hba_port.PortSupportedFC4Types = EXT_DEF_FC4_TYPE_SCSI;
#if defined(FC_IP_SUPPORT)

	tmp_hba_port.PortSupportedFC4Types |= EXT_DEF_FC4_TYPE_IP;
#endif
#if defined(FC_SCTP_SUPPORT)

	tmp_hba_port.PortSupportedFC4Types |= EXT_DEF_FC4_TYPE_SCTP;
#endif

	tmp_hba_port.PortActiveFC4Types = ha->active_fc4_types;

	/* Return supported speed depending on adapter type */
#if defined(ISP2100)

	tmp_hba_port.PortSupportedSpeed = EXT_DEF_PORTSPEED_1GBIT;
#elif defined(ISP2200)

	tmp_hba_port.PortSupportedSpeed = EXT_DEF_PORTSPEED_1GBIT;
#elif defined(ISP2300)

	tmp_hba_port.PortSupportedSpeed = EXT_DEF_PORTSPEED_2GBIT;
#else
	/* invalid */
	tmp_hba_port.PortSupportedSpeed = 0;
#endif

	tmp_hba_port.PortSpeed = ha->current_speed;

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    sizeof(EXT_HBA_PORT));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_query_hba_port: ERROR verify write "
		    "rsp buffer.\n");)
		return ret;
	}

	/* now copy up the HBA_PORT to user */
	if (pext->ResponseLen < sizeof(EXT_HBA_PORT))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_HBA_PORT);

	copy_to_user((uint8_t *)pext->ResponseAdr, (uint8_t *)&tmp_hba_port,
	    transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_hba_port: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_query_disc_port(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EXT_STATUS_OK;
	uint32_t	tgt, transfer_size, inst;
	fc_port_t	*fcport;
	os_tgt_t	*tq;
	EXT_DISC_PORT	tmp_disc_port;

	DEBUG9(printk("qla2x00_query_disc_port: entered. Port inst=%02d.\n",
	    pext->Instance);)

	inst = 0;
	for (fcport = ha->fcport; fcport != NULL; fcport = fcport->next) {
		if (atomic_read(&fcport->state) != FC_ONLINE) {
			/* port does not exist anymore */
			DEBUG9(printk("qla2x00_query_disc_port: fcport "
			    "marked lost. d_id=%06x loop_id=%02x not online.\n",
			    fcport->d_id.b24, fcport->loop_id);)

			continue;
		}

		if (inst != pext->Instance) {
			DEBUG9(printk("qla2x00_query_disc_port: found "
			    "fcport %02d. Skipping.\n", inst);)

			inst++;
			continue;
		}

		DEBUG9(printk("qla2x00_query_disc_port: fcport %02d "
		    "online. d_id=%06x loop_id=%02x online.\n",
		    inst, fcport->d_id.b24, fcport->loop_id);)

		/* Found the matching port still connected. */
		break;
	}

	if (fcport == NULL) {
		DEBUG9_10(printk("qla2x00_query_disc_port: dev not found.\n");)
		    pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return ret;
	}
	if (pext->Status != EXT_STATUS_OK) {
		DEBUG9_10(printk("qla2x00_query_disc_port: device down.\n");)
		return QL_STATUS_SUCCESS;
	}

	memset(&tmp_disc_port, 0, sizeof(EXT_DISC_PORT));

	memcpy(tmp_disc_port.WWNN, fcport->node_name, WWN_SIZE);
	memcpy(tmp_disc_port.WWPN, fcport->port_name, WWN_SIZE);

	tmp_disc_port.Id[0] = 0;
	tmp_disc_port.Id[1] = fcport->d_id.r.d_id[2];
	tmp_disc_port.Id[2] = fcport->d_id.r.d_id[1];
	tmp_disc_port.Id[3] = fcport->d_id.r.d_id[0];

	/* Currently all devices on fcport list are target capable devices */
	/* This default value may need to be changed after we add non target
	 * devices also to this list.
	 */
	tmp_disc_port.Type = EXT_DEF_TARGET_DEV;

	if (fcport->flags & FC_FABRIC_DEVICE) {
		tmp_disc_port.Type |= EXT_DEF_FABRIC_DEV;
	}
	if (fcport->flags & FC_TAPE_DEVICE) {
		tmp_disc_port.Type |= EXT_DEF_TAPE_DEV;
	}
	if (fcport->flags & FC_INITIATOR_DEVICE) {
		tmp_disc_port.Type |= EXT_DEF_INITIATOR_DEV;
	}

	tmp_disc_port.LoopID = fcport->loop_id;
	tmp_disc_port.Status = 0;
	tmp_disc_port.Bus    = 0;

	for (tgt = 0; tgt < MAX_TARGETS; tgt++) {
		if ((tq = ha->otgt[tgt]) == NULL) {
			continue;
		}

		if (tq->vis_port == NULL)  /* dg 08/14/01 */
			continue;

		if (memcmp(fcport->port_name,
					tq->vis_port->port_name,
					EXT_DEF_WWN_NAME_SIZE) == 0) {
			tmp_disc_port.TargetId = tgt;
			break;
		}
	}

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    sizeof(EXT_DISC_PORT));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_query_disc_port: ERROR verify write "
		    "rsp buffer.\n");)
		return ret;
	}

	/* now copy up the DISC_PORT to user */
	if (pext->ResponseLen < sizeof(EXT_DISC_PORT))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_DISC_PORT);

	copy_to_user((uint8_t *)pext->ResponseAdr, (uint8_t *)&tmp_disc_port,
	    transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;


	DEBUG9(printk("qla2x00_query_disc_port: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_query_disc_tgt(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EXT_STATUS_OK;
	uint32_t	tgt, transfer_size, inst;
	uint32_t	cnt, i;
	fc_port_t	*tgt_fcport;
	os_tgt_t	*tq;
	EXT_DISC_TARGET	tmp_disc_target;

	DEBUG9(printk("qla2x00_query_disc_tgt: entered.\n");)

	tq = NULL;
	for (tgt = 0, inst = 0; tgt < MAX_TARGETS; tgt++) {
		if (ha->otgt[tgt] == NULL) {
			continue;
		}
		/* if wrong target id then skip to next entry */
		if (inst != pext->Instance) {
			inst++;
			continue;
		}
		tq = ha->otgt[tgt];
		break;
	}

	if (tq == NULL || tgt == MAX_TARGETS) {
		DEBUG9_10(printk("qla2x00_query_disc_tgt: target dev not "
		    "found. tq=%p, tgt=%x.\n", tq, tgt);)
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return pext->Status;
	}

	if (tq->vis_port == NULL) { 	/* dg 08/14/01 */
		DEBUG9_10(printk("qla2x00_query_disc_tgt: target dev not "
		    "found. tq=%p, tgt=%x.\n", tq, tgt);)
		pext->Status = EXT_STATUS_BUSY;
		return pext->Status;
	}

	memset(&tmp_disc_target, 0, sizeof(EXT_DISC_TARGET));

	tgt_fcport = tq->vis_port;
	memcpy(tmp_disc_target.WWNN, tgt_fcport->node_name, WWN_SIZE);
	memcpy(tmp_disc_target.WWPN, tgt_fcport->port_name, WWN_SIZE);

	tmp_disc_target.Id[0] = 0;
	tmp_disc_target.Id[1] = tgt_fcport->d_id.r.d_id[2];
	tmp_disc_target.Id[2] = tgt_fcport->d_id.r.d_id[1];
	tmp_disc_target.Id[3] = tgt_fcport->d_id.r.d_id[0];

	/* All devices on ha->otgt list are target capable devices. */
	tmp_disc_target.Type = EXT_DEF_TARGET_DEV;

	if (tgt_fcport->flags & FC_FABRIC_DEVICE) {
		tmp_disc_target.Type |= EXT_DEF_FABRIC_DEV;
	}
	if (tgt_fcport->flags & FC_TAPE_DEVICE) {
		tmp_disc_target.Type |= EXT_DEF_TAPE_DEV;
	}
	if (tgt_fcport->flags & FC_INITIATOR_DEVICE) {
		tmp_disc_target.Type |= EXT_DEF_INITIATOR_DEV;
	}

	tmp_disc_target.LoopID   = tgt_fcport->loop_id;
	tmp_disc_target.Status   = 0;
	tmp_disc_target.Bus      = 0;
	tmp_disc_target.TargetId = tgt;

	cnt = 0;
	/* enumerate available LUNs under this TGT (if any) */
	if (ha->otgt[tgt] != NULL) {
		for (i = 0; i < MAX_LUNS ; i++) {
			if ((ha->otgt[tgt])->olun[i] !=0)
				cnt++;
		}
	}

	tmp_disc_target.LunCount = cnt;

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
	    sizeof(EXT_DISC_TARGET));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_query_disc_tgt: ERROR verify write "
		    "rsp buffer.\n");)
		return pext->Status;
	}

	/* now copy up the DISC_PORT to user */
	if (pext->ResponseLen < sizeof(EXT_DISC_PORT))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_DISC_TARGET);

	copy_to_user((uint8_t *)pext->ResponseAdr, (uint8_t *)&tmp_disc_target,
	    transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_disc_tgt: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_query_chip(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EXT_STATUS_OK;
	uint32_t	transfer_size, i;
	EXT_CHIP		tmp_isp;
	struct Scsi_Host	*host;

	DEBUG9(printk("qla2x00_query_chip: entered.\n");)

	memset(&tmp_isp, 0, sizeof(EXT_CHIP));

	host = ha->host;
	tmp_isp.VendorId       = QLA2X00_VENDOR_ID;
	tmp_isp.DeviceId       = ha->device_id;
	tmp_isp.SubVendorId    = ha->subsystem_vendor;
	tmp_isp.SubSystemId    = ha->subsystem_device;
	tmp_isp.PciBusNumber   = ha->pdev->bus->number;
	tmp_isp.PciDevFunc     = ha->pdev->devfn;
	tmp_isp.PciSlotNumber  = PCI_SLOT(ha->pdev->devfn);
	tmp_isp.IoAddr         = host->io_port;
	tmp_isp.IoAddrLen      = 512;
	tmp_isp.MemAddr        = 0; /* ? */
	tmp_isp.MemAddrLen     = 0; /* ? */
	tmp_isp.ChipType       = 0; /* ? */
	tmp_isp.InterruptLevel = host->irq;

	for (i = 0; i < 8; i++)
		tmp_isp.OutMbx[i] = 0;

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    sizeof(EXT_CHIP));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_query_chip: ERROR verify write rsp "
		    "buffer.\n");)
		return pext->Status;
	}

	/* now copy up the ISP to user */
	if (pext->ResponseLen < sizeof(EXT_CHIP))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_CHIP);

	copy_to_user((uint8_t *)pext->ResponseAdr, (uint8_t *)&tmp_isp,
	    transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_chip: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_get_data(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int	tmp_rval = 0;

	switch(pext->SubCode) {
	case EXT_SC_GET_STATISTICS:
		tmp_rval = qla2x00_get_statistics(ha, pext, mode);
		break;

	case EXT_SC_GET_FC_STATISTICS:
		tmp_rval = qla2x00_get_fc_statistics(ha, pext, mode);
		break;

	case EXT_SC_GET_PORT_SUMMARY:
		tmp_rval = qla2x00_get_port_summary(ha, pext, mode);
		break;

	case EXT_SC_QUERY_DRIVER:
		tmp_rval = qla2x00_query_driver(ha, pext, mode);
		break;

	case EXT_SC_QUERY_FW:
		tmp_rval = qla2x00_query_fw(ha, pext, mode);
		break;

	case EXT_SC_GET_RNID:
		tmp_rval = qla2x00_get_rnid_params(ha, pext, mode);
		break;

	default:
		DEBUG10(printk("qla2x00_get_data: unknown SubCode %d.\n",
		    pext->SubCode);)
		pext->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		tmp_rval = pext->Status;
		break;
	 }

	return tmp_rval;
}

/*
 * qla2x00_get_statistics
 *	Issues get_link_status mbx cmd and returns statistics
 *	relavent to the specified adapter.
 *
 * Input:
 *	ha = pointer to adapter struct of the specified adapter.
 *	pext = pointer to EXT_IOCTL structure containing values from user.
 *	mode = not used.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_get_statistics(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	EXT_HBA_PORT_STAT tmp_stat;
	int ret;
	link_stat_t stat_buf;
	uint8_t  rval;
	uint8_t  *usr_temp, *kernel_tmp;
	uint16_t mb_stat[1];
	uint32_t transfer_size;

	DEBUG9(printk("entered qla2x00_get_statistics function.\n");)

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
	    sizeof(EXT_HBA_PORT_STAT));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_get_statistics(%ld): ERROR "
		    "VERIFY_WRITE EXT_HBA_PORT_STAT.\n", ha->instance);)
		return pext->Status;
	}

	/* check on loop down */
	if (ha->loop_state != LOOP_READY || 
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
			(test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk(
		    "qla2x00_get_statistics(%ld): loop not ready.\n",
		    ha->host_no);)

		return pext->Status;
	}

	/* Send mailbox cmd to get more. */
	if ((rval = qla2x00_get_link_status(ha, ha->loop_id, &stat_buf,
	    mb_stat)) != QL_STATUS_SUCCESS) {

		if (rval == BIT_0) {
			pext->Status = EXT_STATUS_NO_MEMORY;
		} else if (rval == BIT_1) {
			pext->Status = EXT_STATUS_MAILBOX;
			pext->DetailStatus = EXT_DSTATUS_NOADNL_INFO;
		} else {
			pext->Status = EXT_STATUS_ERR;
		}

		DEBUG9_10(printk("qla2x00_get_statistics(%ld): ERROR "
		    "mailbox failed. mb[0]=%x.\n", ha->instance, mb_stat[0]);)
		printk(KERN_WARNING
		     "qla2x00_get_statistics(%ld): ERROR "
		     "mailbox failed. mb[0]=%x.\n", ha->instance, mb_stat[0]);

		return pext->Status;
	}

	tmp_stat.ControllerErrorCount   =  ha->total_isp_aborts;
	tmp_stat.DeviceErrorCount       =  ha->total_dev_errs;
	tmp_stat.TotalIoCount           =  ha->total_ios;
	tmp_stat.TotalMBytes            =  ha->total_bytes;
	tmp_stat.TotalLipResets         =  ha->total_lip_cnt;
	/*
	   tmp_stat.TotalInterrupts        =  ha->total_isr_cnt;
	 */

	tmp_stat.TotalLinkFailures               = stat_buf.link_fail_cnt;
	tmp_stat.TotalLossOfSync                 = stat_buf.loss_sync_cnt;
	tmp_stat.TotalLossOfSignals              = stat_buf.loss_sig_cnt;
	tmp_stat.PrimitiveSeqProtocolErrorCount  = stat_buf.prim_seq_err_cnt;
	tmp_stat.InvalidTransmissionWordCount    = stat_buf.inval_xmit_word_cnt;
	tmp_stat.InvalidCRCCount                 = stat_buf.inval_crc_cnt;

	/* now copy up the STATISTICS to user */
	if (pext->ResponseLen < sizeof(EXT_HBA_PORT_STAT))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_HBA_PORT_STAT);


	usr_temp   = (uint8_t *)pext->ResponseAdr;
	kernel_tmp = (uint8_t *)&tmp_stat;
	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("finished qla2x00_get_statistics function.\n");)

	return pext->Status;
}

/*
 * qla2x00_get_fc_statistics
 *	Issues get_link_status mbx cmd to the target device with
 *	the specified WWN and returns statistics relavent to the
 *	device.
 *
 * Input:
 *	ha = pointer to adapter struct of the specified device.
 *	pext = pointer to EXT_IOCTL structure containing values from user.
 *	mode = not used.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_get_fc_statistics(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	EXT_HBA_PORT_STAT tmp_stat;
	EXT_DEST_ADDR addr_struct;
	int ret;
	link_stat_t stat_buf;
	uint8_t  rval, tgt;
	uint8_t  *usr_temp, *kernel_tmp;
	uint8_t  *req_name;
	uint16_t mb_stat[1];
	uint32_t transfer_size;

	DEBUG9(printk("entered qla2x00_get_fc_statistics function.\n");)

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
	    sizeof(EXT_HBA_PORT_STAT));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_get_fc_statistics(%ld): ERROR "
		    "VERIFY_WRITE.\n", ha->instance);)
		return pext->Status;
	}

	ret = copy_from_user(&addr_struct, pext->RequestAdr, pext->RequestLen);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		return pext->Status;
	}

	/* find the device's loop_id */
	switch (addr_struct.DestType) {
	case EXT_DEF_DESTTYPE_WWPN:
		req_name = addr_struct.DestAddr.WWPN;
		for (tgt = 0; tgt < MAX_FIBRE_DEVICES; tgt++) {
			if (memcmp(ha->fc_db[tgt].wwn,
					req_name,
					EXT_DEF_WWN_NAME_SIZE) == 0)
				break;
		}
		break;

	case EXT_DEF_DESTTYPE_WWNN:
	case EXT_DEF_DESTTYPE_PORTID:
	case EXT_DEF_DESTTYPE_FABRIC:
	case EXT_DEF_DESTTYPE_SCSI:
	default:
		pext->Status = EXT_STATUS_INVALID_PARAM;
		pext->DetailStatus = EXT_DSTATUS_NOADNL_INFO;
		DEBUG9_10(printk("qla2x00_get_statistics(%ld): ERROR "
		    "Unsupported subcode address type.\n", ha->instance);)
		return pext->Status;

		break;
	}

	if (tgt == MAX_FIBRE_DEVICES) {
		/* not found */
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		pext->DetailStatus = EXT_DSTATUS_TARGET;
		return pext->Status;
	}

	/* check for suspended/lost device */
	/*
	   if (ha->fcport is suspended/lost) {
	   pext->Status = EXT_STATUS_SUSPENDED;
	   pext->DetailStatus = EXT_DSTATUS_TARGET;
	   return pext->Status;
	   }
	 */

	/* check on loop down */
	if (ha->loop_state != LOOP_READY ||
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {
		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk("qla2x00_get_fc_statistics(%ld): loop not "
		    "ready.\n", ha->host_no);)
		return pext->Status;
	}

	/* Send mailbox cmd to get more. */
	if ((rval = qla2x00_get_link_status(ha, ha->fc_db[tgt].loop_id,
	    &stat_buf, mb_stat)) != QL_STATUS_SUCCESS) {
		if (rval == BIT_0) {
			pext->Status = EXT_STATUS_NO_MEMORY;
		} else if (rval == BIT_1) {
			pext->Status = EXT_STATUS_MAILBOX;
			pext->DetailStatus = EXT_DSTATUS_NOADNL_INFO;
		} else {
			pext->Status = EXT_STATUS_ERR;
		}

		DEBUG9_10(printk("qla2x00_get_fc_statistics(%ld): ERROR "
		    "mailbox failed. mb[0]=%x.\n", ha->instance, mb_stat[0]);)
		return pext->Status;
	}

	tmp_stat.ControllerErrorCount   =  ha->total_isp_aborts;
	tmp_stat.DeviceErrorCount       =  ha->total_dev_errs;
	tmp_stat.TotalIoCount           =  ha->total_ios;
	tmp_stat.TotalMBytes            =  ha->total_bytes;
	tmp_stat.TotalLipResets         =  ha->total_lip_cnt;
	/*
	   tmp_stat.TotalInterrupts        =  ha->total_isr_cnt;
	 */

	tmp_stat.TotalLinkFailures               = stat_buf.link_fail_cnt;
	tmp_stat.TotalLossOfSync                 = stat_buf.loss_sync_cnt;
	tmp_stat.TotalLossOfSignals              = stat_buf.loss_sig_cnt;
	tmp_stat.PrimitiveSeqProtocolErrorCount  = stat_buf.prim_seq_err_cnt;
	tmp_stat.InvalidTransmissionWordCount    = stat_buf.inval_xmit_word_cnt;
	tmp_stat.InvalidCRCCount                 = stat_buf.inval_crc_cnt;

	/* now copy up the STATISTICS to user */
	if (pext->ResponseLen < sizeof(EXT_HBA_PORT_STAT))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_HBA_PORT_STAT);


	usr_temp   = (uint8_t *)pext->ResponseAdr;
	kernel_tmp = (uint8_t *)&tmp_stat;
	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("finished qla2x00_get_fc_statistics function.\n");)

	return pext->Status;
}

STATIC int
qla2x00_get_port_summary(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EXT_STATUS_OK;
	uint8_t		*usr_temp, *kernel_tmp;
	uint32_t	b;
	uint32_t	i, port_cnt, entry;
	uint32_t	number_of_entries = 0;
	uint32_t	tgt_cnt, transfer_size;
	fc_port_t	*fcport;
	void		*start_of_entry_list, *current_offset;

	static EXT_DEVICEDATA		devicedata;
	static EXT_DEVICEDATAENTRY	dd_entry;

	DEBUG9(printk("qla2x00_get_port_summary: entered.\n");)

	port_cnt = 0;
	tgt_cnt  = 0;
	memset(&devicedata, 0, sizeof(EXT_DEVICEDATA));
	memset(&dd_entry, 0, sizeof(EXT_DEVICEDATAENTRY));

	for (fcport = ha->fcport; fcport != NULL; fcport = fcport->next) {
		port_cnt++;
	}
	devicedata.TotalDevices = port_cnt;

	number_of_entries = pext->ResponseLen / sizeof(EXT_DEVICEDATAENTRY);

	/* we want the lesser of port_cnt and number_of_entries */
	if (number_of_entries > port_cnt)
		number_of_entries = port_cnt;
	devicedata.ReturnListEntryCount  = number_of_entries;

	DEBUG9(printk("qla2x00_ioctl: EXT_SC_GET_PORT_SUMMARY port_cnt=%x, "
	    "return entry cnt=%x.\n", port_cnt, number_of_entries);)

	transfer_size = sizeof(devicedata.ReturnListEntryCount) +
	    sizeof(devicedata.TotalDevices);

	/* copy top of devicedata here */
	ret = verify_area(VERIFY_WRITE, (void *)(pext->ResponseAdr),
	    transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_ioctl: ERROR verify WRITE rsp buf.\n");)
		return pext->Status;
	}

	usr_temp   = (uint8_t *)pext->ResponseAdr;
	kernel_tmp = (uint8_t *)&devicedata;
	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	start_of_entry_list = (void *)(pext->ResponseAdr) + transfer_size;

	for (entry = 0, fcport = ha->fcport; (entry < number_of_entries) &&
	    (fcport); entry++, fcport = fcport->next) {

		/* copy from fc_db of this target (port) to dd_entry */

		memcpy(dd_entry.NodeWWN, fcport->node_name, WWN_SIZE);
		memcpy(dd_entry.PortWWN, fcport->port_name, WWN_SIZE);

		for (b = 0; b < 3 ; b++)
			dd_entry.PortID[b] = fcport->d_id.r.d_id[2-b];

		if (fcport->flags & FC_FABRIC_DEVICE) {
			dd_entry.ControlFlags = EXT_DEF_GET_FABRIC_DEVICE;
		} else {
			dd_entry.ControlFlags = 0;
		}

		dd_entry.TargetAddress.Bus    = 0;
		/* Retrieve 'Target' number for port via fc_db */
		for (i = 0; i < MAX_TARGETS; i++) {
			if (ha->fc_db[i].loop_id == PORT_UNUSED)
				continue;

			if (memcmp(fcport->port_name,
					ha->fc_db[i].wwn,
					EXT_DEF_WWN_NAME_SIZE) == 0) {
				dd_entry.TargetAddress.Target = i;
				break;
			}
		}
		dd_entry.TargetAddress.Lun    = 0;
		dd_entry.DeviceFlags          = 0;
		dd_entry.LoopID               = fcport->loop_id;
		dd_entry.BaseLunNumber        = 0;

		current_offset = (void *)(entry * sizeof(EXT_DEVICEDATAENTRY));

		transfer_size = sizeof(EXT_DEVICEDATAENTRY);
		ret = verify_area(VERIFY_WRITE,
		    (void *)((start_of_entry_list + (u_long)current_offset)),
		    transfer_size);

		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG10(printk("qla2x00_ioctl: ERROR verify_area WRITE "
			    "ha=%p\n", ha);)
			return pext->Status;
		}

		/* now copy up this dd_entry to user */
		usr_temp = (uint8_t *)((u_long) start_of_entry_list +
		    (u_long)current_offset);
		kernel_tmp = (uint8_t *)&dd_entry;
	 	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	} /* for number_of_entries */

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_get_port_summary: exiting.\n");)
	return pext->Status;
}

STATIC int
qla2x00_query_driver(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EINVAL;
	uint8_t		*usr_temp, *kernel_tmp;
	uint32_t	transfer_size;
	EXT_DRIVER	driver_prop;

	DEBUG9(printk("qla2x00_query_driver: entered.\n");)

	memset(&driver_prop, 0, sizeof(EXT_DRIVER));

	sprintf(driver_prop.Version, QLA2100_VERSION);
	driver_prop.NumOfBus = MAX_BUSES;
	driver_prop.TargetsPerBus = MAX_FIBRE_DEVICES;
	driver_prop.LunsPerTarget = MAX_LUNS;
	driver_prop.MaxTransferLen  = 0xffffffff;
	driver_prop.MaxDataSegments = 0xffffffff;

	if (ha->flags.enable_64bit_addressing == 1)
		driver_prop.DmaBitAddresses = 64;
	else
		driver_prop.DmaBitAddresses = 32;

	if (pext->ResponseLen < sizeof(EXT_DRIVER))
		transfer_size = pext->ResponseLen;
	else
		transfer_size = sizeof(EXT_DRIVER);

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG10(printk(
		    "qla2x00_query_driver: ERROR verify WRITE rsp buf.\n");)
		return pext->Status;
	}

	/* now copy up the ISP to user */
	usr_temp   = (uint8_t *)pext->ResponseAdr;
	kernel_tmp = (uint8_t *)&driver_prop;
	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_driver: exiting.\n");)

	return pext->Status;
}

STATIC int
qla2x00_query_fw(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = EINVAL;
	qla_boards_t	*bdp;
	uint8_t		*usr_temp, *kernel_tmp;
	uint32_t	transfer_size;
	EXT_FW		fw_prop;

	DEBUG9(printk("qla2x00_query_fw: entered.\n");)

	memset(&fw_prop, 0, sizeof(EXT_FW));

	bdp = &QLBoardTbl_fc[ha->devnum];
	fw_prop.Version[0] = bdp->fwver[0];
	fw_prop.Version[1] = bdp->fwver[1];
	fw_prop.Version[2] = bdp->fwver[2];

	transfer_size = sizeof(EXT_FW);

	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG10(printk(
		    "qla2x00_query_fw: ERROR verify WRITE rsp buf.\n");)
		return pext->Status;
	}

	usr_temp   = (uint8_t *)pext->ResponseAdr;
	kernel_tmp = (uint8_t *)&fw_prop;
	copy_to_user(usr_temp, kernel_tmp, transfer_size);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_query_fw: exiting.\n");)

	return pext->Status;
}

STATIC int
qla2x00_send_fcct(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint16_t	mb[MAILBOX_REGISTER_COUNT];

	static fc_lun_t		temp_fclun;
	fc_lun_t		*fclun;
	static fc_port_t	tmp_fcport;
	os_lun_t		*lq;
	os_tgt_t		*tq;

	static Scsi_Cmnd scsi_cmd;
	Scsi_Cmnd	*pscsi_cmd = &scsi_cmd;
	srb_t		*sp = NULL;

	DEBUG9(printk("qla2x00_send_fcct(%ld): entered.\n", ha->host_no);)

	/* check on current topology */
	if ((ha->current_topology != ISP_CFG_F) &&
	    (ha->current_topology != ISP_CFG_FL)) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): ERROR not in F or FL"
		    "mode.\n",ha->host_no);)
		return ret;
	}

	if (ha->ioctl_mem_size <= 0) {
		if (qla2x00_get_new_ioctl_dma_mem(ha,
		    QLA_INITIAL_IOCTLMEM_SIZE) != QL_STATUS_SUCCESS) {

			DEBUG9_10(printk("qla2x00_send_fcct: ERROR cannot "
			    "alloc DMA buffer size=%lx.\n",
			    QLA_INITIAL_IOCTLMEM_SIZE);)

			pext->Status = EXT_STATUS_NO_MEMORY;
			return pext->Status;
		}
	}

	if (pext->ResponseLen > ha->ioctl_mem_size) {
		if (qla2x00_get_new_ioctl_dma_mem(ha, pext->ResponseLen) !=
		    QL_STATUS_SUCCESS) {
			DEBUG9_10(printk("qla2x00_send_fcct: ERROR "
			    "cannot alloc requested"
			    "DMA buffer size %x.\n",
			    pext->ResponseLen);)
			pext->Status = EXT_STATUS_NO_MEMORY;
			return pext->Status;
		}

		DEBUG9(printk("qla2x00_send_fcct(%ld): rsp buf length larger "
		    "than existing size. Additional"
		    "mem alloc successful.\n",
		    ha->host_no);)
	}

	if (pext->RequestLen > ha->ioctl_mem_size) {
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): ERROR RequestLen"
		    "too big=%x.\n",
		    ha->host_no, pext->RequestLen);)

		return pext->Status;
	}

	ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    pext->RequestLen);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): ERROR"
		    "verify read request buf.\n",
		    ha->host_no);)

		return ret;
	}

	DEBUG9(printk("qla2x00_send_fcct(%ld): req buf verified.\n",
	    ha->host_no);)

	if ((sp = qla2x00_get_new_sp(ha)) == NULL) {

		DEBUG9_10(printk("qla2x00_send_fcct: ERROR "
		    "cannot alloc sp %p.\n", sp);)

		pext->Status = EXT_STATUS_NO_MEMORY;
		return pext->Status;
	}

	DEBUG9(printk("qla2x00_send_fcct(%ld): after alloc sp.\n",
	    ha->host_no);)

	/* clear scsi_cmd to be used */
	memset(&scsi_cmd, 0, sizeof(Scsi_Cmnd));
	memset(ha->ioctl_mem, 0, ha->ioctl_mem_size);

	/* copy request buffer */
	ret = copy_from_user(ha->ioctl_mem, pext->RequestAdr, pext->RequestLen);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): ERROR"
		    "copy_from_user() failed (%d).\n",
		    ha->host_no, ret);)

		return ret;
	}

	DEBUG9(printk("qla2x00_send_fcct(%ld): after copy request.\n",
	    ha->host_no);)

	/* setup sp for this command */
	tq = ha->ioctl->ioctl_tq;
	lq = ha->ioctl->ioctl_lq;
	sp->cmd = pscsi_cmd;
	sp->flags = SRB_IOCTL;
	sp->lun_queue = lq;
	sp->tgt_queue = tq;
	fclun = &temp_fclun;
	fclun->fcport = &tmp_fcport;
	fclun->lun = 0;
	fclun->flags = 0;
	fclun->next = NULL;
	lq->fclun = fclun;
	lq->fclun->fcport->ha = ha;

	/* init scsi_cmd */
	scsi_cmd.host = ha->host;
	scsi_cmd.scsi_done = qla2x00_fcct_done;

	/* check on management server login status */
	if (ha->flags.management_server_logged_in == 0) {
		/* login to management server device */

		ret = qla2x00_login_fabric(ha, MANAGEMENT_SERVER, 0xff, 0xff,
		    0xfa, &mb[0], BIT_1);

		if (ret != 0 || mb[0] != 0x4000) {
			pext->Status = EXT_STATUS_DEV_NOT_FOUND;

	 		DEBUG10(printk("qla2x00_send_fcct(%ld): ERROR"
			    "login to MS.\n",
			    ha->host_no);)
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);

			return pext->Status;
		}

		ha->flags.management_server_logged_in = 1;
	}

	DEBUG9(printk("qla2x00_send_fcct(%ld): success login to MS.\n",
	    ha->host_no);)

	/* check on loop down */
	if (ha->loop_state != LOOP_READY || 
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG10(printk("qla2x00_send_fcct(%ld): loop not ready.\n",
		    ha->host_no);)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return pext->Status;
	}

	DEBUG9(printk("qla2x00_send_fcct(%ld): going to issue command.\n",
	    ha->host_no);)

	ret = qla2x00_start_ms_cmd(ha, pext, sp);

	DEBUG9(printk("qla2x00_send_fcct(%ld): after issue command.\n",
	    ha->host_no);)

	if (ret != 0) {
		/* We waited and post function did not get called */
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): command timed out.\n",
		    ha->host_no);)
		pext->Status = EXT_STATUS_MS_NO_RESPONSE;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return pext->Status;
	}

	if (CMD_COMPL_STATUS(pscsi_cmd) != 0 ||
	    CMD_ENTRY_STATUS(pscsi_cmd) != 0) {
		DEBUG9_10(printk(
		    "qla2x00_send_fcct(%ld): command returned error=%x.\n",
		    ha->host_no, CMD_COMPL_STATUS(pscsi_cmd));)
		pext->Status = EXT_STATUS_ERR;
		return pext->Status;
	}

	/* getting device data and putting in pext->ResponseAdr */
	ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
	    pext->ResponseLen);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_fcct(%ld): ERROR verify write "
		    "response buffer. ha=%p.\n",
		    ha->host_no, ha);)
		return ret;
	}

	/* sending back data returned from Management Server */
	copy_to_user((uint8_t *)pext->ResponseAdr,
	    (uint8_t *)ha->ioctl_mem, pext->ResponseLen);

	DEBUG9(printk("qla2x00_send_fcct(%ld): exiting normally.\n",
	    ha->host_no);)

	return 0;
}

STATIC int
qla2x00_start_ms_cmd(scsi_qla_host_t *ha, EXT_IOCTL *pext, srb_t *sp)
{
	ms_iocb_entry_t	*pkt;
	unsigned long cpu_flags = 0;

	/* get spin lock for this operation */
	spin_lock_irqsave(&ha->hardware_lock, cpu_flags);

	/* Get MS request packet. */
	pkt = (ms_iocb_entry_t *)qla2x00_ms_req_pkt(ha, sp);
	if (pkt == NULL) {
		/* release spin lock and return error. */
		spin_unlock_irqrestore(&ha->hardware_lock, cpu_flags);

		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG10(printk("qla2x00_ioctl: FCCT_PASSTHRU - could not get "
		    "Request Packet.\n");)
		return pext->Status;
	}

	pkt->entry_type  = MS_IOCB_TYPE;
	pkt->entry_count = 1;
	pkt->loop_id     = MANAGEMENT_SERVER;
	pkt->timeout     = QLA_PT_CMD_TOV;
	pkt->DSDcount    = 1;
	pkt->RespDSDcount = 2;
	pkt->Response_bytecount = pext->ResponseLen;
	pkt->Request_bytecount  = pext->RequestLen;

	/* loading command payload address */
	pkt->dseg_req_address[0] = LS_64BITS(ha->ioctl_mem_phys);
	pkt->dseg_req_address[1] = MS_64BITS(ha->ioctl_mem_phys);
	pkt->dseg_req_length = pext->RequestLen;

	/* loading command response address */
	pkt->dseg_rsp_address[0] = LS_64BITS(ha->ioctl_mem_phys);
	pkt->dseg_rsp_address[1] = MS_64BITS(ha->ioctl_mem_phys);
	pkt->dseg_rsp_length = pext->ResponseLen;

	/* set flag to indicate IOCTL FCCT PassThru in progress */
	ha->ioctl->FCCT_InProgress = 1;
	ha->ioctl->ioctl_tov = pkt->timeout + 1; /* 1 second more */

	/* prepare for receiving completion. */
	qla2x00_ioctl_sem_init(ha);

	/* Issue command to ISP */
	qla2x00_isp_cmd(ha);

	ha->ioctl->cmpl_timer.expires = jiffies + ha->ioctl->ioctl_tov * HZ;
	add_timer(&ha->ioctl->cmpl_timer);

	DEBUG9(printk("qla2x00_start_ms_cmd: releasing hardware_lock.\n");)
	spin_unlock_irqrestore(&ha->hardware_lock, cpu_flags);

	DEBUG9(printk("qla2x00_start_ms_cmd: sleep for completion.\n");)
	down(&ha->ioctl->cmpl_sem);

	del_timer(&ha->ioctl->cmpl_timer);

	if (ha->ioctl->FCCT_InProgress == 1) {
	 	DEBUG9_10(printk("qla2x00_start_ms_cmd: timed out. exiting.\n");)
		return 1;
	}

	DEBUG9(printk("qla2x00_start_ms_cmd: done. exiting.\n");)
	return 0;
}

STATIC int
qla2x00_wwpn_to_scsiaddr(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		rval;
	fc_port_t	*tgt_fcport;
	os_tgt_t	*tq;
	uint8_t		tmp_wwpn[EXT_DEF_WWN_NAME_SIZE];
	uint32_t	b, tgt, l;
	EXT_SCSI_ADDR	tmp_addr;


	DEBUG9(printk("qla2x00_wwpn_to_scsiaddr(%ld): entered.\n",
	    ha->host_no);)

	if (pext->RequestLen != EXT_DEF_WWN_NAME_SIZE ||
	    pext->ResponseLen < sizeof(EXT_SCSI_ADDR)) {
		/* error */
		DEBUG9_10(printk("qla2x00_wwpn_to_scsiaddr(%ld): invalid WWN "
		    "buffer size %d received.\n",
		    ha->host_no, pext->ResponseLen);)
		pext->Status = EXT_STATUS_INVALID_PARAM;

		return pext->Status;
	}

	rval = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
			pext->RequestLen);
	if (rval) {
		DEBUG9_10(printk("qla2x00_wwpn_to_scsiaddr(%ld): ERROR "
		    "VERIFY_READ request buf.\n", ha->host_no);)
		pext->Status = EXT_STATUS_COPY_ERR;
		return pext->Status;
	}

	rval = copy_from_user(tmp_wwpn, pext->RequestAdr, pext->RequestLen);
	if (rval) {
		DEBUG9_10(printk("qla2x00_wwpn_to_scsiaddr(%ld): ERROR "
		    "copy_from_user failed (%d) request buf.\n",
		    ha->host_no, rval);)
		pext->Status = EXT_STATUS_COPY_ERR;
		return pext->Status;
	}

	tq = NULL;
	for (tgt = 0; tgt < MAX_TARGETS; tgt++) {
		if (ha->otgt[tgt] == NULL) {
			continue;
		}

		tq = ha->otgt[tgt];
		if (tq->vis_port == NULL) {
			break;
		}

		tgt_fcport = tq->vis_port;
		if (memcmp(tmp_wwpn,
				tgt_fcport->port_name,
				EXT_DEF_WWN_NAME_SIZE) == 0) {
			break;
		}
	}

	if (tq == NULL || tgt >= MAX_TARGETS) {
		DEBUG9_10(printk("qla2x00_query_disc_tgt: target dev not "
		    "found. tq=%p, tgt=%x.\n", tq, tgt);)
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return pext->Status;
	}

	if (tq->vis_port == NULL) { 	/* dg 08/14/01 */
		DEBUG9_10(printk("qla2x00_query_disc_tgt: target port not "
		    "found. tq=%p, tgt=%x.\n", tq, tgt);)
		pext->Status = EXT_STATUS_BUSY;
		return pext->Status;
	}	

	/* Currently we only have bus 0 and no translation on LUN */
	b = 0;
	l = 0;

	/*
	 * Return SCSI address. Currently no translation is done for
	 * LUN.
	 */
	tmp_addr.Bus = b;
	tmp_addr.Target = tgt;
	tmp_addr.Lun = l;
	if (pext->ResponseLen > sizeof(EXT_SCSI_ADDR))
		pext->ResponseLen = sizeof(EXT_SCSI_ADDR);

	rval = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
	    pext->ResponseLen);
	if (rval) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_wwpn_to_scsiaddr(%ld): ERROR "
		    "VERIFY_WRITE response buf.\n", ha->host_no);)
		return pext->Status;
	}

	copy_to_user((uint8_t *)pext->ResponseAdr, &tmp_addr,
	    pext->ResponseLen);

	DEBUG9(printk(KERN_INFO
	    "qla2x00_wwpn_to_scsiaddr: Found t%d l%d for %02x%02x%02x%02x"
	    "%02x%02x%02x%02x.\n",
	    tmp_addr.Target, tmp_addr.Lun,
	    tmp_wwpn[0], tmp_wwpn[1], tmp_wwpn[2], tmp_wwpn[3],
	    tmp_wwpn[4], tmp_wwpn[5], tmp_wwpn[6], tmp_wwpn[7]);)

	pext->Status = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_wwpn_to_scsiaddr(%ld): exiting.\n",
	    ha->host_no);)

	return pext->Status;
}

STATIC int
qla2x00_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int tmp_rval = 0;

	switch(pext->SubCode) {
	case EXT_SC_SEND_SCSI_PASSTHRU:
		tmp_rval = qla2x00_sc_scsi_passthru(ha, pext, mode);
		break;
	case EXT_SC_SEND_FC_SCSI_PASSTHRU:
		tmp_rval = qla2x00_sc_fc_scsi_passthru(ha, pext, mode);
		break;
	case EXT_SC_SCSI3_PASSTHRU:
		tmp_rval = qla2x00_sc_scsi3_passthru(ha, pext, mode);
		break;
	default:
		break;
	}

	return tmp_rval;
}

STATIC int
qla2x00_sc_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint8_t		*usr_temp, *kernel_tmp;
	uint8_t		scsi_direction;
	uint32_t	i;

#if defined(QL_DEBUG_LEVEL_9)
	uint32_t	b, t, l;
#endif
	uint32_t	transfer_len;

	static EXT_SCSI_PASSTHRU	scsi_pass;
	EXT_SCSI_PASSTHRU	*pscsi_pass = &scsi_pass;

	static Scsi_Cmnd	scsi_cmd;
	Scsi_Cmnd		*pscsi_cmd = &scsi_cmd;
	static Scsi_Device	scsi_device;

	DEBUG9(printk("qla2x00_sc_scsi_passthru: entered.\n");)

	/* clear scsi_cmd and scsi_device to be used */
	memset(pscsi_cmd, 0, sizeof(Scsi_Cmnd));
	memset(&scsi_device, 0, sizeof(Scsi_Device));
	memset(ha->ioctl_mem, 0, ha->ioctl_mem_size);

	ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    sizeof(EXT_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR verify READ "
		    "SCSI_PASSTHRU.\n");)
		return ret;
	}

	if (pext->ResponseLen > ha->ioctl_mem_size) {
		if (qla2x00_get_new_ioctl_dma_mem(ha, pext->ResponseLen) !=
		    QL_STATUS_SUCCESS) {
			DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
			    "cannot alloc requested"
			    "DMA buffer size %x.\n",
			    pext->ResponseLen);)
			pext->Status = EXT_STATUS_NO_MEMORY;
			return pext->Status;
		}
	}

	/* Copy request buffer */
	usr_temp = (uint8_t *)pext->RequestAdr;
	kernel_tmp = (uint8_t *)pscsi_pass;
	ret = copy_from_user(kernel_tmp, usr_temp, sizeof(EXT_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
				"copy_from_user() failed (%d).\n",
				ret);)
		return ret;
	}

	/* set target coordinates */
	scsi_cmd.target = pscsi_pass->TargetAddr.Target;
	scsi_cmd.lun    = pscsi_pass->TargetAddr.Lun;

	/* Verify target exists */
	if (TGT_Q(ha, scsi_cmd.target) == NULL) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return pext->Status;
	}

	/* Copy over cdb */

	if (pscsi_pass->CdbLength == 6) {
		scsi_cmd.cmd_len = 6;

	} else if (pscsi_pass->CdbLength == 10) {
		scsi_cmd.cmd_len = 0x0A;

	} else if (pscsi_pass->CdbLength == 12) {
		scsi_cmd.cmd_len = 0x0C;

	} else {
		printk(KERN_WARNING
		    "qla2x00_sc_scsi_passthru: Unsupported Cdb Length=%x.\n",
		pscsi_pass->CdbLength);
		pext->Status = EXT_STATUS_INVALID_PARAM;
		return pext->Status;
	}

	memcpy(scsi_cmd.data_cmnd, pscsi_pass->Cdb, scsi_cmd.cmd_len);
	memcpy(scsi_cmd.cmnd, pscsi_pass->Cdb, scsi_cmd.cmd_len);

	DEBUG9(printk("Dump of cdb buffer:\n");)
	DEBUG9(qla2x00_dump_buffer((uint8_t *)&scsi_cmd.data_cmnd[0],
	    scsi_cmd.cmd_len);)

	pscsi_cmd->host    = ha->host;

	/* mark this as a special delivery and collection command */
	scsi_cmd.flags     = 0;
	scsi_cmd.scsi_done = qla2x00_scsi_pt_done;

	scsi_cmd.device               = &scsi_device;
	scsi_cmd.device->tagged_queue = 0;
	scsi_cmd.use_sg               = 0; /* no ScatterGather */
	scsi_cmd.request_bufflen      = pext->ResponseLen;
	scsi_cmd.request_buffer       = ha->ioctl_mem;
	scsi_cmd.timeout_per_command  = QLA_PT_CMD_TOV * HZ;
	CMD_RESID_LEN(pscsi_cmd) = SRB_IOCTL; /* Used to set sp->flags later */

	if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		/* sending user data from pext->ResponseAdr to device */
		ret = verify_area(VERIFY_READ, (void *)pext->ResponseAdr,
		    pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
			    "verify READ SCSI_PASSTHRU.\n");)
			return pext->Status;
		}

		scsi_cmd.sc_data_direction = SCSI_DATA_WRITE;
		usr_temp   = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		ret = copy_from_user(kernel_tmp, usr_temp, pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
			    "copy_from_user() failed (%d).\n",
			    ret);)
			return pext->Status;
		}
	} else {
		scsi_cmd.sc_data_direction = SCSI_DATA_READ;
	}

	DEBUG9({
		b = SCSI_BUS_32(pscsi_cmd);
		t = SCSI_TCN_32(pscsi_cmd);
		l = SCSI_LUN_32(pscsi_cmd);
	})
	DEBUG9(printk("qla2x00_sc_scsi_passthru: CDB=%02x %02x %02x %02x; "
	    "b=%x t=%x l=%x.\n",
	    scsi_cmd.cmnd[0], scsi_cmd.cmnd[1], scsi_cmd.cmnd[2],
	    scsi_cmd.cmnd[3], b, t, l);)

	/*
	 * Check the status of the port
	 */
	if (qla2x00_check_tgt_status(ha, pscsi_cmd) != QL_STATUS_SUCCESS) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return pext->Status;
	}

	/* set flag to indicate IOCTL SCSI PassThru in progress */
	ha->ioctl->SCSIPT_InProgress = 1;
	ha->ioctl->ioctl_tov = (int)QLA_PT_CMD_TOV + 1;

	/* prepare for receiving completion. */
	qla2x00_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(pscsi_cmd) = (int) IOCTL_INVALID_STATUS;

	/* send command to adapter */
	DEBUG9(printk("qla2x00_sc_scsi_passthru(%ld): sending command.\n",
	    ha->host_no);)

	/* get spin lock for this operation */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	spin_lock_irqsave(&io_request_lock, ha->cpu_flags);
#else
	spin_lock_irqsave(ha->host->host_lock, ha->cpu_flags);
#endif

	qla2x00_queuecommand(pscsi_cmd, (void *) qla2x00_scsi_pt_done);

	ha->ioctl->cmpl_timer.expires = jiffies + ha->ioctl->ioctl_tov * HZ;
	add_timer(&ha->ioctl->cmpl_timer);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	spin_unlock_irqrestore(&io_request_lock, ha->cpu_flags);
#else
	spin_unlock_irqrestore(ha->host->host_lock, ha->cpu_flags);
#endif

	down(&ha->ioctl->cmpl_sem);

	del_timer(&ha->ioctl->cmpl_timer);

	if (ha->ioctl->SCSIPT_InProgress == 1) {

		printk(KERN_WARNING
		    "qla2x00: scsi%ld ERROR passthru command timeout.\n",
		    ha->host_no);

		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		return EXT_STATUS_ERR;
	}

	if (CMD_COMPL_STATUS(pscsi_cmd) == (int)IOCTL_INVALID_STATUS) {

		DEBUG9(printk("qla2x00_sc_scsi_passthru(%ld): ERROR - "
		    "command not completed.\n",
		    ha->host_no);)

		pext->Status = EXT_STATUS_ERR;
		return EXT_STATUS_ERR;
	}

	switch (CMD_COMPL_STATUS(pscsi_cmd)) {
	case CS_INCOMPLETE:
	case CS_ABORTED:
	case CS_PORT_UNAVAILABLE:
	case CS_PORT_LOGGED_OUT:
	case CS_PORT_CONFIG_CHG:
	case CS_PORT_BUSY:
		DEBUG10(printk("qla2x00_sc_scsi_passthru: cs err = %x.\n",
		    CMD_COMPL_STATUS(pscsi_cmd));)
		ret = EXT_STATUS_ERR;
		pext->Status = EXT_STATUS_BUSY;

		return ret;
	}

	if ((CMD_COMPL_STATUS(pscsi_cmd) == CS_DATA_UNDERRUN) ||
	    (CMD_SCSI_STATUS(pscsi_cmd) != 0)) {

		/* have done the post function */
		pext->Status       = EXT_STATUS_SCSI_STATUS;
		pext->DetailStatus = CMD_SCSI_STATUS(pscsi_cmd);
		DEBUG9_10(printk("qla2x00_sc_scsi_passthru: data underrun or "
		    "scsi err. host status =0x%x, scsi status = 0x%x.\n",
		CMD_COMPL_STATUS(pscsi_cmd), CMD_SCSI_STATUS(pscsi_cmd));)

	} else if (CMD_COMPL_STATUS(pscsi_cmd) != 0) {
		DEBUG9_10(printk("qla2x00_sc_scsi_passthru: cs err = %x. "
		    "copying ext stat %x\n",
		    CMD_COMPL_STATUS(pscsi_cmd), pext->Status);)
		return pext->Status;
	}

	/* copy up structure to make sense data available to user */
	pscsi_pass->SenseLength = CMD_ACTUAL_SNSLEN(pscsi_cmd);
	if (CMD_ACTUAL_SNSLEN(pscsi_cmd)) {
		for (i = 0; i < CMD_ACTUAL_SNSLEN(pscsi_cmd); i++)
			pscsi_pass->SenseData[i] = pscsi_cmd->sense_buffer[i];

		ret = verify_area(VERIFY_WRITE, (void *)pext->RequestAdr,
		    sizeof(EXT_SCSI_PASSTHRU));
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
			    "verify WRITE FC_SCSI_PASSTHRU]\n");)
			return ret;
		}

		usr_temp   = (uint8_t *)pext->RequestAdr + i;
		kernel_tmp = (uint8_t *)pscsi_pass + i;
		copy_to_user(usr_temp, kernel_tmp, sizeof(EXT_SCSI_PASSTHRU));
	}

	scsi_direction = pscsi_pass->Direction;

	if (scsi_direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		DEBUG9(printk("qla2x00_sc_scsi_passthru: copying data.\n");)

		/* getting device data and putting in pext->ResponseAdr */
		ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
		    pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_sc_scsi_passthru: ERROR "
			    "verify write ResponseAdr.\n");)
			return ret;
		}

		/* now copy up the READ data to user */
		if ((CMD_COMPL_STATUS(pscsi_cmd) == CS_DATA_UNDERRUN) &&
		    (CMD_RESID_LEN(pscsi_cmd))) {

			transfer_len = pext->ResponseLen -
			    CMD_RESID_LEN(pscsi_cmd);

			pext->ResponseLen = transfer_len;
		} else {
			transfer_len = pext->ResponseLen;
		}

		usr_temp   = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		copy_to_user(usr_temp, kernel_tmp, transfer_len);
	}

	DEBUG9(printk("qla2x00_sc_scsi_passthru: exiting.\n");)
	return ret;
}

STATIC int
qla2x00_sc_fc_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int			ret = 0;
	fc_lun_t		temp_fclun;
	fc_lun_t		*fclun = NULL;
	fc_port_t		*fcport;
	os_lun_t		*lq;
	os_tgt_t		*tq;
	srb_t			*sp = NULL;
	uint8_t			*usr_temp, *kernel_tmp;
	uint32_t		i;

#if defined(QL_DEBUG_LEVEL_9)
	uint32_t		b, t, l;
#endif
	uint32_t		transfer_len;
	uint8_t			scsi_direction;

	static EXT_FC_SCSI_PASSTHRU	fc_scsi_pass;
	EXT_FC_SCSI_PASSTHRU	*pfc_scsi_pass = &fc_scsi_pass;

	static Scsi_Cmnd	fc_scsi_cmd;
	Scsi_Cmnd		*pfc_scsi_cmd = &fc_scsi_cmd;
	static Scsi_Device	fc_scsi_device;

	DEBUG9(printk("qla2x00_fc_scsi_passthru: entered.\n");)

	if ((sp = qla2x00_get_new_sp(ha)) == NULL) {

		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
		    "cannot alloc sp %p.\n", sp);)

		pext->Status = EXT_STATUS_NO_MEMORY;
		return pext->Status;
	}

	/* clear ioctl_sp and fc_scsi_cmd and fc_scsi_device to be used */
	memset(pfc_scsi_cmd, 0, sizeof(Scsi_Cmnd));
	memset(&fc_scsi_device, 0, sizeof(Scsi_Device));
	memset(ha->ioctl_mem, 0, ha->ioctl_mem_size);

	ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    sizeof(EXT_FC_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR verify READ "
		    "SCSI_FC_PASSTHRU.\n");)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return ret;
	}

	if (pext->ResponseLen > ha->ioctl_mem_size) {
		if (qla2x00_get_new_ioctl_dma_mem(ha, pext->ResponseLen) !=
		    QL_STATUS_SUCCESS) {

			DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
			    "cannot alloc requested DMA buffer size %x.\n",
			    pext->ResponseLen);)

			pext->Status = EXT_STATUS_NO_MEMORY;
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);
			return pext->Status;
		}
	}

	/* Copy request buffer */
	usr_temp   = (uint8_t *)pext->RequestAdr;
	kernel_tmp = (uint8_t *)pfc_scsi_pass;
	ret = copy_from_user(kernel_tmp, usr_temp,
			sizeof(EXT_FC_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
		    "copy_from_user() failed (%d).\n",
		    ret);)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return ret;
	}

	if (fc_scsi_pass.FCScsiAddr.DestType != EXT_DEF_DESTTYPE_WWPN) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR - "
					"wrong Dest type. \n");)
		ret = EXT_STATUS_ERR;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return ret;
	}

	fclun = NULL;
	for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
		if (memcmp(fcport->port_name,
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN, 8) != 0) {
			continue;

		}

		for (fclun = fcport->fclun; fclun; fclun = fclun->next) {
			if (fclun->lun == fc_scsi_pass.FCScsiAddr.Lun) {
				/* Found the right LUN */
				break;
			}
		}
		break;
	}

	if (fcport == NULL) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: FC AddrFormat - "
		    "DID NOT FIND Port for WWPN.\n");)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return pext->Status;
	}

	/* v5.21b9 - use a temporary fclun */
	if (fclun == NULL) {
		fclun = &temp_fclun;
		fclun->fcport = fcport;
		fclun->lun = fc_scsi_pass.FCScsiAddr.Lun;
		fclun->flags = 0;
		fclun->next = NULL;
	}

	/* set target coordinates */
	fc_scsi_cmd.target = 0xff; /* not used. just put something there. */
	fc_scsi_cmd.lun    = fc_scsi_pass.FCScsiAddr.Lun;

	DEBUG9(printk("qla2x00_fc_scsi_passthru: cmd for loopid=%04x L=%04x "
	    "WWPN=%02x%02x%02x%02x%02x%02x%02x%02x.\n",
	    fclun->fcport->loop_id, fc_scsi_cmd.lun,
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[0],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[1],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[2],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[3],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[4],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[5],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[6],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[7]);)

	if (pfc_scsi_pass->CdbLength == 6) {
		sp->cmd_length = 6;
		fc_scsi_cmd.cmd_len = 6;

	} else if (pfc_scsi_pass->CdbLength == 0x0A) {
		sp->cmd_length = 0x0A;
		fc_scsi_cmd.cmd_len = 0x0A;

	} else if (pfc_scsi_pass->CdbLength == 0x0C) {
		sp->cmd_length = 0x0C;
		fc_scsi_cmd.cmd_len = 0x0C;

	} else if (pfc_scsi_pass->CdbLength == 0x10) {
		sp->cmd_length = 0x10;
		fc_scsi_cmd.cmd_len = 0x10;

		/* Does the kernel support 16byte CDBs? */
		if (sp->cmd_length > MAX_COMMAND_SIZE) {
			/* No, use workaround method */
			fc_scsi_cmd.cmd_len = 0x0C;

			sp->more_cdb[0] = pfc_scsi_pass->Cdb[12];
			sp->more_cdb[1] = pfc_scsi_pass->Cdb[13];
			sp->more_cdb[2] = pfc_scsi_pass->Cdb[14];
			sp->more_cdb[3] = pfc_scsi_pass->Cdb[15];
		}
	} else {
		printk(KERN_WARNING
		    "qla2x00_ioctl: FC_SCSI_PASSTHRU Unknown Cdb Length=%x.\n",
		    pfc_scsi_pass->CdbLength);
		pext->Status = EXT_STATUS_INVALID_PARAM;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return pext->Status;
	}

	memcpy(fc_scsi_cmd.data_cmnd, pfc_scsi_pass->Cdb, fc_scsi_cmd.cmd_len);
	memcpy(fc_scsi_cmd.cmnd, pfc_scsi_pass->Cdb, fc_scsi_cmd.cmd_len);

	DEBUG9(printk("Dump of cdb buffer:\n");)
	DEBUG9(qla2x00_dump_buffer((uint8_t *)&fc_scsi_cmd.data_cmnd[0], 16);)

	pfc_scsi_cmd->host    = ha->host;
	sp->ha                = ha;
	sp->cmd               = pfc_scsi_cmd;
	sp->flags             = SRB_IOCTL;

	/* set local fc_scsi_cmd's sp pointer to sp */
	CMD_SP(pfc_scsi_cmd)  = (void *) sp;

	/* mark this as a special delivery and collection command */
	fc_scsi_cmd.flags     = 0;
	fc_scsi_cmd.scsi_done = qla2x00_scsi_pt_done;

	fc_scsi_cmd.device               = &fc_scsi_device;
	fc_scsi_cmd.device->tagged_queue = 0;
	fc_scsi_cmd.use_sg               = 0; /* no ScatterGather */
	fc_scsi_cmd.request_bufflen      = pext->ResponseLen;
	fc_scsi_cmd.request_buffer       = ha->ioctl_mem;
	fc_scsi_cmd.timeout_per_command  = QLA_PT_CMD_TOV * HZ;

	if (pfc_scsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		/* sending user data from pext->ResponseAdr to device */
		ret = verify_area(VERIFY_READ, (void *)pext->ResponseAdr,
				pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
			    "verify read ResponseAdr.\n");)
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);

			return pext->Status;
		}

		fc_scsi_cmd.sc_data_direction = SCSI_DATA_WRITE;
		usr_temp   = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		ret = copy_from_user(kernel_tmp, usr_temp, pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
			    "copy_from_user() failed (%d).\n",
			    ret);)
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);

			return pext->Status;
		}
	} else {
		fc_scsi_cmd.sc_data_direction = SCSI_DATA_READ;
	}

	tq = ha->ioctl->ioctl_tq;
	lq = ha->ioctl->ioctl_lq;

	if (fclun && tq && lq ) {
		tq->olun[fclun->lun] = lq;
		tq->ha = ha;

		lq->fclun = fclun;
		fcport = fclun->fcport;

		sp->lun_queue = lq;
		sp->tgt_queue = tq;
		sp->fclun = fclun;
	}

	DEBUG9({
		b = SCSI_BUS_32(pfc_scsi_cmd);
		t = SCSI_TCN_32(pfc_scsi_cmd);
		l = SCSI_LUN_32(pfc_scsi_cmd);
	})
	DEBUG9(printk("qla2x00_sc_scsi_passthru: ha instance=%ld tq=%p lq=%p "
	    "fclun=%p.\n",
	    ha->instance, tq, lq, fclun);)
	DEBUG9(printk("qla2x00_sc_scsi_passthru: CDB=%02x %02x %02x %02x; "
	    "b=%x t=%x l=%x.\n",
	    fc_scsi_cmd.cmnd[0], fc_scsi_cmd.cmnd[1], fc_scsi_cmd.cmnd[2],
	    fc_scsi_cmd.cmnd[3], b, t, l);)

	/*
	 * Check the status of the port
	 */
	if (qla2x00_check_port_status(ha, fcport) != QL_STATUS_SUCCESS) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return pext->Status;
	}

	/* set flag to indicate IOCTL SCSI PassThru in progress */
	ha->ioctl->SCSIPT_InProgress = 1;
	ha->ioctl->ioctl_tov = (int)QLA_PT_CMD_TOV + 1;

	/* prepare for receiving completion. */
	qla2x00_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(pfc_scsi_cmd) = (int) IOCTL_INVALID_STATUS;

	/* send command to adapter */
	DEBUG9(printk("qla2x00_fc_scsi_passthru(%ld): sending command.\n",
	    ha->host_no);)

	add_to_pending_queue(ha, sp);

	qla2x00_next(ha);

	ha->ioctl->cmpl_timer.expires = jiffies + ha->ioctl->ioctl_tov * HZ;
	add_timer(&ha->ioctl->cmpl_timer);

	down(&ha->ioctl->cmpl_sem);

	del_timer(&ha->ioctl->cmpl_timer);

	if (ha->ioctl->SCSIPT_InProgress == 1) {

		printk(KERN_WARNING
		    "qla2x00: scsi%ld ERROR passthru command timeout.\n",
		    ha->host_no);

		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return EXT_STATUS_ERR;
	}

	if (CMD_COMPL_STATUS(pfc_scsi_cmd) == (int)IOCTL_INVALID_STATUS) {

		DEBUG9(printk("qla2x00_fc_scsi_passthru(%ld): ERROR. "
		    "command not completed.\n", ha->host_no);)

		pext->Status = EXT_STATUS_ERR;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return EXT_STATUS_ERR;
	}

	switch (CMD_COMPL_STATUS(pfc_scsi_cmd)) {
	case CS_INCOMPLETE:
	case CS_ABORTED:
	case CS_PORT_UNAVAILABLE:
	case CS_PORT_LOGGED_OUT:
	case CS_PORT_CONFIG_CHG:
	case CS_PORT_BUSY:
		DEBUG10(printk("qla2x00_fc_scsi_passthru: cs err = %x.\n",
		    CMD_COMPL_STATUS(pfc_scsi_cmd));)
			ret = EXT_STATUS_ERR;
		pext->Status = EXT_STATUS_BUSY;
		break;
	}

	if ((CMD_COMPL_STATUS(pfc_scsi_cmd) == CS_DATA_UNDERRUN) ||
	    (CMD_SCSI_STATUS(pfc_scsi_cmd) != 0))  {

		/* have done the post function */
		pext->Status       = EXT_STATUS_SCSI_STATUS;
		/* The SDMAPI is only concerned with the low-order byte */
		pext->DetailStatus = CMD_SCSI_STATUS(pfc_scsi_cmd) & 0xff;

		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: data underrun or "
		    "scsi err. host status =0x%x, scsi status = 0x%x.\n",
		    CMD_COMPL_STATUS(pfc_scsi_cmd),
		    CMD_SCSI_STATUS(pfc_scsi_cmd));)

	} else if (CMD_COMPL_STATUS(pfc_scsi_cmd) != 0) {
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: cs err = %x. "
		    "copying ext stat %x\n",
		    CMD_COMPL_STATUS(pfc_scsi_cmd), pext->Status);)

		return pext->Status;
	}

	/* Process completed command */
	DEBUG9(printk("qla2x00_fc_scsi_passthru: complete. host status =0x%x, "
	    "scsi status = 0x%x.\n",
	    CMD_COMPL_STATUS(pfc_scsi_cmd), CMD_SCSI_STATUS(pfc_scsi_cmd));)

	/* copy up structure to make sense data available to user */
	pfc_scsi_pass->SenseLength = CMD_ACTUAL_SNSLEN(pfc_scsi_cmd);
	if (CMD_ACTUAL_SNSLEN(pfc_scsi_cmd)) {
		DEBUG9_10(printk("qla2x00_fc_scsi_passthru: sense[0]=%x "
		    "sense[2]=%x.\n",
		    pfc_scsi_cmd->sense_buffer[0],
		    pfc_scsi_cmd->sense_buffer[2]);)

		for (i = 0; i < CMD_ACTUAL_SNSLEN(pfc_scsi_cmd); i++) {
			pfc_scsi_pass->SenseData[i] =
			pfc_scsi_cmd->sense_buffer[i];
		}

		ret = verify_area(VERIFY_WRITE, (void *)pext->RequestAdr,
		    sizeof(EXT_FC_SCSI_PASSTHRU));
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
			    "verify WRITE RequestAdr.\n");)
			return ret;
		}

		usr_temp = (uint8_t *)pext->RequestAdr;
		kernel_tmp = (uint8_t *)pfc_scsi_pass;
		copy_to_user(usr_temp, kernel_tmp, sizeof(EXT_FC_SCSI_PASSTHRU));
	}

	scsi_direction = pfc_scsi_pass->Direction;

	if (scsi_direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {

		DEBUG9(printk("qla2x00_fc_scsi_passthru: copying data.\n");)

		/* getting device data and putting in pext->ResponseAdr */
		ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
		    pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;

			DEBUG9_10(printk("qla2x00_fc_scsi_passthru: ERROR "
			    "verify write ResponseAdr.\n");)

			return ret;
		}

		/* now copy up the READ data to user */
		if ((CMD_COMPL_STATUS(pfc_scsi_cmd) == CS_DATA_UNDERRUN) &&
		    (CMD_RESID_LEN(pfc_scsi_cmd))) {

			transfer_len = pext->ResponseLen -
			    CMD_RESID_LEN(pfc_scsi_cmd);

			pext->ResponseLen = transfer_len;
		} else {
			transfer_len = pext->ResponseLen;
		}

		usr_temp = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		copy_to_user(usr_temp, kernel_tmp, transfer_len);
	}

	return ret;
}

STATIC int
qla2x00_sc_scsi3_passthru(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
#define MAX_SCSI3_CDB_LEN	16

	int			ret = 0;
	fc_lun_t		temp_fclun;
	fc_lun_t		*fclun = NULL;
	fc_port_t		*fcport;
	os_lun_t		*lq;
	os_tgt_t		*tq;
	srb_t			*sp = NULL;
	uint8_t			*usr_temp, *kernel_tmp;
	uint32_t		transfer_len;
	uint32_t		i, b, t;
	uint32_t		scsi_direction;

	static EXT_FC_SCSI_PASSTHRU	fc_scsi_pass;
	EXT_FC_SCSI_PASSTHRU	*pfc_scsi_pass = &fc_scsi_pass;

	static Scsi_Cmnd	scsi3_cmd;
	Scsi_Cmnd		*pscsi3_cmd = &scsi3_cmd;
	static Scsi_Device	scsi3_device;


	DEBUG9(printk("qla2x00_scsi3_passthru: entered.\n");)

	if ((sp = qla2x00_get_new_sp(ha)) == NULL) {

		DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR "
		    "cannot alloc sp %p.\n", sp);)

		pext->Status = EXT_STATUS_NO_MEMORY;
		return pext->Status;
	}

	/* clear ioctl_sp and scsi3_cmd and scsi3_device to be used */
	memset(pscsi3_cmd, 0, sizeof(Scsi_Cmnd));
	memset(&scsi3_device, 0, sizeof(Scsi_Device));
	memset(ha->ioctl_mem, 0, ha->ioctl_mem_size);

	ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    sizeof(EXT_FC_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR verify READ "
		    "SCSI_FC_PASSTHRU.\n");)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return ret;
	}

	if (pext->ResponseLen > ha->ioctl_mem_size) {
		if (qla2x00_get_new_ioctl_dma_mem(ha, pext->ResponseLen) !=
		    QL_STATUS_SUCCESS) {

			DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR cannot "
			    "alloc requested DMA buffer size=%x.\n",
			    pext->ResponseLen);)

			pext->Status = EXT_STATUS_NO_MEMORY;
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);
			return pext->Status;
		}
	}

	/* Copy request buffer */
	usr_temp   = (uint8_t *)pext->RequestAdr;
	kernel_tmp = (uint8_t *)pfc_scsi_pass;
	ret = copy_from_user(kernel_tmp, usr_temp,
			sizeof(EXT_FC_SCSI_PASSTHRU));
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR "
		    "copy_from_user() failed (%d).\n",
		    ret);)
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return ret;
	}

	if (pfc_scsi_pass->FCScsiAddr.DestType != EXT_DEF_DESTTYPE_WWPN) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR - "
		    "wrong Dest type. \n");)
		ret = EXT_STATUS_ERR;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return ret;
	}

	/*
	 * For this ioctl command we always assume all 16 bytes are
	 * initialized.
	 */
	if (pfc_scsi_pass->CdbLength != MAX_SCSI3_CDB_LEN) {
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR - "
		    "wrong Cdb Length %d.\n", pfc_scsi_pass->CdbLength);)
		ret = EXT_STATUS_ERR;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return ret;
	}

	for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
		if (memcmp(fcport->port_name,
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN, 8) == 0) {
			break;
		}
	}
	if (fcport == NULL) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;

		DEBUG9_10(printk("qla2x00_scsi3_passthru: "
		    "DID NOT FIND Port for WWPN %02x%02x%02x%02x"
		    "%02x%02x%02x%02x.\n",
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[0],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[1],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[2],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[3],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[4],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[5],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[6],
		    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[7]);)

		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);

		return pext->Status;
	}

	/* Use a temporary fclun to send out the command. */
	fclun = &temp_fclun;
	fclun->fcport = fcport;
	fclun->lun = pfc_scsi_pass->FCScsiAddr.Lun;
	fclun->flags = 0;
	fclun->next = NULL;

	/* set target coordinates */
	scsi3_cmd.target = 0xff;  /* not used. just put something there. */
	scsi3_cmd.lun = pfc_scsi_pass->FCScsiAddr.Lun;

	DEBUG9(printk("qla2x00_scsi3_passthru: cmd for loopid=%04x L=%04x "
	    "WWPN=%02x%02x%02x%02x%02x%02x%02x%02x.\n",
	    fclun->fcport->loop_id, scsi3_cmd.lun,
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[0],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[1],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[2],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[3],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[4],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[5],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[6],
	    pfc_scsi_pass->FCScsiAddr.DestAddr.WWPN[7]);)

	sp->cmd_length = MAX_SCSI3_CDB_LEN;
	scsi3_cmd.cmd_len = MAX_SCSI3_CDB_LEN;

	/* Does the kernel support 16byte CDBs? */
	if (sp->cmd_length > MAX_COMMAND_SIZE) {
		/* No, use workaround method */
		scsi3_cmd.cmd_len = 0x0C;

		sp->more_cdb[0] = pfc_scsi_pass->Cdb[12];
		sp->more_cdb[1] = pfc_scsi_pass->Cdb[13];
		sp->more_cdb[2] = pfc_scsi_pass->Cdb[14];
		sp->more_cdb[3] = pfc_scsi_pass->Cdb[15];
	}

	memcpy(scsi3_cmd.data_cmnd, pfc_scsi_pass->Cdb, scsi3_cmd.cmd_len);
	memcpy(scsi3_cmd.cmnd, pfc_scsi_pass->Cdb, scsi3_cmd.cmd_len);

	DEBUG9(printk("qla2x00_scsi3_passthru: cdb buffer dump:\n");)
	DEBUG9(qla2x00_dump_buffer((uint8_t *)&scsi3_cmd.data_cmnd[0], 16);)

	pscsi3_cmd->host      = ha->host;
	sp->ha                = ha;
	sp->cmd               = pscsi3_cmd;
	sp->flags             = SRB_IOCTL;

	/* set local scsi3_cmd's sp pointer to sp */
	CMD_SP(pscsi3_cmd)    = (void *) sp;

	/* mark this as a special delivery and collection command */
	scsi3_cmd.flags     = 0;
	scsi3_cmd.scsi_done = qla2x00_scsi_pt_done;

	scsi3_cmd.device               = &scsi3_device;
	scsi3_cmd.device->tagged_queue = 0;
	scsi3_cmd.use_sg               = 0; /* no ScatterGather */
	scsi3_cmd.request_bufflen      = pext->ResponseLen;
	scsi3_cmd.request_buffer       = ha->ioctl_mem;
	scsi3_cmd.timeout_per_command  = QLA_PT_CMD_TOV * HZ;

	if (pfc_scsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		/* sending user data from pext->ResponseAdr to device */
		ret = verify_area(VERIFY_READ, (void *)pext->ResponseAdr,
		    pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR "
			    "verify read ResponseAdr.\n");)
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);
			return pext->Status;
		}

		scsi3_cmd.sc_data_direction = SCSI_DATA_WRITE;
		usr_temp   = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		ret = copy_from_user(kernel_tmp, usr_temp, pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR "
			    "copy_from_user() failed (%d).\n",
			    ret);)
			atomic_set(&sp->ref_count, 0);
			add_to_free_queue (ha, sp);
			return pext->Status;
		}

	} else {
		scsi3_cmd.sc_data_direction = SCSI_DATA_READ;
	}

	/* Use temporary LU and TGT queue */
	tq = ha->ioctl->ioctl_tq;
	lq = ha->ioctl->ioctl_lq;

	b = SCSI_BUS_32(pscsi3_cmd);
	t = SCSI_TCN_32(pscsi3_cmd);

	DEBUG9(printk("qla2x00_scsi3_passthru: ha instance=%ld tq=%p lq=%p "
	    "fclun=%p.\n",
	    ha->instance,tq,lq,fclun); )
	DEBUG9(printk("qla2x00_scsi3_passthru: CDB=%02x %02x %02x %02x; "
	    	"b=%x t=%x fclun=%x\n",
	    	scsi3_cmd.cmnd[0],scsi3_cmd.cmnd[1],scsi3_cmd.cmnd[2],
	    	scsi3_cmd.cmnd[3],b,t,SCSI_LUN_32(pscsi3_cmd));)

	if (tq && lq) {
		/*
		 * For now just save lq using the lower LUN byte value,
		 * even though this may not be the actual LUN number.
		 * Since we're only sending out passthru cmd one at a time,
		 * and only passthru is using FCP LUN format now, no need
		 * to change rest of driver just to decode the LUN.
		 */
		tq->olun[fclun->lun & 0xff] = lq;

		tq->ha = ha;
		lq->fclun = fclun;
		fcport = fclun->fcport;

		sp->lun_queue = lq;
		sp->tgt_queue = tq;
		sp->fclun = fclun;
	} else {
		lq = NULL;
		fcport = NULL;
	}

	/*
	 * Check the status of the port
	 */
	if (qla2x00_check_port_status(ha, fcport) != QL_STATUS_SUCCESS) {

		DEBUG9_10(printk("qla2x00_scsi3_passthru: port missing "
		    "or loop down. fcport=%p timer=%d state=%d dpc=%lx.\n",
		    fcport, atomic_read(&ha->loop_down_timer), ha->loop_state,
		    ha->dpc_flags);)

		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return pext->Status;
	}

	/* set flag to indicate IOCTL SCSI PassThru in progress */
	ha->ioctl->SCSIPT_InProgress = 1;
	ha->ioctl->ioctl_tov = (int)QLA_PT_CMD_TOV + 1;

	/* prepare for receiving completion. */
	qla2x00_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(pscsi3_cmd) = (int) IOCTL_INVALID_STATUS;

	/* send command to adapter */

	/*add_to_cmd_queue(ha, lq, sp);*/
	add_to_pending_queue(ha, sp);

	/*qla2x00_next(ha, tq, lq);*/
	qla2x00_next(ha);

	ha->ioctl->cmpl_timer.expires = jiffies + ha->ioctl->ioctl_tov * HZ;
	add_timer(&ha->ioctl->cmpl_timer);

	down(&ha->ioctl->cmpl_sem);

	del_timer(&ha->ioctl->cmpl_timer);

	if (ha->ioctl->SCSIPT_InProgress == 1) {

		printk(KERN_WARNING
		    "qla2x00: scsi%ld ERROR passthru command timeout.\n",
		    ha->host_no);

		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return EXT_STATUS_ERR;

	}
	if (CMD_COMPL_STATUS(pscsi3_cmd) == (int)IOCTL_INVALID_STATUS) {

		DEBUG9(printk("qla2x00_scsi3_passthru(%ld): ERROR - "
		    "command not completed.\n", ha->host_no);)

		pext->Status = EXT_STATUS_ERR;
		atomic_set(&sp->ref_count, 0);
		add_to_free_queue (ha, sp);
		return EXT_STATUS_ERR;
	}

	switch (CMD_COMPL_STATUS(pscsi3_cmd)) {
	case CS_INCOMPLETE:
	case CS_ABORTED:
	case CS_PORT_UNAVAILABLE:
	case CS_PORT_LOGGED_OUT:
	case CS_PORT_CONFIG_CHG:
	case CS_PORT_BUSY:
		DEBUG10(printk("qla2x00_scsi3_passthru: cs err = %x.\n",
		    CMD_COMPL_STATUS(pscsi3_cmd));)
		ret = EXT_STATUS_ERR;
		pext->Status = EXT_STATUS_BUSY;
		break;
	}

	if ((CMD_COMPL_STATUS(pscsi3_cmd) == CS_DATA_UNDERRUN) ||
	    (CMD_SCSI_STATUS(pscsi3_cmd) != 0)) {

		/* have done the post function */
		pext->Status       = EXT_STATUS_SCSI_STATUS;
		pext->DetailStatus = CMD_SCSI_STATUS(pscsi3_cmd);

		DEBUG9_10(printk("qla2x00_scsi3_passthru: data underrun or "
		    "scsi err. host status =0x%x, scsi status = 0x%x.\n",
		    CMD_COMPL_STATUS(pscsi3_cmd), CMD_SCSI_STATUS(pscsi3_cmd));)

	} else if (CMD_COMPL_STATUS(pscsi3_cmd) != 0) {
		DEBUG9_10(printk("qla2x00_scsi3_passthru: cs err = %x. "
		    "ext stat %x\n",
		    CMD_COMPL_STATUS(pscsi3_cmd), pext->Status);)
		return pext->Status;
	}

	/* Process completed command */
	DEBUG9(printk("qla2x00_scsi3_passthru: complete. host status =0x%x, "
	    "scsi status = 0x%x.\n",
	    CMD_COMPL_STATUS(pscsi3_cmd), CMD_SCSI_STATUS(pscsi3_cmd));)

	/* copy up structure to make sense data available to user */
	pfc_scsi_pass->SenseLength = CMD_ACTUAL_SNSLEN(pscsi3_cmd);
	if (CMD_ACTUAL_SNSLEN(pscsi3_cmd)) {
		DEBUG9_10(printk("qla2x00_scsi3_passthru: sense[0]=%x "
		    "sense[2]=%x.\n",
		    pscsi3_cmd->sense_buffer[0],
		    pscsi3_cmd->sense_buffer[2]);)

		for (i = 0; i < CMD_ACTUAL_SNSLEN(pscsi3_cmd); i++) {
			pfc_scsi_pass->SenseData[i] =
			    pscsi3_cmd->sense_buffer[i];
		}

		ret = verify_area(VERIFY_WRITE, (void *)pext->RequestAdr,
		    sizeof(EXT_FC_SCSI_PASSTHRU));
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR verify "
			    "WRITE RequestAdr.\n");)
			return ret;
		}

		usr_temp = (uint8_t *)pext->RequestAdr;
		kernel_tmp = (uint8_t *)pfc_scsi_pass;
		copy_to_user(usr_temp, kernel_tmp, sizeof(EXT_FC_SCSI_PASSTHRU));
	}

	scsi_direction = pfc_scsi_pass->Direction;

	if (scsi_direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {

		DEBUG9(printk("qla2x00_scsi3_passthru: copying data.\n");)

		/* getting device data and putting in pext->ResponseAdr */
		ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
		    pext->ResponseLen);
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;

			DEBUG9_10(printk("qla2x00_scsi3_passthru: ERROR verify "
			    "write ResponseAdr.\n");)

			return ret;
		}

		/* now copy up the READ data to user */
		if ((CMD_COMPL_STATUS(pscsi3_cmd) == CS_DATA_UNDERRUN) &&
		    (CMD_RESID_LEN(pscsi3_cmd))) {

			transfer_len = pext->ResponseLen -
			    CMD_RESID_LEN(pscsi3_cmd);

			pext->ResponseLen = transfer_len;
		} else {
			transfer_len = pext->ResponseLen;
		}

		usr_temp = (uint8_t *)pext->ResponseAdr;
		kernel_tmp = (uint8_t *)ha->ioctl_mem;
		copy_to_user(usr_temp, kernel_tmp, transfer_len);
	}

	return ret;
}

/*
 * qla2x00_send_els_rnid
 *	IOCTL to send extended link service RNID command to a target.
 *
 * Input:
 *	ha = adapter state pointer.
 *	pext = User space CT arguments pointer.
 *	mode = flags.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_send_els_rnid(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
#define TGT_DEV 	 1
#define HOST_DEV 	 2

	EXT_RNID_REQ	tmp_rnid;
	int		rval = 0;
	uint8_t 	dev_found = 0;
	uint16_t	dev_loop_id = 0;
	uint16_t	mb[MAILBOX_REGISTER_COUNT];
	uint32_t	copy_len;
	fc_port_t	*fcport;
	int		found;
	struct list_head *fcil;
	fc_initiator_t	*fcinitiator;

	DEBUG9(printk("qla2x00_send_els_rnid(%ld): entered.\n",
	    ha->host_no);)

	if (ha->ioctl_mem_size < SEND_RNID_RSP_SIZE) {
		if (qla2x00_get_new_ioctl_dma_mem(ha,
		    SEND_RNID_RSP_SIZE) != QL_STATUS_SUCCESS) {

			DEBUG9_10(printk("qla2x00_send_els_rnid(%ld): ERROR "
			    "cannot alloc DMA buffer. size=%x.\n",
			    ha->host_no, SEND_RNID_RSP_SIZE);)

			pext->Status = EXT_STATUS_NO_MEMORY;
			return QL_STATUS_ERROR;
		}
	}

	if (pext->RequestLen != sizeof(EXT_RNID_REQ)) {
		/* parameter error */
		DEBUG9_10(printk("qla2x00_send_els_rnid(%ld): invalid "
		    "request length %d.\n",
		    ha->host_no, pext->RequestLen);)
		pext->Status = EXT_STATUS_INVALID_PARAM;
		return QL_STATUS_ERROR;
	}

	rval = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    pext->RequestLen);

	if (rval != 0) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_send_els_rnid(%ld): request buf verify READ "
		    "FAILED.\n",
		    ha->host_no);)
		return QL_STATUS_ERROR;
	}

	DEBUG9(printk("qla2x00_send_els_rnid(%ld): request buf verified. "
	    " Copying request data.\n",
	    ha->host_no);)

	rval = copy_from_user(&tmp_rnid, pext->RequestAdr, pext->RequestLen);
	if (rval) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_send_els_rnid(%ld): ERROR copy_from_user()"
		    "failed (%d).\n",
		    ha->host_no,
		    rval);)
		return QL_STATUS_ERROR;
	}

	/* Find loop ID of the device */
	fcinitiator = NULL;
	switch (tmp_rnid.Addr.Type) {
	case EXT_DEF_TYPE_WWNN:

		DEBUG9(printk("qla2x00_send_els_rnid(%ld): got node name.\n",
		    ha->host_no);)

		for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
			/* if removed or missing */
			if (atomic_read(&fcport->state) == FC_ONLINE &&
			    memcmp((void *)tmp_rnid.Addr.FcAddr.WWNN,
			    (void *)fcport->node_name,
			    EXT_DEF_WWN_NAME_SIZE) == 0) {
				break;
			}
		}
		if (fcport != NULL) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " target device; loop_id=%x.\n",
			    ha->host_no, fcport->loop_id);)

			dev_found = TGT_DEV;
			dev_loop_id = fcport->loop_id;
			break;
		}

		found = 0;
		fcinitiator = NULL;
		list_for_each(fcil, &ha->fcinitiators) {
			fcinitiator = list_entry(fcil, fc_initiator_t, list);

			if (memcmp(tmp_rnid.Addr.FcAddr.WWNN,
				 fcinitiator->node_name,
				 EXT_DEF_WWN_NAME_SIZE) == 0 &&
				fcinitiator->d_id.b24 != 0) {

				found++;
				break;
			}
		}
		if (found) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " host device; loop_id=%x.\n",
			    ha->host_no, fcinitiator->loop_id);)

			dev_found = HOST_DEV;
			dev_loop_id = fcinitiator->loop_id;
			break;
		}

		break;

	case EXT_DEF_TYPE_WWPN:
		DEBUG9(printk("qla2x00_send_els_rnid(%ld): got port name.\n",
		    ha->host_no);)

		for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
			/* if removed or missing */
			if (atomic_read(&fcport->state) == FC_ONLINE &&
			    memcmp((void *)tmp_rnid.Addr.FcAddr.WWPN,
			    (void *)fcport->port_name,
			    EXT_DEF_WWN_NAME_SIZE) == 0) {
				break;
			}
		}
		if (fcport != NULL) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " target device; loop_id=%x.\n",
			    ha->host_no, fcport->loop_id);)

			dev_found = TGT_DEV; /* target device */
			dev_loop_id = fcport->loop_id;
			break;
		}

		found = 0;
		fcinitiator = NULL;
		list_for_each(fcil, &ha->fcinitiators) {
			fcinitiator = list_entry(fcil, fc_initiator_t, list);

			if (memcmp(tmp_rnid.Addr.FcAddr.WWPN,
				 fcinitiator->port_name,
				 EXT_DEF_WWN_NAME_SIZE) == 0 &&
				fcinitiator->d_id.b24 != 0) {

				found++;
				break;
			}
		}
		if (found) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " host device; loop_id=%x.\n",
			    ha->host_no, fcinitiator->loop_id);)

			dev_found = HOST_DEV;
			dev_loop_id = fcinitiator->loop_id;
			break;
		}

		break;

	case EXT_DEF_TYPE_PORTID:
		DEBUG9(printk("qla2x00_send_els_rnid(%ld): got port ID.\n",
		    ha->host_no);)

		/* PORTID bytes entered must already be big endian */
		for (fcport = ha->fcport; (fcport); fcport = fcport->next) {
			/* if removed or missing */
			if (atomic_read(&fcport->state) == FC_ONLINE &&
			    memcmp((void *)&tmp_rnid.Addr.FcAddr.Id[1],
			    (void *)(fcport->d_id.r.d_id),
			    EXT_DEF_PORTID_SIZE_ACTUAL) == 0) {
				break;
			}
		}
		if (fcport != NULL) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " target device; loop_id=%x.\n",
			    ha->host_no, fcport->loop_id);)

			dev_found = TGT_DEV; /* target device */
			dev_loop_id = fcport->loop_id;
			break;
		}

		found = 0;
		fcinitiator = NULL;
		list_for_each(fcil, &ha->fcinitiators) {
			fcinitiator = list_entry(fcil, fc_initiator_t, list);

			if (memcmp(&tmp_rnid.Addr.FcAddr.Id[1],
				&fcinitiator->d_id,
				EXT_DEF_PORTID_SIZE_ACTUAL) == 0) {

				found++;
				break;
			}
		}
		if (found) {
			DEBUG9(printk("qla2x00_send_els_rnid(%ld): found "
			    " host device; loop_id=%x.\n",
			    ha->host_no, fcinitiator->loop_id);)

			dev_found = HOST_DEV;
			dev_loop_id = fcinitiator->loop_id;
			break;
		}

		break;
	default:
		/* parameter error */
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk("qla2x00_send_els_rnid(%ld): invalid "
		    "addressing type.\n",
		    ha->host_no);)
		return pext->Status;
	}

	if (!dev_found ||
	    (dev_found == TGT_DEV && dev_loop_id > LAST_SNS_LOOP_ID)) {
		/* No matching device or the target device is not
		 * configured; just return error.
		 */
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		rval = pext->Status;
		DEBUG9_10(printk(
		    "qla2x00_send_els_rnid(%ld): device not found. "
		    "dev_found=%d dev_loop_id=%x.\n",
		    ha->host_no, dev_found, dev_loop_id);)
		return (rval);
	}

	/* check on loop down */
	if (ha->loop_state != LOOP_READY || 
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk(
		    "qla2x00_send_els_rnid(%ld): loop not ready.\n",
		    ha->host_no);)

		return pext->Status;
	}

	/* Check whether we need to login first. */
	if (dev_found == HOST_DEV && dev_loop_id > LAST_SNS_LOOP_ID) {
		/*
		 * Search for a usable loop ID before try to login to it.
		 */
		if ((dev_loop_id &= ~PORT_LOST_ID) > LAST_SNS_LOOP_ID) {
			/* Just start searching from first possible ID. */
			dev_loop_id = ha->min_external_loopid;
		}
		for (;;) {
			if (ha->fabricid[dev_loop_id].in_use == TRUE) {
				dev_loop_id++;
			} else {
				ha->fabricid[dev_loop_id].in_use = TRUE;
				break;
			}
		}

		DEBUG9(printk("qla2x00_send_els_rnid(%ld): try relogin to "
		    " host device; dev_loop_id=%x.\n",
		    ha->host_no, dev_loop_id);)

		for (;;) {
			if (dev_loop_id > LAST_SNS_LOOP_ID) {
				/* error */
				DEBUG10(printk("qla2x00_send_els_rnid(%ld): "
				    " no valid loop_id for login.\n",
				    ha->host_no);)

				break;
			}

			qla2x00_login_fabric(ha, 
			    dev_loop_id,
			    fcinitiator->d_id.b.domain,
			    fcinitiator->d_id.b.area,
			    fcinitiator->d_id.b.al_pa,
			    &mb[0], 0);

			if (mb[0] != MBS_CMD_CMP &&
			    mb[0] != MBS_PORT_ID_IN_USE &&
			    mb[0] != MBS_LOOP_ID_IN_USE) {

	 			DEBUG10(printk("qla2x00_send_els_rnid(%ld): "
				    "ERROR login mb[0]=%x mb[1]=%x.\n",
				    ha->host_no, mb[0], mb[1]);)
				break;
			}

			if (mb[0] == MBS_CMD_CMP) {
				DEBUG9(printk("qla2x00_send_els_rnid(%ld): "
				    " host login success; loop_id=%x.\n",
				    ha->host_no, dev_loop_id);)

				fcinitiator->loop_id = dev_loop_id;
				break;
			} else if (mb[0] == MBS_PORT_ID_IN_USE) {
				ha->fabricid[dev_loop_id].in_use = FALSE;
				dev_loop_id = mb[1];

				DEBUG9(printk("qla2x00_send_els_rnid(%ld): "
				    "port %06x using loop id=0x%04x.\n",
				    ha->host_no, ha->phost_db[host].d_id.b24,
				    dev_loop_id);)

				if (dev_loop_id <= LAST_SNS_LOOP_ID)
					ha->fabricid[dev_loop_id].in_use = TRUE;
				else
					/* Error */
					break;

			} else if (mb[0] == MBS_LOOP_ID_IN_USE) {
				/* Search for another usable loop_id */
				dev_loop_id++;
				while (ha->fabricid[dev_loop_id].in_use) {
					if (dev_loop_id++ > LAST_SNS_LOOP_ID) {
						/* Error */
						break;
					}
				}

				if (dev_loop_id <= LAST_SNS_LOOP_ID) {
					DEBUG9(printk(
					    "qla2x00_send_els_rnid(%ld): "
					    "previous loop id in use. Retry "
					    "with 0x%04x.\n",
					    ha->host_no, dev_loop_id);)

					ha->fabricid[dev_loop_id].in_use = TRUE;
				} else {
					/* Error */
					break;
				}
			}
		}

		if (mb[0] != MBS_CMD_CMP) {
			pext->Status = EXT_STATUS_ERR;
			DEBUG9_10(printk(
			    "qla2x00_send_els_rnid(%ld): login failed.\n",
			    ha->host_no);)

			return pext->Status;
		}
	}

	/* Send command */
	DEBUG9(printk("qla2x00_send_els_rnid(%ld): sending rnid cmd.\n",
	    ha->host_no);)

	rval = qla2x00_send_rnid_mbx(ha, dev_loop_id,
	    (uint8_t)tmp_rnid.DataFormat, ha->ioctl_mem_phys,
	    SEND_RNID_RSP_SIZE, &mb[0]);

	if (rval != QLA2X00_SUCCESS) {
		/* error */
		pext->Status = EXT_STATUS_ERR;

                DEBUG9_10(printk(
                    "qla2x00_send_els_rnid(%ld): FAILED. rval = %x.\n",
                    ha->host_no, mb[0]);)
		return (0);
	}

	DEBUG9(printk("qla2x00_send_els_rnid(%ld): rnid cmd sent ok.\n",
	    ha->host_no);)

	/* Copy the response */
	copy_len = (pext->ResponseLen > SEND_RNID_RSP_SIZE) ?
	    SEND_RNID_RSP_SIZE : pext->ResponseLen;

	rval = verify_area(VERIFY_WRITE, (void  *)pext->ResponseAdr,
	    copy_len);

	if (rval != 0) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_send_els_rnid(%ld): response buf verify WRITE "
		    "failed.\n",
		    ha->host_no);)
		rval = EFAULT;
	} else {
		copy_to_user((uint8_t *)pext->ResponseAdr,
		    (uint8_t *)ha->ioctl_mem, copy_len);

		if (SEND_RNID_RSP_SIZE > pext->ResponseLen) {
			pext->Status = EXT_STATUS_DATA_OVERRUN;
			DEBUG9(printk(
	 		    "qla2x00_send_els_rnid(%ld): data overrun. "
			    "exiting normally.\n",
			    ha->host_no);)
		} else {
			pext->Status = EXT_STATUS_OK;
			DEBUG9(printk(
			    "qla2x00_send_els_rnid(%ld): exiting normally.\n",
			    ha->host_no);)
		}
		pext->ResponseLen = copy_len;
		rval = 0;
	}

	return (rval);
}

static int
qla2x00_get_rnid_params(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		rval = 0;
	uint32_t	copy_len;
	uint16_t	mb[MAILBOX_REGISTER_COUNT];

	DEBUG9(printk("qla2x00_get_rnid_params(%ld): entered.\n",
	    ha->host_no);)

	/* check on loop down */
	if (ha->loop_state != LOOP_READY || 
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk(
		    "qla2x00_get_rnid_params(%ld): loop not ready.\n",
		    ha->host_no);)

		return pext->Status;
	}

	/* Send command */
	rval = qla2x00_get_rnid_params_mbx(ha, ha->ioctl_mem_phys,
	    sizeof(EXT_RNID_DATA), &mb[0]);

	if (rval != QLA2X00_SUCCESS) {
		/* error */
		pext->Status = EXT_STATUS_ERR;

		DEBUG9_10(printk(
		    "qla2x00_get_rnid_params(%ld): cmd FAILED=%x.\n",
		    ha->host_no, mb[0]);)
		return (0);
	}

	/* Copy the response */
	copy_len = (pext->ResponseLen > sizeof(EXT_RNID_DATA)) ?
	    (uint32_t)sizeof(EXT_RNID_DATA) : pext->ResponseLen;
	rval = verify_area(VERIFY_WRITE, (void  *)pext->ResponseAdr,
	    copy_len);

	if (rval != 0) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_get_rnid_params(%ld): verify WRITE response buf "
		    "failed.\n",
		    ha->host_no);)
		rval = EFAULT;
	} else {
		copy_to_user((void *)pext->ResponseAdr, (void *)ha->ioctl_mem,
		    copy_len);

		pext->ResponseLen = copy_len;
		if (copy_len < sizeof(EXT_RNID_DATA)) {
			pext->Status = EXT_STATUS_DATA_OVERRUN;
			DEBUG9_10(printk(
			    "qla2x00_get_rnid_params(%ld): data overrun. "
			    "exiting normally.\n",
			    ha->host_no);)
 		} else if (pext->ResponseLen > sizeof(EXT_RNID_DATA)) {
 			pext->Status = EXT_STATUS_DATA_UNDERRUN;
 			DEBUG9_10(printk(
			    "qla2x00_get_rnid_params(%ld): data underrun. "
 			    "exiting normally.\n",
 			    ha->host_no);)
		} else {
			pext->Status = EXT_STATUS_OK;
			DEBUG9(printk(
			    "qla2x00_get_rnid_params(%ld): exiting normally.\n",
			    ha->host_no);)
		}
		rval = 0;
	}

	return (rval);
}

/*
 * qla2x00_set_host_data
 *	IOCTL command to set host/adapter related data.
 *
 * Input:
 *	ha = adapter state pointer.
 *	pext = User space CT arguments pointer.
 *	mode = flags.
 *
 * Returns:
 *	0 = success
 *
 * Context:
 *	Kernel context.
 */
STATIC int
qla2x00_set_host_data(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int	rval = 0;

	DEBUG9(printk("qla2x00_set_host_data(%ld): entered.\n",
	    ha->host_no);)

	/* check on loop down */
	if (ha->loop_state != LOOP_READY || 
		test_bit(CFG_ACTIVE, &ha->cfg_flags) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk(
		    "qla2x00_set_host_data(%ld): loop not ready.\n",
		    ha->host_no);)

		return pext->Status;
	}

	/* switch on command subcode */
	switch (pext->SubCode) {
	case EXT_SC_SET_RNID:
		rval = qla2x00_set_rnid_params(ha, pext, mode);
		break;
	default:
		/* function not supported. */
		pext->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		rval = ENODEV;
		break;
	}

	DEBUG9(printk("qla2x00_set_host_data(%ld): exiting.\n",
	    ha->host_no);)

	return (rval);
}

STATIC int
qla2x00_set_rnid_params(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	EXT_SET_RNID_REQ	tmp_set;
	EXT_RNID_DATA	*tmp_buf;
	int		rval = 0;
	uint16_t	mb[MAILBOX_REGISTER_COUNT];

	DEBUG9(printk("qla2x00_set_rnid_params(%ld): entered.\n",
	    ha->host_no);)

	if (pext->RequestLen != sizeof(EXT_SET_RNID_REQ)) {
		/* parameter error */
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk("qla2x00_set_rnid_params(%ld): invalid "
		    "request length.\n",
		    ha->host_no);)
		return(0);
	}

	rval = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
	    pext->RequestLen);

	if (rval != 0) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_set_rnid_params(%ld): verify READ request buf.\n",
		    ha->host_no);)
		return(EFAULT);
	}

	rval = copy_from_user(&tmp_set, pext->RequestAdr, pext->RequestLen);
	if (rval) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_set_rnid_params(%ld): ERROR copy_from_user() "
		    "failed (%d)\n", 
		    ha->host_no,
		    rval);)
		return(EFAULT);
	}

	rval = qla2x00_get_rnid_params_mbx(ha, ha->ioctl_mem_phys,
	    sizeof(EXT_RNID_DATA), &mb[0]);
	if (rval != QLA2X00_SUCCESS) {
		/* error */
		pext->Status = EXT_STATUS_ERR;

                DEBUG9_10(printk(
                    "qla2x00_set_rnid_params(%ld): read cmd FAILED=%x.\n",
                    ha->host_no, mb[0]);)
		return (0);
	}

	tmp_buf = (EXT_RNID_DATA *)ha->ioctl_mem;
	/* Now set the params. */
	memcpy(tmp_buf->IPVersion, tmp_set.IPVersion, 2);
	memcpy(tmp_buf->UDPPortNumber, tmp_set.UDPPortNumber, 2);
	memcpy(tmp_buf->IPAddress, tmp_set.IPAddress, 16);
	rval = qla2x00_set_rnid_params_mbx(ha, ha->ioctl_mem_phys,
	    sizeof(EXT_RNID_DATA), &mb[0]);

	if (rval != QLA2X00_SUCCESS) {
		/* error */
		pext->Status = EXT_STATUS_ERR;

		DEBUG9_10(printk(
		    "qla2x00_set_rnid_params(%ld): set cmd FAILED=%x.\n",
		    ha->host_no, mb[0]);)
		rval = 0;
	} else {
		pext->Status = EXT_STATUS_OK;
		DEBUG9(printk(
		    "qla2x00_set_rnid_params(%ld): exiting normally.\n",
		    ha->host_no);)
	}

	return (rval);
}

STATIC void
qla2x00_waitq_sem_timeout(unsigned long data)
{
	wait_q_t *tmp_ptr = (wait_q_t *)data;

	DEBUG9(printk("qla2x00_sem_timeout: entered.\n");)

	if (tmp_ptr != NULL)
		up(&tmp_ptr->wait_q_sem);

	DEBUG9(printk("qla2x00_sem_timeout: exiting.\n");)
}

/*
 *  tov = timeout value in seconds
 */
STATIC uint8_t
qla2x00_get_ioctl_access(scsi_qla_host_t *ha, uint32_t tov)
{
	int		prev_val = 1;
	uint8_t		ret;
	unsigned long	cpu_flags;
	struct timer_list	tmp_access_timer;
	wait_q_t	*ptmp_wq = NULL;


	DEBUG9(printk("qla2x00_get_ioctl_access(%ld): entered.\n",
	    ha->host_no);)

	while (1) {
		if (test_bit(IOCTL_WANT, (void *)&(ha->ioctl->access_bits)) ==
		    0) {

			DEBUG9(printk("qla2x00_get_ioctl_access(%ld): going "
			    " to test access_bits.\n", ha->host_no);)

			/* No one else is waiting. Go ahead and try to
			 * get access.
			 */
			if ((prev_val = test_and_set_bit(IOCTL_ACTIVE,
			    (void *)&ha->ioctl->access_bits)) == 0) {
				break;
			}
		}

		/* wait for previous command to finish */
		DEBUG9(printk("qla2x00_get_ioctl_access(%ld): access_bits=%x. "
		    "busy. Waiting for access. curr time=0x%lx.\n",
		    ha->host_no, ha->ioctl->access_bits, jiffies);)

		/*
		 * Init timer and get semaphore from wait_q. if we got valid
		 * semaphore pointer the IOCTL_WANT flag would also had
		 * been set.
		 */
		qla2x00_wait_q_add(ha, &ptmp_wq);

		if (ptmp_wq == NULL) {
			/* queue full? problem? can't proceed. */
			DEBUG9_10(printk("qla2x00_get_ioctl_access(%ld): ERROR "
			    "no more wait_q allowed. exiting.\n", ha->host_no);)

			break;
		}

		init_timer(&tmp_access_timer);

		tmp_access_timer.data = (unsigned long)ptmp_wq;
		tmp_access_timer.function =
		    (void (*)(unsigned long))qla2x00_waitq_sem_timeout;
		tmp_access_timer.expires = jiffies + tov * HZ;

		DEBUG9(printk("get_ioctl_access(%ld): adding timer. "
		    "curr time=0x%lx timeoutval=0x%lx.\n",
		    ha->host_no, jiffies, tmp_access_timer.expires);)

		/* wait. */
		add_timer(&tmp_access_timer);

		DEBUG9(printk("get_ioctl_access(%ld): going to sleep. current "
		    "time=0x%lx.\n", ha->host_no, jiffies);)

		down_interruptible(&ptmp_wq->wait_q_sem);

		DEBUG9(printk("get_ioctl_access(%ld): woke up. current "
		    "time=0x%lx.\n", ha->host_no, jiffies);)

		del_timer(&tmp_access_timer);

		/* try to get lock again. we'll test later to see
		 * if we actually got the lock.
		 */
		prev_val = test_and_set_bit(IOCTL_ACTIVE,
		    (void *)&(ha->ioctl->access_bits));

		/*
		 * After we tried to get access then we check to see
		 * if we need to clear the IOCTL_WANT flag. Don't clear
		 * this flag before trying to get access or another
		 * new thread might grab it before we did.
		 */
		spin_lock_irqsave(&ha->ioctl->wait_q_lock, cpu_flags);
		if (ha->ioctl->wait_q_head == NULL) {
			/* We're the last thread in queue. */
			clear_bit(IOCTL_WANT, (void *)&ha->ioctl->access_bits);
		}
		qla2x00_wait_q_memb_free(ha, ptmp_wq);
		spin_unlock_irqrestore(&ha->ioctl->wait_q_lock, cpu_flags);

		break;
	}

	if (prev_val == 0) {
		/* We got the lock */

		DEBUG9(printk("qla2x00_get_ioctl_access(%ld): got access.\n",
					ha->host_no);)

		ret = QL_STATUS_SUCCESS;
	} else {
		/* Timeout or resource error. */
		DEBUG9_10(printk("qla2x00_get_ioctl_access(%ld): timed out "
		    "or wait_q error.\n", ha->host_no);)

		ret = QL_STATUS_TIMEOUT;
	}

	return ret;
}

STATIC uint8_t
qla2x00_release_ioctl_access(scsi_qla_host_t *ha)
{
	wait_q_t	*next_thread = NULL;

	DEBUG9(printk("qla2x00_release_ioctl_access:(%ld): entered.\n",
	    ha->host_no);)

	clear_bit(IOCTL_ACTIVE, (void *)&(ha->ioctl->access_bits));

	/* Wake up one pending ioctl thread in wait_q */
	qla2x00_wait_q_remove(ha, &next_thread);
	if (next_thread) {
		DEBUG9(printk("qla2x00_release_ioctl_access: found wait_q. "
		    "Waking up waitq %p.\n", &next_thread);)
		up(&next_thread->wait_q_sem);
	}

	DEBUG9(printk("qla2x00_release_ioctl_access:(%ld): exiting.\n",
	    ha->host_no);)

	return QL_STATUS_SUCCESS;
}

/* Find a free wait_q member from the array. Must already got the
 * wait_q_lock spinlock.
 */
STATIC void
qla2x00_wait_q_memb_alloc(scsi_qla_host_t *ha, wait_q_t **ret_wait_q_memb)
{
	uint8_t		i;
	wait_q_t	*ptmp = NULL;

	DEBUG9(printk("qla2x00_wait_q_memb_alloc: entered. "
	    "Inst=%d.\n", apiHBAInstance);)

	for (i = 0; i < MAX_IOCTL_WAIT_THREADS; i++) {
		if (!(ha->ioctl->wait_q_arr[i].flags & WQ_IN_USE)) {
			ha->ioctl->wait_q_arr[i].flags |= WQ_IN_USE;
			ptmp = &ha->ioctl->wait_q_arr[i];
			break;
		}
	}

	*ret_wait_q_memb = ptmp;

	DEBUG9(printk("qla2x00_wait_q_memb_alloc: return waitq_memb=%p. "
	    "Inst=%d.\n", *ret_wait_q_memb, apiHBAInstance);)
}

/* Free the specified wait_q member. Must already got the wait_q_lock
 * spinlock.
 */
STATIC void
qla2x00_wait_q_memb_free(scsi_qla_host_t *ha, wait_q_t *pfree_wait_q_memb)
{
	DEBUG9(printk("qla2x00_wait_q_memb_free: entered. "
	    "Inst=%d.\n", apiHBAInstance);)

	if (pfree_wait_q_memb != NULL)
		pfree_wait_q_memb->flags &= ~WQ_IN_USE;

	DEBUG9(printk("qla2x00_wait_q_memb_free: exiting. "
	    "Inst=%d.\n", apiHBAInstance);)
}

/* Allocates a wait_q_t struct and add to the wait_q list. */
STATIC uint8_t
qla2x00_wait_q_add(scsi_qla_host_t *ha, wait_q_t **ret_wq)
{
	uint8_t		ret;
	unsigned long	cpu_flags;
	wait_q_t	*ptmp = NULL;

	spin_lock_irqsave(&ha->ioctl->wait_q_lock, cpu_flags);

	DEBUG9(printk("qla2x00_wait_q_add: got wait_q spinlock. "
	    "Inst=%d.\n", apiHBAInstance);)

	qla2x00_wait_q_memb_alloc(ha, &ptmp);
	if (ptmp == NULL) {
		/* can't add any more threads */
		DEBUG9_10(printk("qla2x00_wait_q_add: ERROR no more "
		    "ioctl threads allowed. Inst=%d.\n", apiHBAInstance);)

		ret = QL_STATUS_RESOURCE_ERROR;
	} else {
		if (ha->ioctl->wait_q_tail == NULL) {
			/* First thread to queue. */
			set_bit(IOCTL_WANT, (void *)&ha->ioctl->access_bits);

			ha->ioctl->wait_q_head = ptmp;
		} else {
			ha->ioctl->wait_q_tail->pnext = ptmp;
		}
		ha->ioctl->wait_q_tail = ptmp;

		*ret_wq = ptmp;

		/* Now init the semaphore */

		init_MUTEX_LOCKED(&ptmp->wait_q_sem);

		ret = QL_STATUS_SUCCESS;
	}

	DEBUG9(printk("qla2x00_wait_q_add: going to release spinlock. "
	    "ret_wq=%p, ret=%d. Inst=%d.\n", *ret_wq, ret, apiHBAInstance);)

	spin_unlock_irqrestore(&ha->ioctl->wait_q_lock, cpu_flags);

	return ret;
}

/* Just remove one member from wait_q.  Don't free anything. */
STATIC void
qla2x00_wait_q_remove(scsi_qla_host_t *ha, wait_q_t **ret_wq)
{
	unsigned long	cpu_flags;

	spin_lock_irqsave(&ha->ioctl->wait_q_lock, cpu_flags);

	DEBUG9(printk("qla2x00_wait_q_remove: got wait_q spinlock. "
	    "Inst=%d.\n", apiHBAInstance);)

	/* Remove from head */
	*ret_wq = ha->ioctl->wait_q_head;
	if (ha->ioctl->wait_q_head != NULL) {
		ha->ioctl->wait_q_head = ha->ioctl->wait_q_head->pnext;
		if (ha->ioctl->wait_q_head == NULL) {
			/* That's the last one in queue. */
			ha->ioctl->wait_q_tail = NULL;
		}
		(*ret_wq)->pnext = NULL;
	}

	DEBUG9(printk("qla2x00_wait_q_remove: return ret_wq=%p. Going to "
	    "release spinlock. Inst=%d.\n", *ret_wq, apiHBAInstance);)

	spin_unlock_irqrestore(&ha->ioctl->wait_q_lock, cpu_flags);
}


