/*
 * linux/fs/lif/super.c
 *
 *
 *
 */
#include <linux/module.h>

#include <stdarg.h>

#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#include <linux/errno.h>
#include <linux/fs.h>

#include <linux/malloc.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/blkdev.h>
#include <linux/init.h>

#include <asm/iodc.h>

#define MAX_LIF_FILES		64
#define LIF_ROOT_INO		1
#define  ROUND_UP(x,y)       ((((x) + ((y) - 1)) / (y)) * (y))

/* Static memory for inode information */
typedef struct file_inode_info
{
    char	fileName[MAXFILENAME+1]; /* Extra character for null termination */
    struct inode *inode;
}file_inode_info;

/* Note that index 0 is not used and index 1 is not used since the rooot
 * inode has a value of 1. The first lif file gets an inode of 2 and then
 * on the next number...
 */

file_inode_info	FileInodeInfo[MAX_LIF_FILES+2];
static		NextInodeIndx = 2; /* First inode number for LIF files */

/* LIF inode operation */
struct dentry *lif_lookup(struct inode *dir,struct dentry *dentry);
/* LIF File operations */
ssize_t lif_file_read(struct file *filp, char *buf, size_t count, loff_t *ppos);
int lif_file_release(struct inode *inode, struct file *filp);

/* This global variable is initialises in crt0.s. The loader loads the LIF
 * volume ( including directory entries ) in memory and passes the address
 * of its start to the kernel */
extern long lifVolDir;	

/* Note that operations in this structure is not really needed for the
 * LIF file system. lif_read_super() takes care of setting up the root
 * inode and inodes for the files and the lookup() function in the file
 * inode does the rest.
 */
static struct super_operations lif_sops = {
	NULL, /*lif_read_inode*/
	NULL, /*lif_write_inode*/
	NULL, /*lif_put_inode*/
	NULL, /*lif_delete_inode*/
	NULL, /*lif_notify_change*/
	NULL, /*lif_put_super*/
	NULL, /*lif_write_super*/
	NULL, /*lif_statfs*/
	NULL /*lif_remount*/
};


/* Directory operations needed for LIF.Add any more operations if needed
 * by VFS */
struct file_operations lif_dir_operations = {
	NULL,			/* lseek - default */
	NULL,			/* read */
	NULL,			/* write - bad */
	NULL,			/* readdir */
	NULL,			/* select v2.0.x/poll v2.1.x - default */
	NULL,			/* ioctl - default */
	NULL,			/* mmap */
	NULL,			/* no special open code */
	NULL,			/* flush */
	NULL,			/* no special release code */
	file_fsync,		/* fsync */
        NULL,			/* fasync */
        NULL,			/* check_media_change */
        NULL,			/* revalidate */
        NULL,			/* lock */
};

/* The public inode operations for the lif fs directory */
struct inode_operations lif_dir_inode_operations = {
	&lif_dir_operations,	/* default directory file-ops */
	NULL,			/* create */
	lif_lookup,		/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL,                   /* smap */
	NULL,                   /* updatepage */
	NULL,                   /* revalidate */
};


/* These are needed for the file operations to work */
static struct file_operations lif_file_operations = {
	NULL,			/* lseek - default */
	lif_file_read,		/* lif file read function */
	NULL,			/* write */
	NULL,			/* readdir - bad */
	NULL,			/* select v2.0.x/poll v2.1.x - default */
	NULL,			/* ioctl - default */
	generic_file_mmap,	/* mmap */
	NULL,			/* no special open is needed */
	NULL,			/* flush */
	lif_file_release,	/* release */
	file_fsync,		/* fsync */
        NULL,			/* fasync */
        NULL,			/* check_media_change */
        NULL,			/* revalidate */
        NULL,			/* lock */
};

/* Don't think any of the inode specific operations are needed,
 * check later??
 */
struct inode_operations lif_file_inode_operations = {
	&lif_file_operations,	/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL,			/* permission */
	NULL,			/* smap */
        NULL,			/* updatepage */
        NULL,			/* revalidate */
        
};


/* LIF Super block read function. In addition to creating a superblock,
 * this function also creates the inodes for the LIF files. Note that
 * initialisation of inodes may not be complete if the VFS block buffer
 * mechanism is used to read LIF files. Currently, reading LIF files
 * bypasses the block buffer and all reads are managed here.
 */
struct super_block * lif_read_super (struct super_block * sb, void * data,
				      int silent)
{
    struct inode *root_inode;

    MOD_INC_USE_COUNT;


    /* First, setup the super block fields */
    sb->s_flags = MS_RDONLY;
    sb->s_op =&lif_sops;

    /* First initialise the internal data structures */
    memset( FileInodeInfo, 0, MAX_LIF_FILES * ( sizeof(file_inode_info)) );

    printk("lif_read_super():Internal memory initialised\n");
        
    /* Create the root inode for the file system */
    root_inode=get_empty_inode();
    printk("lif_read_super():Obtained the root inode for fs\n");
    if (!root_inode)
    {
	printk("Could not create root inode\n");
        MOD_DEC_USE_COUNT;
        return NULL;
    }

    printk("lif_read_super():Initialising root inode\n");
    /* make sure that all the relevant fields of the root inode are 
     * initialised. TBD */
    root_inode->i_sb = sb;
    root_inode->i_dev = sb->s_dev;
    root_inode->i_ino = LIF_ROOT_INO;


    root_inode->i_uid = 0;	/* TBD Check?? */
    root_inode->i_gid = 0;	/* TBD Check?? */
    root_inode->i_version = ++event; /* Guess this needs to be done */
    root_inode->i_mode = S_IFDIR | S_IRUGO | S_IXUGO ; /* RX perm */
    root_inode->i_op = &lif_dir_inode_operations;

    /* Not sure what this needs to be ?? */
    root_inode->i_size = 0; 
			

    /* Currently uses the LIF's block size */
    root_inode->i_blksize = LIFAU;

    /* No physical blocks needed for root inode */
    root_inode->i_blocks = 0;

    root_inode->i_mtime = root_inode->i_atime = root_inode->i_ctime = 0;
    /* No extra links */
    root_inode->i_nlink = 1 ; 


    /* Not sure if any more fields need to be set */


    insert_inode_hash(root_inode);
    sb->s_root = d_alloc_root(root_inode, NULL);

    if (!sb->s_root)
    {
	printk("get root inode failed\n");
	iput(root_inode);
        MOD_DEC_USE_COUNT;
        return NULL;
    }
 
    /* Now read the LIF directory entries from the reserved memory
    * area and fill in the internal data structures and also create
    * the necessary inodes and its information.
    */

    printk("lif_read_super():Creating inodes for LIF\n");
    if ( create_lif_inodes(sb) == -1 )
    {
        printk("Could not create lif file inodes\n");
	iput(root_inode);
        MOD_DEC_USE_COUNT;
        return NULL;
    }
        

    /* Other fields of the superblock may have to be set */
    return sb;
}

/* Read the LIF directory entries and create their corresponding
 * inodes. The lookup() function will use the internal data
 * structure that also gets set in this function
 */

int create_lif_inodes(struct super_block *sb)
{
    struct lvol 	*lifVol;
    struct direntry 	*lifDirEntry;
    int			numLifFiles,i;
    char		*lifFileName;
    
    printk("create_lif_inodes(): Reading LIF info\n");
    lifVol = (struct lvol *)lifVolDir;

    /* Add a Check to see if the number of files exceeds the array size */
    numLifFiles = (lifVol->dsize*LIFAU) / (sizeof (struct direntry) ) ;
    printk("Number of LIF Files = %d\n",numLifFiles);

    /* Get the start of the directory entry */
    lifDirEntry = (struct direntry *)( lifVolDir + ( (lifVol->dstart) * LIFAU) );
    for (i=0; i< numLifFiles; i++)
    {
        /* Create an entry only for RAGE type files only */
        if ( lifDirEntry->ftype == RAGE_FILE_TYPE )
        {
           /* Return error code after checking the return results TBD */
           if ( generate_lif_inodes(lifDirEntry,sb) )
           {
              return -1;
           }
        }
        /* Get successive directory entries */
	lifDirEntry++; 
    }

    return 0;

}

int generate_lif_inodes(struct direntry *lifDirEntry, struct super_block *sb)
{
    struct inode *file_inode;
    int i;
	
    /* Create the file inode for LIF files */
    file_inode=get_empty_inode();
    if (!file_inode)
    {
	printk("Could not create file inode\n");
        return -1;
    }

    /* make sure that all the relevant fields of the inode are 
     * initialised. TBD */
    file_inode->i_sb = sb;
    file_inode->i_dev = sb->s_dev;
    file_inode->i_ino = NextInodeIndx;


    file_inode->i_uid = 0;	/* TBD Check?? */
    file_inode->i_gid = 0;	/* TBD Check?? */
    file_inode->i_version = ++event; /* Guess this needs to be done */
    file_inode->i_mode = S_IRUGO | S_IXUGO ; /* RX perm */
    file_inode->i_op = &lif_file_inode_operations;

    /* Not sure what this needs to be ?? */
    file_inode->i_size = 0; 
			

    /* Note that the block size and number of blocks are from the LIF format
     * Not sure if VFS needs a different format */
    /* Same about this ??? */
    file_inode->i_blksize = LIFAU ;

    /* Get this from the LIF directory */
    file_inode->i_blocks = lifDirEntry->size;

    /* Get this from the LIF directory?? */
    /* The times need to be in Linux format. Right now, there is no conversion
     * method from LIF to Linux */
    file_inode->i_mtime = file_inode->i_atime = file_inode->i_ctime = 0;
    /* No extra links ??? */
    file_inode->i_nlink = 1 ; 

    /* Add the LIF Union specific information */
    file_inode->u.lif_i.file_start = lifDirEntry->start;

    /* Not sure if any more fields need to be set */

    /* First, copy the inode and filename in the internal table */

    /* Note that the file name stored is a null terminated string */
    /* LIF filenames have leading spaces */
    for(i=0; i<MAXFILENAME; i++)
    {
        if ( lifDirEntry->fname[i] == 0x20 )
        {
	    break;
        }
        FileInodeInfo[NextInodeIndx].fileName[i] = lifDirEntry->fname[i] ;
    }
    FileInodeInfo[NextInodeIndx].fileName[i] = 0x00; /* Null terminate */

    FileInodeInfo[NextInodeIndx].inode = file_inode;
    NextInodeIndx++;
    
    /* Not sure if this is needed */
    insert_inode_hash(file_inode);

    return 0;

}


/***** Get inode using directory and name */
struct dentry *lif_lookup(struct inode *dir,struct dentry *dentry)
{
	struct super_block *sb = dir->i_sb;
	struct inode *inode = NULL;
	int ino,i;
	
        /* Index 0 and 1 are not used in the internal array */
        /* Also note that NextInodeIndx holds the maximum number of inodes created
         * statically during mounting */
        ino = 0;
	for(i = 2; i<NextInodeIndx; i++)
        {
	   if( strcmp(dentry->d_name.name,FileInodeInfo[i].fileName) == 0 )
           {
              ino = i;
              break;
           }
        }

        printk("ino = %d\n",ino);
        /* If the filename is not found, no inode exists */
        if ( ino == 0 )
        {
            return ERR_PTR(-EACCES);
        }
        
        
	inode = FileInodeInfo[ino].inode;
        /* Do any fields in the inode need to change??? */

        /* Look closer at this logic. Seems to be different between MSDOS and Ext2 */
	d_add(dentry, inode);
	return NULL;
}

/* This function implements the LIF file read function. Note that even though
 * the lif file system calls the iodc driver that is registered a block driver
 * it does not use the buffer cache mechanism. This function will manage the
 * file access and will call the driver to read the entire file into its memory
 * that will be released when a close call is made.
 *
 * A return value of 0 signifile EOF 
 */


ssize_t lif_file_read(struct file *filp, char *buf, size_t count, loff_t *ppos)
{
   struct dentry *f_dentry = filp->f_dentry;
   struct inode *f_inode = f_dentry->d_inode;
   int	file_size  = f_inode->i_blocks * f_inode->i_blksize; /* in bytes */
   int bytesToRead = count;

   printk("\nlif_file_read has been entered.\n");
   if ( filp->f_pos == 0 )
   {
      /* Allocate space to hold the file here */
      /* Save this space in the file pointers private data */
      filp->private_data = kmalloc( 
	  ROUND_UP(((f_inode->i_blocks) * (f_inode->i_blksize)),2048) , 
					GFP_KERNEL );

      if ( filp->private_data != NULL )
      {
         /* Call the block driver to read the entire LIF file into memory.
          * The block driver only needs the start of the file and the size 
          * which is  stored in the inode. The file will be read into the
          * buffer stored in the private area */
         if ( iodc_ioctl (f_inode, filp, IODC_READ_FILE, 0) < 0 )
         {
           printk("iodc failed to read the file\n");
           /* No data could be read */
            return 0;
         }
      }
      else
      {
         /* No data could be read */
         return 0;
      }
      
   }

   if ( filp->f_pos >= file_size )
   {
     /* End of FIle */
      return 0;
   }

   if ( (filp->f_pos + count) >= file_size )
   {
      bytesToRead = file_size - filp->f_pos ;
   }
   
   /* If the file has already been read, f_pos will be non-zero and hence
    * the driver need not be called to read the file into memory */

   memcpy(buf, (char *)filp->private_data + filp->f_pos, bytesToRead );

   filp->f_pos += bytesToRead;

   return bytesToRead;
}

/* Since this is called from the close system call, the only thing done here
 * is to release the memory that was allocated on the first open
 */
int lif_file_release(struct inode *inode, struct file *filp)
{
   printk("\nWe are releasing now!");
   kfree(filp->private_data);
   return 0;
}



static struct file_system_type lif_fs_type = {
	"lif",
	FS_REQUIRES_DEV,     
	lif_read_super,
	NULL
};

__initfunc(int init_lif_fs(void))
{
	return register_filesystem(&lif_fs_type);
}
