#ifndef __ASM_PARISC_PCI_H
#define __ASM_PARISC_PCI_H


/*
** HP PCI platforms generally support multiple bus adapters.
**    (workstations 1-~4, servers 2-~32)
**
** Under a PCI bus, most HP platforms support PPBs up to two or three
** levels deep. See "Bit3" product line. 
**
** Size this constant appriopriately.
*/
#define PCI_MAX_BUSSES	16


/* [soapbox on]
** Who the hell can develope stuff without ASSERT or VASSERT?
** No one understands all the modules across all platforms.
** For linux add another dimension - processor architectures.
**
** This should be a standard/global macro used liberally
** in all code. Every respectable engineer I know in HP
** would support this argument. - grant
** [soapbox off]
*/
#ifdef PCI_DEBUG
#define ASSERT(expr) \
	if(!(expr)) { \
		printk( "\nAssertion failed! %s,%s,%s,line=%d\n\n",\
			#expr,__FILE__,__FUNCTION__,__LINE__); \
		panic("Assertion failure " #expr); \
	}
#else
#define ASSERT(expr)
#endif


/*
** Most PCI devices (eg Tulip, NCR720) also export the same registers
** to both MMIO and I/O port space.  Due to poor performance of I/O Port
** access under HP PCI bus adapters, strongly reccomend use of MMIO
** address space.
**
** While I'm at it more PA programming notes:
**
** 1) MMIO stores (writes) are posted operations. This means the processor
**    gets an "ACK" before the write actually gets to the device. A read
**    to the same device (or typically the bus adapter above it) will
**    force in-flight write transaction(s) out to the targeted device
**    before the read can complete.
**
** 2) The Programmed I/O (PIO) data may not always be strongly ordered with
**    respect to DMA on all platforms. Ie PIO data can reach the processor
**    before in-flight DMA reaches memory. Since most SMP PA platforms
**    are I/O coherent, it generally doesn't matter...but sometimes
**    it does.
**
** I've helped device driver writers debug both types of problems.
*/
struct pci_port_ops {
	  u8 (*inb)  (void *dev, u8 bus, u16 port);
	 u16 (*inw)  (void *dev, u8 bus, u16 port);
	 u32 (*inl)  (void *dev, u8 bus, u16 port);
	void (*outb) (void *dev,  u8 data, u8 bus, u16 port);
	void (*outw) (void *dev, u16 data, u8 bus, u16 port);
	void (*outl) (void *dev, u32 data, u8 bus, u16 port);
};

/*
** PCI bus adapter drivers can reference "device" specific data by
** storing a pointer into bus->sysdata when initializing struct pci_bus.
*/
struct pci_config_ops {
	int (*read_byte)  (unsigned char dinonum, u8 dev_fn, u8 where, u8 *val);
	int (*read_word)  (unsigned char dinonum, u8 dev_fn, u8 where, u16 *val);
	int (*read_dword) (unsigned char dinonum, u8 dev_fn, u8 where, u32 *val);
	int (*write_byte)  (unsigned char dinonum, u8 dev_fn, u8 where, u8 val);
	int (*write_word)  (unsigned char dinonum, u8 dev_fn, u8 where, u16 val); 
	int (*write_dword) (unsigned char dinonum, u8 dev_fn, u8 where, u32 val);
};


#ifdef USE_PCI_INTR_OPS
/*
** PA interrupt handling will need to invoke
** platform/bus specific interrupt handling code.
**
** irq.c will also need this if USE_PCI_INTR_OPS is defined.
*/
struct pci_intr_ops {
	int (*alloc_irq)(void *dev_id);	/* get IRQ handle */
	void (*free_irq)(int);		/* release handle/resources */
	int (*request_irq)(int irq,
			void (*handler)(int, void *, struct pt_regs *),
			unsigned long irqflags,
			const char * devname,
			void *dev_id);		/* register ISR for IRQ */
	void (*enable_irq)(int);		/* enable intr for this IRQ */
	void (*disable_irq)(int);		/* block intr for this IRQ */
};
#else

// REVISIT: define this stuff in include/asm-parisc/irq.h
//  from arch/parisc/kernel/irq.c
struct irq_region *alloc_irq_region(
	int count, struct irq_region_ops *ops, unsigned long flags,
	const char *name, void *dev);

#endif


struct pci_bios_ops {
	void (*init)(void);
	void (*fixup)(void);
	void (*fixup_bus)(struct pci_bus *bus);
	char * (*setup)(char *str);
};

struct pci_bus_ops {
	struct pci_port_ops port;
	struct pci_config_ops config;
#ifdef USE_PCI_INTR_OPS
	struct pci_intr_ops intr;
#endif
	struct pci_bios_ops bios;
};


// This is how PCI bus adapter driver tells the parisc subsystem
// a PCI bus is present and how to access it.
//
int pci_register_ops(struct pci_bus_ops *local_bus_ops, struct pci_bus *bus);

extern int pci_post_reset_delay;	// delay after de-asserting #RESET

#endif /* __ASM_PARISC_PCI_H */
