#ifndef __ASM_SPINLOCK_H
#define __ASM_SPINLOCK_H

#include <linux/kernel.h>

/* Here we have the definition of the locking data structure */
typedef struct {
	volatile unsigned long pad;
	volatile unsigned int lock;
} spinlock_t __attribute__ ((aligned(8)));


#define	spin_lock_init(x) spin_unlock(x)

#define spin_lock_irq(lock) do { __cli(); spin_lock(lock); } while(0)

#ifdef OUT
static __inline__ unsigned long spin_lock_irq(lock)
{
	__cli();
	spin_lock(lock);
}
#endif /*OUT*/

#define spin_lock_irqsave(lock, flags) \
do { save_flags(flags); cli(); spin_lock(lock); } while(0)

#define spin_unlock_irqrestore(lock, flags) \
do { spin_unlock(lock); restore_flags(flags);  } while(0)


#ifdef OUT_ORG_LINUX
#define spin_unlock_irq(lock)   sti()
#endif /*OUT_ORG_LINUX*/
#define spin_unlock_irq(lock) do { spin_unlock(lock); sti(); } while(0)

/*
* Read-write spinlocks, allowing multiple readers
* but only one writer.
*
* NOTE! it is quite common to have readers in interrupts
* but no interrupt writers. For those circumstances we
* can "mix" irq-safe locks - any writer needs to get a
* irq-safe write-lock, but readers can get non-irqsafe
* read-locks.
*/
typedef struct {
	volatile unsigned int lock;
	unsigned long previous;
} rwlock_t;

#define RW_LOCK_UNLOCKED (rwlock_t) { 0, 0 }

#define SPIN_LOCK_UNLOCKED (spinlock_t) { 0, -1 }

/*
 * On x86, we implement read-write locks as a 32-bit counter
 * with the high bit (sign) being the "write" bit.
 *
 * The inline assembly is non-obvious. Think about it.
*/
#define read_lock(rw) do { } while (0) 

#define read_unlock(rw) do { } while (0)

#define write_lock(rw) do { } while (0)

#define write_unlock(rw) do { } while (0)

#define read_lock_irq(lock)     do { __cli(); read_lock(lock); } while (0)
#define read_unlock_irq(lock)   do { read_unlock(lock); __sti(); } while (0)
#define write_lock_irq(lock)    do { __cli(); write_lock(lock); } while (0)
#define write_unlock_irq(lock)  do { write_unlock(lock); __sti(); } while (0)

#define read_lock_irqsave(lock, flags)  \
	     do { __save_flags(flags); __cli(); read_lock(lock); } while (0)
#define read_unlock_irqrestore(lock, flags) \
	     do { read_unlock(lock); __restore_flags(flags); } while (0)
#define write_lock_irqsave(lock, flags) \
	     do { __save_flags(flags); __cli(); write_lock(lock); } while (0)
#define write_unlock_irqrestore(lock, flags) \
	     do { write_unlock(lock); __restore_flags(flags); } while (0)

#endif /* __ASM_SPINLOCK_H */
